/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.    
 */
package org.apache.tuscany.sca.core.invocation;

import java.util.List;

import org.apache.tuscany.sca.runtime.Invocable;
import org.oasisopen.sca.ServiceReference;

/**
 * Creates proxies that implement Java interfaces and invocation handlers for fronting wires
 *
 * @version $Rev: 1303855 $ $Date: 2012-03-22 08:53:41 -0700 (Thu, 22 Mar 2012) $
 * @tuscany.spi.extension.asclient
 */

public interface ProxyFactory {

    /**
     * Creates a Java proxy for the given wire
     *
     * @param interfaze the interface the proxy implements
     * @param invocable the wire to proxy
     * @return the proxy
     * @throws ProxyCreationException
     */
    <T> T createProxy(Class<T> interfaze, Invocable invocable) throws ProxyCreationException;

    /**
     * Creates a Java proxy for the given CallableReference
     *
     * @param callableReference The CallableReference
     * @return the proxy
     * @throws ProxyCreationException
     */
    <T> T createProxy(ServiceReference<T> callableReference) throws ProxyCreationException;

    /**
     * Creates a Java proxy for the service contract callback
     *
     * @param interfaze the interface the proxy should implement
     * @return the proxy
     * @throws ProxyCreationException
     */
    <T> T createCallbackProxy(Class<T> interfaze, List<? extends Invocable> invocables) throws ProxyCreationException;

    /**
     * Creates a Java proxy for the given callback reference
     *
     * @param callableReference The CallableReference
     * @return the proxy
     * @throws ProxyCreationException
     */
    <T> T createCallbackProxy(ServiceReference<T> callbackReference) throws ProxyCreationException;

    /**
     * Cast a proxy to a CallableReference.
     *
     * @param target a proxy generated by this implementation
     * @return a CallableReference (or subclass) equivalent to this proxy
     * @throws IllegalArgumentException if the object supplied is not a proxy
     */
    <B, R extends ServiceReference<B>> R cast(B target) throws IllegalArgumentException;
    
    /**
     * Test if a given class is a generated proxy class by this factory
     * @param clazz A java class or interface
     * @return true if the class is a generated proxy class by this factory 
     */
    boolean isProxyClass(Class<?> clazz);
    
    /**
     * Allow cached proxies to be removed when a contribution is removed. The proxy 
     * cache holds the application interface so will pin the contribution classloader
     * 
     * @param contributionClassloader the classloader of the contribution being removed
     */
    void removeProxiesForContribution(ClassLoader contributionClassloader);

}
