
package com.ptteng.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.gemantic.commons.code.annotation.FileTemplate;
import com.ptteng.common.skill.model.*;
import com.ptteng.common.skill.service.*;
import com.ptteng.controller.RecordParam.RecordParam;
import com.ptteng.util.DataValidatorUtil;
import com.ptteng.util.DynamicUtil;
import com.qding.common.util.DataUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;

@Controller
public class RecordController {

    public static final Log log = LogFactory.getLog(RecordController.class);

    @Autowired
    private UserService userService;

    @Autowired
    private RecordService recordService;

    @Autowired
    private ManagerService managerService;

    @Autowired
    private RecordLogService recordLogService;

    @Autowired
    private BranchInstituteService branchInstituteService;

    @Autowired
    private OccupationService occupationService;

    @Autowired
    private SchoolService schoolService;

    @Autowired
    private MessageService messageService;

    @Autowired
    private TemplateTestService templateTestService;

    @Autowired
    private ProvincesService provincesService;

    @Autowired
    private CitiesService citiesService;

    @Autowired
    private CountiesService countiesService;

    @Autowired
    private MajorService majorService;

    /*
     * 新增档案
     *
     * @param uid 用户ID
     */
    @RequestMapping(value = "/a/u/record", method = RequestMethod.POST)
    public String createRecord(HttpServletRequest request, ModelMap model, Long uid) {

        log.info("POST /a/u/record: create new record for uid: " + uid);

        try {
            //判断用户是否存在
            User user = userService.getObjectById(uid);
            if (DataUtils.isNullOrEmpty(user)) {
                log.info("user is no exist: uid = " + uid);
                model.addAttribute("code", -5003);
                return "/common/failure";
            }

            //判断当前用户是否已经创建档案
            Long rid = recordService.getRecordIdByUidOrderByCreateAt(uid);
            if (DataUtils.isNotNullOrEmpty(rid)) {
                log.info("unsupported create record repeatedly, user's recordId: " + rid);
                model.addAttribute("code", -39026);
                return "/common/failure";
            }

            //获取当前登录的管理员
            Manager manager = (Manager) request.getAttribute("manager");
            log.info("get manager: " + manager);

            //设定record的字段的默认值为-1, 档案的状态status设为“申请中”
            Record record = new Record();
            record.setUid(uid);
            record.setAvatar(user.getThumb());
            record.setName(user.getNick());
            record.setSex(-1);
            record.setIdentityNum("-1");
            record.setBornAt(-1L);
            record.setProvince("-1");
            record.setCity("-1");
            record.setCounty("-1");
            record.setEducatedStartAt(-1L);
            record.setEducatedEndAt(-1L);
            record.setGrade(-1);
            record.setSchool("-1");
            record.setSchoolType(-1);
            record.setMajor("-1");
            record.setIsWork(-1);
            record.setWorkExp("-1");
            record.setContactName("-1");
            record.setContactRelation("-1");
            record.setContactMobile("-1");
            record.setCardNum("-1");
            record.setBank("-1");
            record.setSource("-1");
            record.setIdentityFrontUrl("-1");
            record.setIdentityReverseUrl("-1");
            record.setIsLoan(-1);
            record.setHouseholdRegisterUrl("-1");
            record.setStatus(Record.STATUS_APPLYING);
            record.setCreateBy(manager.getId());
            record.setUpdateBy(manager.getId());

            log.info("prepare to insert new record：" + record);
            Long recordId = recordService.insert(record);

            //同时，要更新recordStatus字段为“申请中”
            user.setRecordStatus(Record.STATUS_APPLYING);
            userService.update(user);
            log.info("update user's recordStatus, set value: 2");

            model.addAttribute("code", 0);
            model.addAttribute("id", recordId);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("insert new record failure, try again later");
            model.addAttribute("code", -39028);
        }
        return "/common/insert";
    }

    /**
     * 对应档案模块下面的用户列表，实际是以user表为基础进行动态查询
     *
     * @param uid           用户ID
     * @param branchId      分院ID
     * @param name          昵称
     * @param school        毕业院校
     * @param grade         学历
     * @param schoolType    学校类型
     * @param minStartAt    入学时间下限
     * @param maxStartAt    入学时间上限
     * @param minGraduateAt 结业时间下限
     * @param maxGraduateAt 结业时间上限
     * @param major         专业
     * @param place         籍贯
     * @param recordStatus  档案状态
     * @param age           年龄
     * @param occupationId  职业ID
     * @param minPassAt     存档时间下限
     * @param maxPassAt     存档时间上限
     * @param page          页码
     * @param size          每页数据个数
     */
    @RequestMapping(value = "/a/u/record/search", method = RequestMethod.GET)
    public String getRecordList(ModelMap model, Long uid, Long branchId, String name, String school, Integer grade,
                                Integer schoolType, Long minStartAt, Long maxStartAt, Long minGraduateAt, Long maxGraduateAt,
                                String major, String place, Integer recordStatus, Integer age, Integer occupationId, Long minPassAt,
                                Long maxPassAt, Integer page, Integer size) {

        log.info("GET /a/u/record/search: search recordList by params: uid = " + uid + ", branchId = " + branchId
                + ", name = " + name + ", school = " + school + ", grade = " + grade + ", schoolType = " + schoolType
                + ", minStartAt = " + minStartAt + ", maxStartAt = " + maxStartAt + ", minGraduateAt = " + minGraduateAt
                + ", maxGraduateAt = " + maxGraduateAt + ". major = " + major + ". place = " + place + ", recordStatus = "
                + recordStatus + ", age = " + age + ", occupationId = " + occupationId + ", minPassAt = " + minPassAt + ", maxPassAt = "
                + maxPassAt + ", page = " + page + ", size = " + size);

        if (page == null) {
            page = 1;
        }
        if (size == null) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        try {
            //获取搜索userIds的动态SQl
            Map<String, Object> params = DynamicUtil.getUserListParam(uid, branchId, occupationId, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, recordStatus, age, minPassAt, maxPassAt, false);
            log.info("search userIds by params: " + params);

            //获取userIds
            List<Long> userIds = userService.getIdsByDynamicCondition(User.class, params, start, size);
            log.info("get userIds size: " + userIds.size());
            if (CollectionUtils.isEmpty(userIds)) { // 若userIds为空，直接中断
                model.addAttribute("code", -39002);
                return "/common/failure";
            }

            //获取userList
            List<User> userList = userService.getObjectsByIds(userIds);
            log.info("get userList size: " + userList.size());

            //获取计算总数的动态SQl
            Map<String, Object> params2 = DynamicUtil.getUserListParam(uid, branchId, occupationId, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, recordStatus, age,  minPassAt, maxPassAt, true);
            log.info("count userIds by params:" + params2);

            List<Long> countUserIds = userService.getIdsByDynamicCondition(User.class, params, 0, Integer.MAX_VALUE);
            log.info("get countUserIds size:: " + countUserIds.size());

            int total = 0;
            if (!org.springframework.util.CollectionUtils.isEmpty(countUserIds)) {
                total = countUserIds.size();
            }

            //根据userIds得到查询recordIds的动态SQl
            Map<String, Object> params3 = DynamicUtil.getRecordListByUserIdsParam(userIds, false);
            log.info("search recordIds by params: " + params3);

            //获取recordIds
            List<Long> recordIds = recordService.getIdsByDynamicCondition(Record.class, params3, 0, Integer.MAX_VALUE);
            log.info("get recordIds size: " + recordIds.size() + ", detail: " + recordIds);

            List<Record> recordList = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(recordIds)) {
                recordList = recordService.getObjectsByIds(recordIds);
                log.info("get recordList size: " + recordList.size());
            }

            List<Long> occupationIds = new ArrayList<>();
            List<Long> branchIds = new ArrayList<>();
            for (User user : userList) {
                if (DataUtils.isNotNullOrEmpty(user.getOid()) && !occupationIds.contains(user.getOid())) {
                    occupationIds.add(user.getOid());
                }
                if (DataUtils.isNotNullOrEmpty(user.getBranch()) && !branchIds.contains(user.getBranch().longValue())) {
                    branchIds.add(user.getBranch().longValue());
                }
            }

            List<Occupation> occupationList = occupationService.getObjectsByIds(occupationIds);
            log.info("get occupationList size: " + occupationList.size());

            List<BranchInstitute> branchInstituteList = branchInstituteService.getObjectsByIds(branchIds);
            log.info("get branchInstituteList size: " + branchInstituteList.size());

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);
            model.addAttribute("userList", userList);
            model.addAttribute("recordList", recordList);
            model.addAttribute("occupationList", occupationList);
            model.addAttribute("branchList", branchInstituteList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("search record failure, try again later");
            model.addAttribute("code", -39003);
        }
        return "common-skill-service/user/json/userListAndRecordListJson";
    }


    /**
     * 查看档案详情
     *
     * @param uid 用户ID
     */
    @RequestMapping(value = "/a/u/record", method = RequestMethod.GET)
    public String getRecordDetail(ModelMap model, Long uid) {

        log.info("GET /a/u/record/{recordId}: get user's record by uid: " + uid);

        try {
            //校验用户是否存在
            User user = userService.getObjectById(uid);
            log.info("get user: " + user);
            if (DataUtils.isNullOrEmpty(user)) {
                log.info("user does not exist of uid: " + uid);
                model.addAttribute("code", -5003);
                return "/common/failure";
            }
            
            //根据入参recordId获取档案，并判断档案是否存在
            Long rid  = recordService.getRecordIdByUidOrderByCreateAt(uid);
            log.info("get recordId: " + rid);
            if (DataUtils.isNotNullOrEmpty(rid)) {
                Record record = recordService.getObjectById(rid);
                log.info("get record: " + record);
                model.addAttribute("record", record);
            }

            //获取用户对应的职业
            if (DataUtils.isNotNullOrEmpty(user.getOid())) {
                Occupation occupation = occupationService.getObjectById(user.getOid());
                log.info("get occupation: " + occupation.getName());
                model.addAttribute("occupation", occupation);
            }

            //获取用户对应的分院
            if (DataUtils.isNotNullOrEmpty(user.getBranch())) {
                BranchInstitute branchInstitute = branchInstituteService.getObjectById(Long.valueOf(user.getBranch()));
                log.info("get branchInstitute: " + branchInstitute.getName());
                model.addAttribute("branch", branchInstitute);
            }

            model.addAttribute("code", 0);
            model.addAttribute("user", user);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get record detail failure, try again later");
            model.addAttribute("code", -39005);
        }
        return "/common-skill-service/record/json/recordDetailJson";
    }

    /**
     * 获取省市县编号
     *
     * @param province 省名称
     * @param city 城市名称
     * @param county 区县名称
     */
    @RequestMapping(value = "/a/area/number", method = RequestMethod.GET)
    public String getAreaNumberByName(ModelMap model, String province, String city, String county) {

        log.info("GET /a/u/area/number: get provinceID && cityID && countyID by params: province = " + province
                + ", city = " + city + ", county = " + county);

        try {
            Long provinceId = -1L; //若是通过后台新增的档案，因为省市县都是-1，所以省市县编号的默认值也设为-1
            Long cityId = -1L;
            Long countyId = -1L;

            if (StringUtils.isNotEmpty(province)) {
                provinceId = templateTestService.getProvinceIdByProvince(province);
                log.info("get provinceID : " + provinceId);
            }

            if (StringUtils.isNotEmpty(city)) {
                cityId = templateTestService.getCityIdByCity(city);
                log.info("get cityID: " + cityId);
            }

            if (StringUtils.isNotEmpty(county) && ObjectUtils.notEqual(cityId, -1L)) {
                countyId = templateTestService.getCountyIdByCounty(county, cityId);
                log.info(" get countyID: " +countyId);
            }

            model.addAttribute("code", 0);
            model.addAttribute("provinceId", provinceId);
            model.addAttribute("cityId", cityId);
            model.addAttribute("countyId", countyId);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get area number failure, try again later");
            model.addAttribute("code", -39037);
        }
        return "/common-skill-service/record/json/areaNumberJson";
    }


    /**
     * 编辑档案
     *
     * @param record   要修改的档案信息
     * @param recordId 档案ID
     */
    @RequestMapping(value = "/a/u/record/{recordId}", method = RequestMethod.PUT)
    public String modifyRecord(HttpServletRequest request, ModelMap model, @PathVariable Long recordId, Record record) {

        log.info("PUT /a/u/record/{recordId}: modify record feild to: " + record);

        // 参数校验
        if (!DataValidatorUtil.checkRecord(record, model)) {
            log.info("params error");
            return "/common/failure";
        }

        try {
            //根据入参recordId获取档案并判断档案是否存在
            Record realRecord = recordService.getObjectById(recordId);
            log.info("get record before modify: " + realRecord);
            if (DataUtils.isNullOrEmpty(realRecord)) {
                log.info("the record is not exist, recordId = " + recordId);
                model.addAttribute("code", -39004);
                return "/common/failure";
            }

            //如果原档案对应的uid跟目标档案的uid不一致，则中断后续操作
            if (!realRecord.getUid().equals(record.getUid())) {
                log.info("record's uid do not support modify: " + realRecord.getUid() + " to " + record.getUid());
                model.addAttribute("code", -39006);
                return "/common/failure";
            }

            //获取当前登录的管理员
            Manager manager = (Manager) request.getAttribute("manager");
            log.info("get manager: " + manager);

            //更新record, 档案的状态status修改为申请中
            realRecord.setAvatar(record.getAvatar());
            realRecord.setName(record.getName());
            realRecord.setSex(record.getSex());
            realRecord.setIdentityNum(record.getIdentityNum());
            realRecord.setBornAt(record.getBornAt()); // 允许为空
            realRecord.setProvince(record.getProvince());
            realRecord.setCity(record.getCity());
            realRecord.setCounty(record.getCounty());
            realRecord.setEducatedStartAt(record.getEducatedStartAt());
            realRecord.setEducatedEndAt(record.getEducatedEndAt());
            realRecord.setGrade(record.getGrade());
            realRecord.setSchool(record.getSchool()); // 允许为空
            realRecord.setSchoolType(getSchoolType(record.getSchool())); // 允许为空
            realRecord.setMajor(record.getMajor()); // 允许为空
            realRecord.setIsWork(record.getIsWork());
            realRecord.setWorkExp(record.getWorkExp()); // 允许为空
            realRecord.setContactName(record.getContactName());
            realRecord.setContactRelation(record.getContactRelation());
            realRecord.setContactMobile(record.getContactMobile());
            realRecord.setCardNum(record.getCardNum());
            realRecord.setBank(record.getBank());
            realRecord.setSource(record.getSource());
            realRecord.setIdentityFrontUrl(record.getIdentityFrontUrl());
            realRecord.setIdentityReverseUrl(record.getIdentityReverseUrl());
            realRecord.setIsLoan(record.getIsLoan());
            realRecord.setHouseholdRegisterUrl(record.getHouseholdRegisterUrl()); // 允许为空
            realRecord.setUpdateBy(manager.getId());

            User user = userService.getObjectById(realRecord.getUid()); //获取原档案回应的用户User
            if (ObjectUtils.equals(Record.STATUS_SAVE, realRecord.getStatus())) {
                realRecord.setStatus(Record.STATUS_SAVE);  //原状态为已存档的，编辑后的状态仍为已存档

                //编辑后，还要以档案表信息覆盖个人信息
                user.setThumb(record.getAvatar());
                user.setNick(record.getName());
                user.setSex(Long.valueOf(record.getSex()));
                user.setBirthday(record.getBornAt());
                user.setSchool(record.getSchool());
                user.setProvince(record.getProvince());
                user.setCity(record.getCity());
                user.setRecordStatus(Record.STATUS_SAVE);

            } else if (ObjectUtils.equals(Record.STATUS_APPLYING, realRecord.getStatus()) || ObjectUtils.equals(Record.STATUS_NO_PASS, realRecord.getStatus())) {
                realRecord.setStatus(Record.STATUS_APPLYING);  //原状态为申请中或未通过的，编辑后的状态均设置为申请中
                user.setRecordStatus(Record.STATUS_APPLYING);
            }
            user.setUpdateBy(manager.getId());

            recordService.update(realRecord);
            userService.update(user);

            //新增审核记录
            RecordLog recordLog = new RecordLog();
            recordLog.setUid(realRecord.getUid());
            recordLog.setOperate(RecordLog.OPERATE_EDIT);
            recordLog.setCreateBy(manager.getId());
            recordLog.setUpdateBy(manager.getId());
            Long recordLogId = recordLogService.insert(recordLog);
            log.info("insert new recordLog: recordLogId = " + recordLogId + ", operate = \"edit\"");

            model.addAttribute("code", 0);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("modify record failure, try again later");
            model.addAttribute("code", -39007);
        }
        return "/common/success";
    }


    /**
     * 审核档案
     *
     * @param recordId     档案ID
     * @param operate      审核操作（1-存档，2-驳回）
     * @param refuseReason 驳回理由
     */
    @RequestMapping(value = "/a/u/record/check/{recordId}/{operate}", method = RequestMethod.POST)
    public String checkRecord(HttpServletRequest request, ModelMap model, @PathVariable Long recordId, @PathVariable Integer operate, String refuseReason) {

        log.info("POST /a/u/record/check/{recordId}/{operate}: recordId = " + recordId + ", operate = " + operate + ", refuseReason =" + refuseReason);

        //此接口对应的审核操作只可能是 1-存档 2-驳回
        if (operate > RecordLog.OPERATE_REFUSE || operate < RecordLog.OPERATE_SAVE) {
            log.info("not supported this operate: " + operate);
            model.addAttribute("code", -39008);
            return "/common/insert";
        }

        //驳回时，驳回理由不能为空
        if (operate.equals(RecordLog.OPERATE_REFUSE) && StringUtils.isEmpty(refuseReason)) {
            log.info("operate equals \"refuse\", refuseReason cant not be empty");
            model.addAttribute("code", -39009);
            return "/common/insert";
        }

        //存档时，不需要驳回理由
        if (operate.equals(RecordLog.OPERATE_SAVE) && StringUtils.isNotEmpty(refuseReason)) {
            log.info("operate equals \"save\", don't need refuseReason");
            model.addAttribute("code", -39010);
            return "/common/insert";
        }

        try {
            //根据入参recordId获取档案并判断档案是否存在
            Record record = recordService.getObjectById(recordId);
            log.info("get record by recordId: " + record);
            if (DataUtils.isNullOrEmpty(record)) {
                log.info("the record is not exist, recordId = " + recordId);
                model.addAttribute("code", -39004);
                return "/common/insert";
            }

            //已存档的档案，审核操作不能是驳回
            if (ObjectUtils.equals(Record.STATUS_SAVE, record.getStatus()) && ObjectUtils.equals(RecordLog.OPERATE_REFUSE, operate)) {
                log.info("the record has been save, unsupported reject");
                model.addAttribute("code", -39012);
                return "/common/insert";
            }

            //获取当前登录的管理员
            Manager manager = (Manager) request.getAttribute("manager");
            log.info("get manager: " + manager);

            //新增审核记录
            RecordLog recordLog = new RecordLog();
            recordLog.setUid(record.getUid());
            recordLog.setOperate(operate);
            recordLog.setRefuseReason(refuseReason);
            recordLog.setCreateBy(manager.getId());
            recordLog.setUpdateBy(manager.getId());
            Long recordLogId = recordLogService.insert(recordLog);
            log.info("insert new recordLog: recordLogId = " + recordLogId);

            //更新user的recordStatus字段
            User user = userService.getObjectById(record.getUid());
            if (operate.equals(RecordLog.OPERATE_SAVE)) { //审核结果为存档的
                record.setStatus(Record.STATUS_SAVE);
                user.setRecordStatus(Record.STATUS_SAVE);

                //存档后，还要以档案表信息覆盖个人信息
                user.setThumb(record.getAvatar());
                user.setNick(record.getName());
                user.setSex(Long.valueOf(record.getSex()));
                user.setBirthday(record.getBornAt());
                user.setSchool(record.getSchool());
                user.setProvince(record.getProvince());
                user.setCity(record.getCity());

            } else if (operate.equals(RecordLog.OPERATE_REFUSE)) { //审核结果为驳回
                record.setStatus(Record.STATUS_NO_PASS);
                user.setRecordStatus(Record.STATUS_NO_PASS);

                //还要给用户发送一条类型为“class”的消息，更新未读消息数，最新的消息类型
                Long messageId = sendRefuseMessageToStudent(manager, user, recordLogId);
                log.info("send refuse message to student: messageId = " + messageId);

                user.setIsLook(user.getIsLook() + 1);
                user.setLatestMsg(Message.CLASS_TYPE);
            }

            recordService.update(record);
            userService.update(user);

            model.addAttribute("code", 0);
            model.addAttribute("id", recordLogId);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("check record failure, try again later");
            model.addAttribute("code", -39011);
        }
        return "/common/insert";
    }

    /**
     * 查看折线图 - 根据条件返回内门弟子随时间变化的折线图数据
     *
     * @param school       毕业院校
     * @param place        籍贯
     * @param major        专业
     * @param grade        学历
     * @param occupationId 职业ID
     * @param age          年龄
     * @param minStartAt   入学时间下限
     * @param maxStartAt   入学时间上限
     */
    @RequestMapping(value = "/a/u/chart/line", method = RequestMethod.GET)
    public String getLineChartList(ModelMap model, String school, String place, String major, Integer grade, Integer occupationId,
                                   Integer age, Long minStartAt, Long maxStartAt) {

        log.info("GET /a/u/chart/line: by params: school = " + school + ", place = " + place + ", major = "
                + major + ", grade = " + grade + ", occupationId = " + occupationId + ", age = " + age + ", minStartAt = "
                + minStartAt + ", maxStartAy = " + maxStartAt);

        try {
            //使用JDBCTemplate直接返回包装好的统计对象
            List<InDoorStudentStatistics> inDoorStudentStatisticsList = templateTestService.getLineChartByParams(school,
                    place, major, grade, occupationId, age, minStartAt, maxStartAt);
            log.info("get inDoorStudentStatisticsList size: " + inDoorStudentStatisticsList.size());

            if (CollectionUtils.isEmpty(inDoorStudentStatisticsList)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -39002);
                return "/common/failure";
            } else {
                model.addAttribute("code", 0);
                model.addAttribute("total", inDoorStudentStatisticsList.size());
                model.addAttribute("statisticsList", inDoorStudentStatisticsList);
            }

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get linechart failure, try again later");
            model.addAttribute("code", -39015);
        }
        return "/common-skill-service/record/json/statisticsListJson";
    }

    /**
     * 查看饼图
     *
     * @param minPassAt    存档时间下限
     * @param maxPassAt    存档时间上限
     * @param occupationId 职业ID
     * @param condition    条件
     */
    @RequestMapping(value = "/a/u/chart/pie", method = RequestMethod.GET)
    public String getPieChartList(ModelMap model, Long minPassAt, Long maxPassAt, Long minStartAt, Long maxStartAt, Long minGraduateAt, Long maxGraduateAt, Integer occupationId, Integer condition) {

        log.info("GET /a/u/chart/pie: by params: " + "minPassAt = " + minPassAt + ", maxPassAt = " + maxPassAt
                + ", minStartAt = " + minStartAt + ", maxStartAt = " + maxStartAt + ", minGraduateAt = " + minGraduateAt
                + ", maxGraduateAt = " + maxGraduateAt + ", occupationId = " + occupationId + ", condition = " + condition);

        //存档时间下限和存档时间上限不能为空，默认值前请输入近7天的时间戳
        if (DataUtils.isNullOrEmpty(minPassAt) || DataUtils.isNullOrEmpty(maxPassAt)) {
            log.info("minPassAt & maxPassAt can no be null");
            model.addAttribute("code", -39013);
            return "/common/failure";
        }

        //条件不允许为空，默认值请输入 4-按专业统计
        if (DataUtils.isNullOrEmpty(condition)) {
            log.info("condition can no be null, please entry default value: 4");
            model.addAttribute("code", -39014);
            return "/common/failure";
        }

        try {
            //直接使用JDBCTemplate返回包装好的统计对象
            List<InDoorStudentStatistics> inDoorStudentStatisticsList = templateTestService.getPieCharByParams(minPassAt, maxPassAt, minStartAt, maxStartAt, minGraduateAt, maxGraduateAt, occupationId, condition);
            log.info("get inDoorStudentStatisticsList size: " + inDoorStudentStatisticsList.size());

            //计算当前条件下的总数
            Integer total = templateTestService.getRecordTotalNumber(Record.STATUS_SAVE);
            log.info("get record total number by status = 3: " + total);

            if (CollectionUtils.isEmpty(inDoorStudentStatisticsList)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -39002);
                return "/common/failure";
            } else {
                model.addAttribute("code", 0);
                model.addAttribute("total", total);
                model.addAttribute("statisticsList", inDoorStudentStatisticsList);
            }

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get pieChartList failure, try again later");
            model.addAttribute("code", -39015);
        }
        return "/common-skill-service/record/json/statisticsListJson";
    }


    /**
     * 导出Excel
     *
     * @param uid           用户ID
     * @param branchId      分院ID
     * @param name          昵称
     * @param school        毕业院校
     * @param grade         学历
     * @param schoolType    学校类型
     * @param minStartAt    入学时间下限
     * @param maxStartAt    入学时间上限
     * @param minGraduateAt 结业时间下限
     * @param maxGraduateAt 结业时间上限
     * @param major         专业
     * @param place         籍贯
     * @param recordStatus  档案状态
     * @param age           年龄
     * @param oid           职业ID
     * @param minPassAt     存档时间下限
     * @param maxPassAt     存档时间上限
     */
    @FileTemplate(templatePath = "excel/excel.vm", filePath = "/data/webs/skill-admin-home/excel/", fileName = "")
    @RequestMapping(value = "/a/u/record/excel", method = RequestMethod.GET)
    public String exportExcel(ModelMap model, Long uid, Long branchId, String name, String school, Integer grade,
                              Integer schoolType, Long minStartAt, Long maxStartAt, Long minGraduateAt, Long maxGraduateAt,
                              String major, String place, Integer recordStatus, Integer age, Integer oid, Long minPassAt,
                              Long maxPassAt) {

        log.info("GET /a/u/record/excel: export excel by params: uid = " + uid + ", branchId = " + branchId
                + ", name = " + name + ", school = " + school + ", grade = " + grade + ", schoolType = " + schoolType
                + ", minStartAt = " + minStartAt + ", maxStartAt = " + maxStartAt + ", minGraduateAt = " + minGraduateAt
                + ", maxGraduateAt = " + maxGraduateAt + ". major = " + major + ". place = " + place + ", recordStatus = "
                + recordStatus + ", age = " + age + ", oid = " + oid + ", minPassAt = " + minPassAt + ", maxPassAt = " + maxPassAt);

        try {
            Map<String, Object> params = DynamicUtil.getUserListParam(uid, branchId, oid, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, recordStatus, age,  minPassAt, maxPassAt, false);
            log.info("search userIds by params: " + params);

            List<Long> userIds = userService.getIdsByDynamicCondition(User.class, params, 0, Integer.MAX_VALUE);
            log.info("get userIds size: " + userIds.size());
            if (CollectionUtils.isEmpty(userIds)) {
                log.info("export excel failure: there are no qualified results");
                model.addAttribute("code", -39016);
                return "/common/failure";
            }

            List<User> userList = userService.getObjectsByIds(userIds);
            log.info("get userList size: " + userList.size());

            Map<String, Object> params2 = DynamicUtil.getUserListParam(uid, branchId, oid, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, recordStatus, age,  minPassAt, maxPassAt, true);
            log.info("count userIds by params:" + params2);

            List<Long> countUserIds = userService.getIdsByDynamicCondition(User.class, params, 0, Integer.MAX_VALUE);
            log.info("get countUserIds size:: " + countUserIds.size());

            int total;
            if (!org.springframework.util.CollectionUtils.isEmpty(countUserIds)) {
                total = countUserIds.size();
                log.info("get total number of users：" + total);
            }

            Map<String, Object> params3 = DynamicUtil.getRecordListByUserIdsParam(userIds, false);
            log.info("search recordIds by params: " + params3);

            List<Long> recordIds = recordService.getIdsByDynamicCondition(Record.class, params3, 0, Integer.MAX_VALUE);
            log.info("get recordIds size: " + recordIds.size() + ", detail: " + recordIds);

            List<Record> recordList;
            Map<Long, Record> recordMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(recordIds)) {
                recordList = recordService.getObjectsByIds(recordIds);
                log.info("get recordList size: " + recordList.size());

                for (Record record : recordList) {
                    recordMap.put(record.getUid(), record);
                }

            }

            List<Long> occupationIds = new ArrayList<>();
            List<Long> branchIds = new ArrayList<>();
            for (User user : userList) {
                if (DataUtils.isNotNullOrEmpty(user.getOid())) {
                    occupationIds.add(user.getOid());
                }
                if (DataUtils.isNotNullOrEmpty(user.getBranch())) {
                    branchIds.add(Long.valueOf(user.getBranch()));
                }
            }

            List<Occupation> occupationList = occupationService.getObjectsByIds(occupationIds);
            log.info("get occupationList size: " + occupationList.size());
            List<BranchInstitute> branchInstituteList = branchInstituteService.getObjectsByIds(branchIds);
            log.info("get branchInstituteList size: " + branchInstituteList.size());

            Map<Long, Occupation> occupationMap = new HashMap<>();
            Map<Integer, BranchInstitute> branchInstituteMap = new HashMap<>();
            for (Occupation o : occupationList) {
                occupationMap.put(o.getId(), o);
            }
            for (BranchInstitute b : branchInstituteList) {
                branchInstituteMap.put(b.getId().intValue(), b);
            }

            //把上面得到的userList、recordMap、occupationMap、branchInstituteMap包装成recordParamList
            List<RecordParam> recordParamList = new ArrayList<>();
            for (User user : userList) {
                RecordParam recordParam = new RecordParam();
                recordParam.setUid(user.getId());
                recordParam.setName(user.getNick());
                if (DataUtils.isNotNullOrEmpty(occupationMap.get(user.getOid()))) {
                    recordParam.setOccupation(occupationMap.get(user.getOid()).getName());
                }
                if (DataUtils.isNotNullOrEmpty(branchInstituteMap.get(user.getBranch()))) {
                    recordParam.setBranch(branchInstituteMap.get(user.getBranch()).getName());
                }
                recordParam.setStartAt(transformDate(user.getStartAt()));//转换成日期格式
                recordParam.setGraduateAt(transformDate(user.getGraduateAt()));//转换成日期格式
                if (DataUtils.isNotNullOrEmpty(recordMap.get(user.getId()))) {
                    recordParam.setGrade(transformGrade(recordMap.get(user.getId()).getGrade()));//转成“本科、硕士”
                    recordParam.setSchool(transformSchool(recordMap.get(user.getId()).getSchool()));
                    recordParam.setSchoolType(transformSchoolType(recordMap.get(user.getId()).getSchoolType()));//转成985/211
                    recordParam.setMajor(transformMajor(recordMap.get(user.getId()).getMajor()));
                    recordParam.setSource(transformSource(recordMap.get(user.getId()).getSource()));
                }
                recordParam.setStatus(transformUserStatus(user.getStatus()));//转成“结业状态”/“在学”
                recordParam.setCompanyName(user.getCompanyName());
                recordParam.setSalary(transformSalary(user.getSalary()));//转成 10K
                recordParam.setRecordStatus(transformRecordStatus(user.getRecordStatus()));

                recordParamList.add(recordParam);
            }

            //设定excel的表头
            RecordParam recordParam = new RecordParam();
            Map<String, String> config = recordParam.getConfig();
            model.put("config", config);

            String[] columns = recordParam.getKeys();
            model.put("configKeys", columns);
            for (String fieldName : columns) {
                Field field = RecordParam.class.getDeclaredField(fieldName);
                field.setAccessible(true);
                field.get(recordParam);
            }
            log.info("recordParam is " + recordParam);

            model.addAttribute("code", 0);
            model.addAttribute("list", recordParamList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("export excel failure, try again later");
            model.addAttribute("code", -39017);
        }
        return "/data/json";
    }

    /**
     * 获取全国各省市区县数据
     */
    @RequestMapping(value = "/a/u/area/list", method = RequestMethod.GET)
    public String getAreaList(ModelMap model) {

        log.info("GET /a/u/area/list");

        try {
            List<Long> provincesIds = provincesService.getProvincesIds(0, Integer.MAX_VALUE);
            log.info("get provincesIds size = " + provincesIds.size());

            List<Long> citiesIds = citiesService.getCitiesIds(0, Integer.MAX_VALUE);
            log.info("get citiesIds  size = " + citiesIds.size());

            List<Long> countiesIds = countiesService.getCountiesIds(0, Integer.MAX_VALUE);
            log.info("get countiesIds size = " + countiesIds.size());

            if (org.springframework.util.CollectionUtils.isEmpty(provincesIds) || org.springframework.util.CollectionUtils.isEmpty(citiesIds)
                    || org.springframework.util.CollectionUtils.isEmpty(countiesIds)) {
                log.info("get area failure");
                model.addAttribute("code", -38010); //省市县数据不存在
                return "/common/failure";
            }

            List<Provinces> provincesList = provincesService.getObjectsByIds(provincesIds);
            log.info("get provincesList success, size = " + provincesList.size());

            List<Cities> citiesList = citiesService.getObjectsByIds(citiesIds);
            log.info("get citiesList success, size = " + citiesList.size());

            List<Counties> countiesList = countiesService.getObjectsByIds(countiesIds);
            log.info("get countiesList success, size = " + countiesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("provinceList", provincesList);
            model.addAttribute("cityList", citiesList);
            model.addAttribute("countyList", countiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get areas failure, try again later");
            model.addAttribute("code", -38011);
        }
        return "/common-skill-service/provinces/json/areaListJson";
    }


    /**
     * 获取全国各省数据
     */
    @RequestMapping(value = "/a/u/province/list", method = RequestMethod.GET)
    public String getProvinceList(ModelMap model) {

        log.info("GET /a/u/province");

        try {
            List<Long> provincesIds = provincesService.getProvincesIds(0, Integer.MAX_VALUE);
            log.info("get provincesIds size = " + provincesIds.size());

            if (org.springframework.util.CollectionUtils.isEmpty(provincesIds)) {
                log.info("get provinceList failure");
                model.addAttribute("code", -38012); //省数据不存在
                return "/common/failure";
            }

            List<Provinces> provincesList = provincesService.getObjectsByIds(provincesIds);
            log.info("get provincesList success, size = " + provincesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", provincesList.size());
            model.addAttribute("provincesList", provincesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get provinceList failure, try again later");
            model.addAttribute("code", -38013);
        }
        return "/common-skill-service/provinces/json/provincesListJson";
    }


    /**
     * 获取某省的市数据
     *
     * @param provinceId 某省ID
     */
    @RequestMapping(value = "/a/u/city/list", method = RequestMethod.GET)
    public String getCityList(ModelMap model, Long provinceId) {

        log.info("GET /a/u/city/list: which provinceId = " + provinceId);

        try {
            List<Long> citiesIds = citiesService.getCitiesIdsByProvinceIdOrderByCreateAt(provinceId, 0, Integer.MAX_VALUE);
            log.info("get citiesIds  size = " + citiesIds.size());

            if (org.springframework.util.CollectionUtils.isEmpty(citiesIds)) {
                log.info("get cityList failure");
                model.addAttribute("code", -38014); //市数据不存在
                return "/common/failure";
            }

            List<Cities> citiesList = citiesService.getObjectsByIds(citiesIds);
            log.info("get citiesList success, size = " + citiesList.size());


            model.addAttribute("code", 0);
            model.addAttribute("total", citiesList.size());
            model.addAttribute("citiesList", citiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get cityList failure, ty again later");
            model.addAttribute("code", -38015);
        }
        return "/common-skill-service/cities/json/citiesListJson";
    }


    /**
     * 获取某市的区县数据
     *
     * @param cityId 某市ID
     */
    @RequestMapping(value = "/a/u/county/list", method = RequestMethod.GET)
    public String getCountyList(ModelMap model, Long cityId) {

        log.info("GET /a/u/county/list: which cityId = " + cityId);

        try {

            List<Long> countiesIds = countiesService.getCountiesIdsByCityIdOrderByCreateAt(cityId, 0, Integer.MAX_VALUE);
            log.info("get countiesIds size = " + countiesIds.size());

            if (org.springframework.util.CollectionUtils.isEmpty(countiesIds)) {
                log.info("get countyList failure");
                model.addAttribute("code", -38016); //县数据不存在
                return "/common/failure";
            }

            List<Counties> countiesList = countiesService.getObjectsByIds(countiesIds);
            log.info("get countiesList success, size = " + countiesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", countiesList.size());
            model.addAttribute("countiesList", countiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get cityList failure, ty again later");
            model.addAttribute("code", -38017);
        }
        return "/common-skill-service/counties/json/countiesListJson";
    }


    /**
     * 按名称查找学校-模糊搜索
     *
     * @param name 学校名称
     */
    @RequestMapping(value = "/a/u/school/search", method = RequestMethod.GET)
    public String getSchoolByName(ModelMap model, String name) {

        log.info("GET /a/u/school/search: prepare to get school list by name: " + name);

        if (StringUtils.isEmpty(name)) {
            log.info("params are null: name = " + name);
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {

            Map<String, Object> params = DynamicUtil.getSchoolByName(name);
            log.info("get params:" + params);

            int size = 10;  //默认size=10
            List<Long> schoolIds = schoolService.getIdsByDynamicCondition(School.class, params, 0, size);
            log.info("get schoolIds size = " + schoolIds.size());
            if (org.springframework.util.CollectionUtils.isEmpty(schoolIds)) {
                model.addAttribute("code", -39018);
                return "/common/failure";
            }

            List<School> schoolList = schoolService.getObjectsByIds(schoolIds);
            log.info("get schoolList size = " + schoolList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", schoolList.size());
            model.addAttribute("schoolList", schoolList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get school failure, try again later");
            model.addAttribute("code", -39019);
        }
        return "/common-skill-service/school/json/schoolListJson";
    }


    //搜索专业列表 - 按第一级专业名搜索
    @RequestMapping(value = "/a/u/major/primary/search", method = RequestMethod.GET)
    public String getSecondaryMajorByPrimaryMajor(ModelMap model, String major) {

        log.info("GET /a/u/major/primary/search: get secondaryMajorList by primaryMajor: " + major);

        if (StringUtils.isEmpty(major)) {
            log.info("param is null");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            List<Long> majorIds = majorService.getMajorIdsByPrimaryMajor(major, 0, Integer.MAX_VALUE);
            log.info("get majorIds size: " + majorIds.size());
            if (org.springframework.util.CollectionUtils.isEmpty(majorIds)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -39022);
                return "/common-skill-service/major/json/majorListJson";
            }

            List<Major> majorList = majorService.getObjectsByIds(majorIds);
            log.info("get majorList size: " + majorList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", majorList.size());
            model.addAttribute("majorList", majorList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get majorList failure, try again later");
            model.addAttribute("code", -39024);
        }
        return "/common-skill-service/major/json/majorListJson";
    }

    //搜索专业列表
    @RequestMapping(value = "/a/u/major/{type}", method = RequestMethod.GET)
    public String getMajorList(ModelMap model, @PathVariable Integer type, String major) {

        log.info("Get /a/u/major/{type}: get majorList by type: " + type + " & major: " + major);

        if (StringUtils.isEmpty(major)) {
            log.info("param is null");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        if (type < 1 && type > 3) {
            log.info("unsupported this value of type: " + type);
            model.addAttribute("code", -39025);
            return "/common/failure";
        }

        try {
            List<Long> majorIds = new ArrayList<>();

            if (ObjectUtils.equals(1, type)) {
                log.info("get majorList by major: " + major);
                majorIds = majorService.getMajorIdsByPrimaryMajor(major, 0, Integer.MAX_VALUE);
            } else if (ObjectUtils.equals(2, type)) {
                log.info("get majorList by major: " + major);
                majorIds = majorService.getMajorIdsBySecondryMajor(major, 0, Integer.MAX_VALUE);
            } else if (ObjectUtils.equals(3, type)) {
                Map<String, Object> params = DynamicUtil.getMajorByName(major);
                log.info("get majorList by params: " + params);
                majorIds = majorService.getIdsByDynamicCondition(Major.class, params, 0, Integer.MAX_VALUE);
            }

            log.info("get majorIds size: " + majorIds.size());

            if (org.springframework.util.CollectionUtils.isEmpty(majorIds)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -39022);
                return "/common-skill-service/major/json/majorListJson";
            }

            List<Major> majorList = majorService.getObjectsByIds(majorIds);
            log.info("get majorList size: " + majorList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", majorList.size());
            model.addAttribute("majorList", majorList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get majorList failure, try again later");
            model.addAttribute("code", -39024);
        }
        return "/common-skill-service/major/json/majorListJson";
    }

    /**
     * 根据学校名返回学校的类型
     *
     * @param schoolName 学校名称
     * @return 学校类型（985、211、非重点）
     */
    private Integer getSchoolType(String schoolName) throws ServiceException, ServiceDaoException {
        if (StringUtils.isNotEmpty(schoolName)) {
            Long schoolId = schoolService.getSchoolIdByName(schoolName);
            if (DataUtils.isNotNullOrEmpty(schoolId)) {
                School school = schoolService.getObjectById(schoolId);
                return school.getSchoolType();
            } else {
                return School.SCHOOL_TYPE_NORMAL;
            }
        }
        return -1;
    }


    /**
     * 档案被驳回时，发送一条消息给学员
     *
     * @param user    接收消息的学员
     * @param recordLogId  审核记录ID
     * @param manager 进行审核操作的后台管理员
     */
    private Long sendRefuseMessageToStudent(Manager manager, User user, Long recordLogId) throws ServiceException, ServiceDaoException {
        String content = "<span class=\"font-16px system-alert-outline\">你提交的档案表被驳回，请 <a target=\"_blank\" href=\"/school/" + user.getId() + "/data?fromMsg=1\"><span class=\"font-green\">“查看原因”</span> </a> 并重新填写提交</span>";

        Message message = new Message();
        message.setSendId(manager.getId());
        message.setReceiveId(user.getId());
        message.setSendName(manager.getName());
        message.setReceiveName(user.getNick());
        message.setContent(content);
        message.setStatus(Message.STATUS_UNDELETE);
        message.setUnread(Message.Status_Unread);
        message.setType(Message.CLASS_TYPE);
        message.setTemplate(-1);
        message.setRelationId(recordLogId);
        message.setCreateBy(manager.getId());
        message.setUpdateBy(manager.getId());
        Long messageId = messageService.insert(message);

        Message message1 = messageService.getObjectById(messageId);
        String content1 = "<span class=\"font-16px system-alert-outline\">你提交的档案表被驳回，请 <a target=\"_blank\" data-mid=96978 ng-click=\"sendAlertMsg(li.id)\" href=\"/school/" + user.getId() + "/data?fromMsg=1&recordLogId=" + recordLogId + "\"><span class=\"font-green\">“查看原因”</span> </a> 并重新填写提交</span>";
        message1.setContent(content1);
        messageService.update(message1);

        return messageId;
    }


    private String transformDate(Long timestamp) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("YYYY-MM-dd");
        if (DataUtils.isNotNullOrEmpty(timestamp) && !ObjectUtils.equals(-1L, timestamp)) {
            Date date = new Date(timestamp);
            return simpleDateFormat.format(date);
        }
        return "";
    }

    private String transformGrade(Integer grade) {
        if (ObjectUtils.equals(Record.GRADE_PRIMARY, grade)) {
            return "小学";
        }
        if (ObjectUtils.equals(Record.GRADE_JUNIOR_MIDDLE_SCHOOL, grade)) {
            return "初中";
        }
        if (ObjectUtils.equals(Record.GRADE_SENIOR_HIGH_SCHOOL, grade)) {
            return "高中";
        }
        if (ObjectUtils.equals(Record.GRADE_COMMUNITY_COLLEGE, grade)) {
            return "专科";
        }
        if (ObjectUtils.equals(Record.GRADE_SECONDARY, grade)) {
            return "中专";
        }
        if (ObjectUtils.equals(Record.GRADE_JUNIOR_COLLEGE, grade)) {
            return "大专";
        }
        if (ObjectUtils.equals(Record.GRADE_BACHELOR_DEGREE, grade)) {
            return "本科";
        }
        if (ObjectUtils.equals(Record.GRADE_MASTER_DEGREE, grade)) {
            return "硕士";
        }
        if (ObjectUtils.equals(Record.GRADE_DOCTOR, grade)) {
            return "博士";
        }
        return "";
    }

    private String transformSchool(String school) {
        if (ObjectUtils.equals("-1", school)) {
            return "";
        }
        return school;
    }

    private String transformSchoolType(Integer schoolType) {
        if (ObjectUtils.equals(1, schoolType)) {
            return "985";
        }
        if (ObjectUtils.equals(2, schoolType)) {
            return "211";
        }
        if (ObjectUtils.equals(3, schoolType)) {
            return "非重点";
        }
        return "";
    }

    private String transformMajor(String major) {
        if (ObjectUtils.equals("-1", major)) {
            return "";
        }
        return major;
    }

    private String transformSource(String source) {
        if (ObjectUtils.equals("-1", source)) {
            return "";
        }
        return source;
    }

    private String transformUserStatus(Integer status) {
        if (ObjectUtils.equals(User.Status_Student, status)) {
            return "在学";
        }
        if (ObjectUtils.equals(User.STATUS_1, status)) {
            return "退学";
        }
        if (ObjectUtils.equals(User.STATUS_2, status)) {
            return "待业";
        }
        if (ObjectUtils.equals(User.STATUS_3, status)) {
            return "结业";
        }
        if (ObjectUtils.equals(User.STATUS_5, status)) {
            return "请假";
        }
        return "";
    }

    private String transformRecordStatus(Integer recordStatus) {
        if (ObjectUtils.equals(Record.STATUS_UNCOMMITTED, recordStatus)) {
            return "未提交";
        }
        if (ObjectUtils.equals(Record.STATUS_APPLYING, recordStatus)) {
            return "申请中";
        }
        if (ObjectUtils.equals(Record.STATUS_SAVE, recordStatus)) {
            return "已存档";
        }
        if (ObjectUtils.equals(Record.STATUS_NO_PASS, recordStatus)) {
            return "未通过";
        }
        return "";
    }

    private String transformSalary(Integer salary) {
        if (DataUtils.isNotNullOrEmpty(salary) && !ObjectUtils.equals(-1, salary)) {
            return salary + "K";
        }
        return "";
    }
}

