package com.ptteng.controller;

import com.ptteng.common.skill.model.*;
import com.ptteng.common.skill.service.*;
import com.ptteng.util.DynamicUtil;
import com.ptteng.util.LogUtil;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Label  crud
 *
 * @author guchen
 * @Date 2018-01-19
 */
@Controller
public class LabelController {
    private static final Log log = LogFactory.getLog(LabelController.class);

    @Autowired
    private LabelService labelService;

    @Autowired
    private OccupationService occupationService;


    @Autowired
    private QuestionLabelRelationService questionLabelRelationService;

    @Autowired
    private QuestionService questionService;

    @Autowired
    private TaskService taskService;

    @Autowired
    private CookieUtil cookieUtil;


    /**
     * 1 新增标签
     *
     * @param request
     * @param response
     * @param model
     * @param label
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label", method = RequestMethod.POST)
    public String addLabelJson(HttpServletRequest request,
                               HttpServletResponse response, ModelMap model, Label label) throws Exception {

        log.info("update label : label= " + label);

        if (label.getOid() == null) {
            model.addAttribute("code", -24001);
            log.info("oid is null");
            return "/common/failure";
        }

        if (label.getTid() == null) {
            model.addAttribute("code", -27002);
            log.info("tid is null");
            return "/common/failure";
        }
        if (label.getTitle() == null) {
            model.addAttribute("code", -27003);
            log.info("title is null");
            return "/common/failure";
        }

        try {
            //标签重名查询-精确查询
            Map<String, Object> paramList = DynamicUtil.getLabelListParam(null, label.getOid(), label.getTid(), label.getTitle(), null, Constants.QUERY_REFINED);
            log.info(" paramList size =  " + paramList.size());
            //获取list
            List<Long> labelIdList = labelService.getIdsByDynamicCondition(Label.class, paramList, 0, Integer.MAX_VALUE);
            if (CollectionUtils.isEmpty(labelIdList)) {
                log.info(" label title is unique ");
            } else {
                log.info(" label title is not unique ");
                model.addAttribute("code", -27009);
                return "/common/failure";
            }

            label.setId(null);
            label.setQuestionNum(0L);
            label.setStatus(Constants.STATUS_SHOW);
            String userId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            label.setCreateBy(Long.parseLong(userId));
            label.setUpdateBy(Long.parseLong(userId));

            Long id = labelService.insert(label);
            if (DataUtils.isNullOrEmpty(id)) {
                log.info("add label error ");
                model.addAttribute("code", -1);
                return "/common/failure";
            }
            model.addAttribute("code", 0);
            model.addAttribute("id", id);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add label error ");
            model.addAttribute("code", -1);
        }

        return "/common/postSuccess";
    }

    /**
     * 2 获取标签详情
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label/{id}", method = RequestMethod.GET)
    public String getLabelJson(HttpServletRequest request,
                               HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("get data : id= " + id);

        if (id == null) {
            model.addAttribute("code", -27005);
            log.info("id is null");
            return "/common/failure";
        }

        try {
            Label label = labelService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(label)) {
                log.info("get label error :label == null");
                model.addAttribute("code", -27004);
                return "/common/failure";
            } else {
                log.info("get label data is " + label);
            }

            List<Long> questionIdList = new ArrayList<>();

            List<QuestionLabelRelation> questionLabelRelationList = new ArrayList<>();
            List<Question> questionList = new ArrayList<>();


            //动态查询-获取参数列表
            Map<String, Object> paramList = DynamicUtil.getQuestionLabelRelationListParam(id.toString(), null);
            log.info(" paramList size =  " + paramList.size());

            //获取关系list
            List<Long> questionLabelRelationIdList = questionLabelRelationService.getIdsByDynamicCondition(QuestionLabelRelation.class, paramList, 0, Integer.MAX_VALUE);
            if (CollectionUtils.isEmpty(questionLabelRelationIdList)) {
                log.info(" questionLabelRelationIdList is null ");
            } else {
                log.info(" questionLabelRelationIdList size =  " + questionLabelRelationIdList.size());
                questionLabelRelationList = questionLabelRelationService.getObjectsByIds(questionLabelRelationIdList);

            }

            for (QuestionLabelRelation questionLabelRelation : questionLabelRelationList) {
                if (questionLabelRelation.getQuestionId() != null) {
                    questionIdList.add(questionLabelRelation.getQuestionId());
                }
            }

            if (CollectionUtils.isEmpty(questionIdList)) {
                log.info(" questionIdList is null ");
            } else {
                log.info(" questionIdList size =  " + questionIdList.size());
                questionList = questionService.getObjectsByIds(questionIdList);
            }

            model.addAttribute("code", 0);
            model.addAttribute("label", label);
            model.addAttribute("questionLabelRelationList", questionLabelRelationList);
            model.addAttribute("questionList", questionList);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get label error,id is  " + id);
            model.addAttribute("code", -1);
        }

        return "/common-skill-service/label/json/labelDetailJson";
    }

    /**
     * 3 修改标签
     *
     * @param request
     * @param response
     * @param model
     * @param label
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label/{id}", method = RequestMethod.PUT)
    public String updateLabelJson(HttpServletRequest request,
                                  HttpServletResponse response, ModelMap model, Label label,
                                  @PathVariable Long id) throws Exception {

        log.info("update label : label= " + label);
        if (id == null) {
            model.addAttribute("code", -27005);
            log.info("id is null");
            return "/common/failure";
        }

        if (label.getOid() == null) {
            model.addAttribute("code", -27001);
            log.info("oid is null");
            return "/common/failure";
        }

        if (label.getTid() == null) {
            model.addAttribute("code", -27002);
            log.info("tid is null");
            return "/common/failure";
        }
        if (label.getTitle() == null) {
            model.addAttribute("code", -27003);
            log.info("title is null");
            return "/common/failure";
        }

        try {

            Label oldLabel = labelService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(oldLabel)) {
                log.info("get oldLabel error :oldLabel == null");
                model.addAttribute("code", -27004);
                return "/common/failure";
            } else {
                log.info("get oldLabel data is " + oldLabel);
            }

            //标签重名查询-精确查询
            Map<String, Object> paramList = DynamicUtil.getLabelListParam(null, label.getOid(), label.getTid(), label.getTitle(), null, Constants.QUERY_REFINED);
            log.info(" paramList size =  " + paramList.size());
            //获取list
            List<Long> labelIdList = labelService.getIdsByDynamicCondition(Label.class, paramList, 0, Integer.MAX_VALUE);
            if (CollectionUtils.isEmpty(labelIdList) || (labelIdList.size() == 1 && labelIdList.get(0).equals(id))) {
                log.info(" label title is unique ");
            } else {
                log.info(" label title is not unique ");
                model.addAttribute("code", -27009);
                return "/common/failure";
            }


            oldLabel.setTid(label.getOid());
            oldLabel.setTid(label.getTid());
            oldLabel.setTitle(label.getTitle());

            String userId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            oldLabel.setUpdateBy(Long.parseLong(userId));

            boolean isUpdate = labelService.update(oldLabel);
            if (isUpdate) {
                log.info(" update success.");
            } else {
                model.addAttribute("code", -1);
                return "/common/failure";
            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update label error,id is  " + label.getId());
            model.addAttribute("code", -1);

        }

        return "/common/success";
    }

    /**
     * 4 上架/下架标签
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @param status
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label/{id}/status/{status}", method = RequestMethod.PUT)
    public String updateLabelJson(HttpServletRequest request,
                                  HttpServletResponse response, ModelMap model,
                                  @PathVariable Long id, @PathVariable Integer status
    ) throws Exception {

        log.info("label id is: " + id + ". status is: " + status);

        if (id == null) {
            model.addAttribute("code", -27005);
            log.info("id is null");
            return "/common/failure";
        }

        if (status == null) {
            model.addAttribute("code", -27006);
            log.info("status is null");
            return "/common/failure";
        }

        try {

            Label label = labelService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(label)) {
                log.info("get label error ");
                model.addAttribute("code", -27004);
                return "/common/failure";
            } else {
                log.info("get label data is " + label);
            }

            if (status.equals(label.getStatus())) {
                model.addAttribute("code", -27007);
                return "/common/failure";
            } else {
                log.info("label status is different from database");
            }

            label.setStatus(status);
            Boolean isUpdate = labelService.update(label);
            if (isUpdate) {
                log.info(" update success.");
            } else {
                model.addAttribute("code", -1);
                return "/common/failure";
            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update label error,id is  " + id);
            model.addAttribute("code", -1);

        }

        return "/common/success";
    }


    /**
     * 5 删除标签
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label", method = RequestMethod.DELETE)
    public String deleteLabelJson(HttpServletRequest request,
                                  HttpServletResponse response, ModelMap model, Long[] id)
            throws Exception {


        if (DataUtils.isNullOrEmpty(id)) {
            model.addAttribute("code", -27005);
            log.info("id is null");
            return "/common/failure";
        }
        List<Long> list = Arrays.asList(id);

        try {
            List<Label> labelList = labelService.getObjectsByIds(list);
            if (CollectionUtils.isEmpty(labelList)) {
                log.info(" labelList is null ");
            } else {
                log.info(" labelList size =  " + labelList.size());
                //判断是否有关联问题
                for (Label label : labelList) {
                    if (label.getQuestionNum() > 0) {
                        model.addAttribute("code", -27008);
                        log.info("label questionNum > 0");
                        return "/common/failure";
                    }
                }
            }

            labelService.deleteList(Label.class, list);
            log.info("delete  success");
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("delete  error,list is  " + list);
            model.addAttribute("code", -1);

        }
        return "/common/success";
    }

    /**
     * 6 搜索标签列表
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @param oid
     * @param tid
     * @param title
     * @param status
     * @param page
     * @param size
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/label/search", method = RequestMethod.GET)
    public String searchLabelList(
            HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Long id, Long oid, Long tid, String title,
            Integer status, Integer page, Integer size)
            throws Exception {

        //打印入参
        log.info("============================================");
        String paramStr = LogUtil.convertParam("/a/u/label/search  by==id", id, "oid", oid, "tid", tid, "page", page, "size", size);
        log.info(paramStr);

        //初始化page size start
        if (page == null) {
            page = 1;
        }
        if (size == null) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        String pageStr = LogUtil.convertParam("page", page, "start", start, "size", size);
        log.info(pageStr);

        try {
            //初始值
            List<Long> labelIdList = new ArrayList<>();
            List<Long> oidList = new ArrayList<>();
            List<Long> tidList = new ArrayList<>();

            List<Label> labelList = new ArrayList<>();
            List<Occupation> occupationList = new ArrayList<>();
            List<Task> taskList = new ArrayList<>();
            //动态查询-获取参数列表
            Map<String, Object> paramList = DynamicUtil.getLabelListParam(id, oid, tid, title, null, Constants.QUERY_FUZZY);
            log.info(" paramList size =  " + paramList.size());
            //获取list
            labelIdList = labelService.getIdsByDynamicCondition(Label.class, paramList, start, size);
            if (CollectionUtils.isEmpty(labelIdList)) {
                log.info(" labelIdList are null ");
                model.addAttribute("code", 0);
                model.addAttribute("size", 0);
                model.addAttribute("total", 0);
                return "/common-skill-service/label/json/labelListJson";
            } else {
                log.info(" labelIdList size =  " + labelIdList.size());

            }
            //获取total
            List<Long> allLabel = labelService.getIdsByDynamicCondition(Label.class, paramList, 0, Integer.MAX_VALUE);
            log.info("total is:" + allLabel.size());

            //获取相关关系list
            labelList = labelService.getObjectsByIds(labelIdList);
            for (Label label : labelList) {
                if (label.getOid() != null) {
                    oidList.add(label.getOid());
                }
                if (label.getTid() > 0) {
                    tidList.add(label.getTid());
                }

            }

            if (CollectionUtils.isEmpty(oidList)) {
                log.info(" oidList is null ");
            } else {
                log.info(" oidList size =  " + oidList.size());
                occupationList = occupationService.getObjectsByIds(oidList);
            }

            if (CollectionUtils.isEmpty(tidList)) {
                log.info(" tidList is null ");
            } else {
                log.info(" tidList size =  " + tidList.size());
                taskList = taskService.getObjectsByIds(tidList);
            }


            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", labelList.size());
            model.addAttribute("total", allLabel.size());
            model.addAttribute("labelList", labelList);
            model.addAttribute("occupationList", occupationList);
            model.addAttribute("taskList", taskList);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get label list error,page is " + page + " , size " + size);
            model.addAttribute("code", -1);
        }

        return "/common-skill-service/label/json/labelListJson";
    }
}

