package com.ptteng.academy.admin.util;

import com.gemantic.common.util.PasswordUtils;
import com.ptteng.academy.user.model.User;
import com.qding.common.util.DataUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.hssf.usermodel.HSSFDataFormatter;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 解析盘库excel和倒查excel
 *
 * @author liut
 */
public class ParseExcel {


    private static final Log log = LogFactory.getLog("parseExcel");

    /**
     * 上传的临时文件在服务器上的存放路径
     */
    public static final String FILE_PATH = "/data/temp/academy";

    /**
     * 本项目excel相关，根目录
     */
    public static final String excelPath = "/data/webs/academy-admin-web/excel";
    /**
     * 审计详情第一部分的excel模版
     */
    public static final String report_part1 = "report_part1_template.xls";
    /**
     * 审计详情第二部分的excel模版
     */
    public static final String report_part2 = "report_part2_template.xls";
    /**
     * 厂家任务列表的excel模版
     */
    public static final String venderTask_list = "venderTask_list_template.xls";

    /**
     * 2003和2007excel文件的尾缀
     */
    public static final String[] EXCEL_SUFFIX = {"xls", "xlsx", "csv"};

    /**
     * 用户excel表中有用的五个字段，按序排列
     */
    static final int[] inventory_colums = {0, 1, 2, 3, 4, 5};

    /**
     * 倒查模板表中有用的三个字段，按序排列
     */
    static final int[] inspect_colums = {3, 7, 13};

    /**
     * 私有化构造器
     */
    private ParseExcel() {
    }

    /**
     * 解析excel
     *
     * @param inventoryUrl
     * @return
     */
    public static List<User> parseUserExcel(String inventoryUrl) {
        List<User> list = new ArrayList<>();
        log.info("argument inventoryUrl is: " + inventoryUrl);
        if (StringUtils.isEmpty(inventoryUrl)) {
            log.info("arg inventoryUrl is empty");
            return list;
        }
        //服务器用
        Workbook wb = getWb(getExcelFilePath(inventoryUrl));
        //本地调试用
//    Workbook wb = getWb(inventoryUrl);
        if (wb == null) {
            log.info("no workBook, can not parse");
            return list;
        }
        /* 固定只取出第一张sheet */
        Sheet sheet_0 = wb.getSheetAt(0);
        if (sheet_0 == null) {
            log.info("no first sheet, can not parse");
            return list;
        }
        int rowNum = sheet_0.getLastRowNum() + 1; // 共有多少行
        Row row_0 = sheet_0.getRow(0); // 取出第一行，标题栏
        int colNum = row_0.getLastCellNum() + 1; // 共有多少列
        log.info("total rowNum: " + rowNum + ", total colNum: " + colNum);

        /* 从第二行开始遍历每一行每一列的数据 */
        for (int i = 1; i < rowNum; i++) {
            log.info("============ i ============ " + i);
            Row row = sheet_0.getRow(i);
            User user = new User();
            for (int j = 0; j < colNum; j++) {
                Cell cell = row.getCell(j);
                log.info("j :" + j + ", cell is " + cell);

                if (DataUtils.isNullOrEmpty(cell)) {
                    continue;
                }
                if (DataUtils.isNullOrEmpty(getValue(cell))) {
                    continue;
                }
                if (j == inventory_colums[0]) {
                    //设置年级
                    log.info("年级：" + cell + "----->" + Integer.valueOf(getValue(cell)));
                    user.setGrade(Integer.valueOf(getValue(cell)));
                } else if (j == inventory_colums[1]) {
                    //设置班级
                    log.info("班级：" + cell + "----->" + getValue(cell));
                    user.setClassName(getValue(cell));
                } else if (j == inventory_colums[2]) {
                    //设置姓名
                    log.info("姓名：" + cell + "----->" + getValue(cell));
                    user.setAlias(getValue(cell));
                } else if (j == inventory_colums[3]) {
                    //设置手机号
                    HSSFDataFormatter dataFormatter = new HSSFDataFormatter();
                    String cellFormatted = dataFormatter.formatCellValue(cell);
                    user.setMobile(cellFormatted);

                } else if (j == inventory_colums[4]) {
                    //设置邮件
                    log.info("邮件：" + cell + "----->" + getValue(cell));
                    user.setMail(getValue(cell));
                } else if (j == inventory_colums[5]) {
                    //设置密码
                    log.info("密码：" + cell + "----->" + getValue(cell));
                    String pwd = getValue(cell);
                    String encodePass = PasswordUtils.encode("000000");
                    if (pwd != null) {
                        encodePass = PasswordUtils.encode(pwd);
                    }
                    user.setPwd(encodePass);
                } else {
                    continue;
                }
            }

            if (DataUtils.isNotNullOrEmpty(user.getMobile()) || DataUtils.isNotNullOrEmpty(user.getMail())) {
                list.add(user);
                log.info("mobile is not null " + list);
            }

        }
        log.info(list);
        return list;
    }
//
//  /**
//   * 解析倒查excel
//   *
//   * @param inventoryUrl
//   * @return
//   */
//  public static List<Inspect> parseInspectExcel(String inspectUrl) {
//    List<Inspect> list = new ArrayList<>();
//    log.info("argument inspectUrl is: " + inspectUrl);
//    if (StringUtils.isEmpty(inspectUrl)) {
//      log.info("arg inspectUrl is empty");
//      return list;
//    }
//    Workbook wb = getWb(getExcelFilePath(inspectUrl));
//    if (wb == null) {
//      log.info("no workBook, can not parse");
//      return list;
//    }
//    /* 固定只取出第一张sheet */
//    Sheet sheet_0 = wb.getSheetAt(0);
//    if (sheet_0 == null) {
//      log.info("no first sheet, can not parse");
//      return list;
//    }
//    int rowNum = sheet_0.getLastRowNum() + 1; // 共有多少行
//    Row row_0 = sheet_0.getRow(0); // 取出第一行，标题栏
//    int colNum = row_0.getLastCellNum() + 1; // 共有多少列
//    log.info("total rowNum: " + rowNum + ", total colNum: " + colNum);
//
//    /* 从第二行开始遍历每一行每一列的数据 */
//    for (int i = 1; i < rowNum; i++) {
//      Row row = sheet_0.getRow(i);
//      Inspect ins = new Inspect();
//      for (int j = 0; j < colNum; j++) {
//        Cell cell = row.getCell(j);
//        if (j == inspect_colums[0]) {
//          ins.setDistributorName(getValue(cell));
//        }
//        else if (j == inspect_colums[1]) {
//          ins.setVinNo(getValue(cell));
//        }
//        else if (j == inspect_colums[2]) {
//          ins.setSettlementDate(getValue(cell));
//        }
//        else {
//          continue;
//        }
//      }
//      list.add(ins);
//    }
//    return list;
//  }
//

    /**
     * 每个cell中的值
     *
     * @param cell
     * @return
     */
    private static String getValue(Cell cell) {

        log.info("取值为 " + cell);
        if (DataUtils.isNullOrEmpty(cell)) {

            return "";
        }
        if (cell.getCellType() == Cell.CELL_TYPE_BOOLEAN) {
            return String.valueOf(cell.getBooleanCellValue());
        } else if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
            if (HSSFDateUtil.isCellDateFormatted(cell)) {
                Date d = cell.getDateCellValue();
                return DateFormatUtils.format(d, "yyyy/MM/dd");
            } else {
                return String.valueOf((int) cell.getNumericCellValue());
            }
        } else {
            return String.valueOf(cell.getStringCellValue());
        }
    }

    /**
     * 拿到workbook对象
     *
     * @param excelFilePath
     * @return
     */
    private static Workbook getWb(String excelFilePath) {
        String suffix = excelFilePath.substring(excelFilePath.lastIndexOf(".") + 1);
        log.info("suffix is: " + suffix);
        InputStream is;
        Workbook wb = null;
        try {
            is = new FileInputStream(excelFilePath);
            if (EXCEL_SUFFIX[0].equals(suffix)) {
                log.info("file is xls");
                wb = new HSSFWorkbook(is);
            } else if (EXCEL_SUFFIX[1].equals(suffix)) {
                log.info("file is xlsx");
                wb = new XSSFWorkbook(is);
            } else {
                log.info("file is other, illegal");
                wb = null;
            }
        } catch (FileNotFoundException ex) {
            log.error("getWb error...", ex);
            wb = null;
        } catch (IOException ex) {
            log.error("getWb error...", ex);
            wb = null;
        }
        return wb;
    }

    /**
     * 根据url拼接excel文件在服务器上的临时存放路径
     *
     * @param url
     * @return
     */
    public static String getExcelFilePath(String url) {
        log.info("argument url is: " + url);
        int begin = url.lastIndexOf("/") - 1;
        String temp = url.substring(begin, url.length());
        log.info("temp is:" + temp);
        return FILE_PATH + "/" + temp;
    }

    public static void main(String[] args) {
    }
}
