package com.ptteng.model;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.util.MyListUtil;
import com.ptteng.util.TimeUtil;
import com.ptteng.yi.nucleus.model.EnrollmentDetail;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.math.BigDecimal;
import java.util.List;

public class SchoolFinance {
    private static final Log log = LogFactory.getLog(SchoolFinance.class);

    private BigDecimal prepareWithdraw;
    private BigDecimal withdrawalBeforeZero;
    private BigDecimal withdrawalAfterZero;

    public SchoolFinance() {
        this.prepareWithdraw = new BigDecimal("0");
        this.withdrawalBeforeZero = new BigDecimal("0");
        this.withdrawalAfterZero = new BigDecimal("0");
    }

    public BigDecimal getPrepareWithdraw() {
        return prepareWithdraw;
    }

    public void setPrepareWithdraw(BigDecimal prepareWithdraw) {
        this.prepareWithdraw = prepareWithdraw;
    }

    public BigDecimal getWithdrawalBeforeZero() {
        return withdrawalBeforeZero;
    }

    public void setWithdrawalBeforeZero(BigDecimal withdrawalBeforeZero) {
        this.withdrawalBeforeZero = withdrawalBeforeZero;
    }

    public BigDecimal getWithdrawalAfterZero() {
        return withdrawalAfterZero;
    }

    public void setWithdrawalAfterZero(BigDecimal withdrawalAfterZero) {
        this.withdrawalAfterZero = withdrawalAfterZero;
    }

    @Override
    public String toString() {
        return "SchoolFinance{" + "prepareWithdraw=" + prepareWithdraw + ", withdrawalBeforeZero="
                + withdrawalBeforeZero + ", withdrawalAfterZero=" + withdrawalAfterZero + '}';
    }

    /**
     * 获取驾校财务记录方法
     *
     * @param prepareReceiveList 待提现的记录集合
     * @param receivedList 可提现的记录集合
     * @return SchoolFinance 驾校统计对象
     * @throws ServiceDaoException
     *
     * */
    public static SchoolFinance getSchoolFinanceRecord(List<EnrollmentDetail> prepareReceiveList,
            List<EnrollmentDetail> receivedList) throws ServiceDaoException {
        SchoolFinance schoolFinance = new SchoolFinance();

        try {
            schoolFinance.setPrepareWithdraw(calculatePrepareAmount(prepareReceiveList));

            BigDecimal[] result = calculateAvailableAmount(receivedList);
            schoolFinance.setWithdrawalBeforeZero(result[0]);
            schoolFinance.setWithdrawalAfterZero(result[1]);
        } catch (Throwable e) {
            log.error(" calculatePrepareAmount error");
            log.error(e);
            e.printStackTrace();
            throw new ServiceDaoException(e);
        }

        return schoolFinance;
    }

    /**
     * 计算驾校待提现的金额
     *
     * @param enrollmentDetailList 待提现的记录集合
     * @return BigDecimal 待提现的总金额
     * */
    public static BigDecimal calculatePrepareAmount(List<EnrollmentDetail> enrollmentDetailList)
            throws IllegalAccessException, NoSuchFieldException {
        BigDecimal result = new BigDecimal("0");

        if (CollectionUtils.isEmpty(enrollmentDetailList)) {
            return result;
        }

        List<BigDecimal> prepareWithdrawFieldList = MyListUtil.getFieldValueListFromModelList(enrollmentDetailList,
                false, EnrollmentDetail.class.getDeclaredField("actualIncome"));

        for (int i = 0; i < prepareWithdrawFieldList.size(); i++) {
            result = result.add(prepareWithdrawFieldList.get(i));
        }

        return result;
    }

    /**
     * 计算驾校可提现的金额
     *
     * @param enrollmentDetailList 可提现的记录集合
     * @return BigDecimal[] 可提现的总金额数组,包括0点前和0点之后
     * */
    public static BigDecimal[] calculateAvailableAmount(List<EnrollmentDetail> enrollmentDetailList) {
        BigDecimal[] result = { new BigDecimal("0"), new BigDecimal("0") };

        if (CollectionUtils.isEmpty(enrollmentDetailList)) {
            return result;
        }

        for (int i = 0; i < enrollmentDetailList.size(); i++) {
            EnrollmentDetail enrollmentDetail = enrollmentDetailList.get(i);

            if (enrollmentDetail.getWithdrawStatus() == EnrollmentDetail.NOT_WITHDRAW
                    && enrollmentDetail.getStatus() == EnrollmentDetail.HAS_RECEIVED) {
                if (enrollmentDetail.getReceiveTime() <= TimeUtil.getTimeStampOfToday()) {
                    result[0] = result[0].add(enrollmentDetail.getActualIncome());
                } else {
                    result[1] = result[1].add(enrollmentDetail.getActualIncome());
                }
            }
        }

        return result;
    }
}
