package com.ptteng.controller;

import com.ptteng.util.DynamicUtil;
import com.ptteng.util.ManagerUtil;
import com.ptteng.util.ParamsUtil;
import com.ptteng.yi.nucleus.model.*;
import com.ptteng.yi.nucleus.service.*;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * @Description: 包含提现相关接口
 * @author: zhangxin
 * @Date: 2017-10-12
 */
@Controller
public class WithdrawController {
    private static final Log log = LogFactory.getLog(WithdrawController.class);

    @Resource
    private WithdrawService withdrawService;

    @Resource
    private EnrollmentDetailService enrollmentDetailService;

    @Resource
    private SettlementService settlementService;

    @Resource
    private RecordService recordService;

    @Resource
    private SchoolService schoolService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 新增提现订单接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/withdrawal", method = RequestMethod.POST)
    public String addWithdrawOrder(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @RequestParam Long[] idArray, Long schoolId, BigDecimal withdrawAmount) throws Exception {
        log.info("Interface POST /a/u/withdrawal, Method addWithdrawOrder() parameters: schoolId is : " + schoolId
                + ", withdrawAmount: " + withdrawAmount + ", idArray: " + Arrays.toString(idArray));

        int validateResult = ParamsUtil.checkParametersOfAddWithdrawOrder(schoolId, withdrawAmount, idArray);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            School school = schoolService.getObjectById(schoolId);
            if (DataUtils.isNullOrEmpty(school)) {
                log.info("Can not find school by schoolId, schoolId is : " + schoolId);
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("get school data by schoolId, school id is : " + school.getId());

            List<EnrollmentDetail> enrollmentDetailList = enrollmentDetailService
                    .getObjectsByIds(Arrays.asList(idArray));
            if (CollectionUtils.isEmpty(enrollmentDetailList)) {
                log.info("Can not query data by idList, idList is : " + Arrays.asList(idArray));
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("get enrollmentDetail list by idList, list size is : " + enrollmentDetailList.size());

            Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
            Long withdrawId = withdrawService.generateWithdraw(school, withdrawAmount, managerId, Withdraw.NOT_DEAL);
            log.info("generate withdraw success, withdraw Id is : " + withdrawId);

            for (int i = 0; i < enrollmentDetailList.size(); i++) {
                enrollmentDetailList.get(i).setWithdrawId(withdrawId);
                enrollmentDetailList.get(i).setWithdrawStatus(EnrollmentDetail.WITHDRAWING);
                enrollmentDetailList.get(i).setUpdateBy(managerId);
            }
            enrollmentDetailService.updateList(enrollmentDetailList);
            log.info("update enrollmentDetailList success, list size is " + enrollmentDetailList.size());

            model.addAttribute("code", 0);
            model.addAttribute("id", withdrawId);
            return "/common/insert";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method addWithdrawOrder() failed, schoolId: " + schoolId + ", withdrawAmount: " + withdrawAmount
                    + ", idArray: " + Arrays.toString(idArray));
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 查询提现订单列表接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/withdrawal/list", method = RequestMethod.GET)
    public String getWithdrawalList(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Integer page, Integer size, Long schoolId, Long startTime, Long endTime, Integer status) throws Exception {
        log.info("Interface GET /a/u/withdrawal/list, Method getWithdrawalList() parameters, schoolId: " + schoolId
                + ", startTime: " + startTime + ", endTime: " + endTime + ", status: " + status);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int start = ParamsUtil.calculateStartPosition(page, actualSize);
        log.info("pageList is : page : " + start + ", size :" + actualSize);

        int validate = ParamsUtil.checkParametersOfGetWithdrawalList(status);
        if (validate != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validate);
            return "/common/failure";
        }

        try {
            Map<String, Object> queryWithdrawMap = DynamicUtil.getWithdrawListParams(startTime, endTime, schoolId,
                    status, false);
            List<Long> withdrawIdList = withdrawService.getIdsByDynamicCondition(Withdraw.class, queryWithdrawMap,
                    start, actualSize);
            log.info("get withdraw Id list, list is : " + withdrawIdList);

            List<Withdraw> withdrawList = null;
            Long total = 0L;
            if (CollectionUtils.isNotEmpty(withdrawIdList)) {
                withdrawList = withdrawService.getObjectsByIds(withdrawIdList);
                log.info("get withdraw list, list size is : " + withdrawList.size());

                Map<String, Object> countWithdrawMap = DynamicUtil.getWithdrawListParams(startTime, endTime, schoolId,
                        status, true);
                total = withdrawService.getIdsByDynamicCondition(Withdraw.class, countWithdrawMap, 0, Integer.MAX_VALUE)
                        .get(0);
                log.info("get withdraw list total is : " + total);
            }

            model.addAttribute("code", 0);
            model.addAttribute("size", actualSize);
            model.addAttribute("total", total);
            model.addAttribute("withdrawList", withdrawList);
            return "/yi-nucleus-service/withdraw/json/withdrawListJson";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getWithdrawalList() failed, page is : " + page + " size is : " + size);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 9.处理提现订单接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/withdrawal/{id}", method = RequestMethod.PUT)
    public String dealWithdrawOrder(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, @RequestParam Long[] idArray, Integer operate) throws Exception {

        log.info("Interface PUT /a/u/withdrawal/{id}, Method dealWithdrawOrder() parameter, id: " + id + ", operate: "
                + operate + ", idArray: " + Arrays.toString(idArray));

        int validateResult = ParamsUtil.checkParametersOfDealWithdrawOrder(operate, idArray);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            Withdraw withdrawal = withdrawService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(withdrawal)) {
                log.info("Can not find data by withdraw id, withdraw id is : " + id);
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("Get withdraw object by id, id is : " + withdrawal.getId());

            List<EnrollmentDetail> enrollmentDetailList = enrollmentDetailService
                    .getObjectsByIds(Arrays.asList(idArray));
            if (CollectionUtils.isEmpty(enrollmentDetailList)) {
                log.info("Can not query data by idList, idList is : " + Arrays.asList(idArray));
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("get enrollmentDetail list by idList, list size is : " + enrollmentDetailList.size());

            Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
            withdrawal.setStatus(Withdraw.HAS_DEAL);
            withdrawal.setUpdateBy(managerId);
            withdrawService.update(withdrawal);
            log.info("update withdraw's status success");

            Long settlementId = settlementService.generateSettlement(withdrawal, managerId, operate);
            log.info("insert settlement success, settlement id is : " + settlementId);

            Long recordId = 0L;
            if (operate.equals(Settlement.HAS_CONSENT)) {
                recordId = recordService.generateRecord(withdrawal, managerId, Record.EXPENSES);
                log.info("insert record success, record id is : " + recordId);
            }

            for (int i = 0; i < enrollmentDetailList.size(); i++) {
                EnrollmentDetail enrollmentDetail = enrollmentDetailList.get(i);
                enrollmentDetail.setSettlementId(settlementId);
                if (operate.equals(Settlement.HAS_CONSENT)) {
                    enrollmentDetail.setExpensesId(recordId);
                }
                enrollmentDetailList.get(i).setWithdrawStatus(operate);
                enrollmentDetailList.get(i).setUpdateBy(managerId);
            }
            enrollmentDetailService.updateList(enrollmentDetailList);
            log.info("update enrollmentDetailList success, list size is " + enrollmentDetailList.size());

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method dealWithdrawOrder() failed, parameter id is : " + id);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }
}
