package com.ptteng.controller;

import com.ptteng.util.DynamicUtil;
import com.ptteng.util.ManagerUtil;
import com.ptteng.util.ParamsUtil;
import com.ptteng.yi.nucleus.model.SubjectBroadcast;
import com.ptteng.yi.nucleus.service.SubjectBroadcastService;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * SubjectBroadcast  crud
 *
 * @author magenm
 * @Date 2014-4-16 13:43
 */
@Controller
public class SubjectBroadcastController {
    private static final Log log = LogFactory.getLog(SubjectBroadcastController.class);

    @Resource
    private SubjectBroadcastService subjectBroadcastService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 查询音频列表接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio/list", method = RequestMethod.GET)
    public String getSubjectBroadcastList(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String title, Integer subjectType, Integer type, Integer page, Integer size) throws Exception {

        log.info("Interface GET /a/u/audio/list, Method getSubjectBroadcastList() parameters: title: " + title
                + ", type: " + type + ", subjectType: " + subjectType);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int startPosition = ParamsUtil.calculateStartPosition(page, actualSize);
        log.info("page: " + startPosition + ", size: " + actualSize);

        int validateResult = ParamsUtil.checkParametersOfGetSubjectBroadcastList(subjectType, type);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            Map<String, Object> querySubjectBroadcastMap = DynamicUtil.getSubjectBroadcastListParams(title, subjectType,
                    type, false);
            List<Long> subjectBroadcastIdList = subjectBroadcastService.getIdsByDynamicCondition(SubjectBroadcast.class,
                    querySubjectBroadcastMap, startPosition, actualSize);
            log.info("get subjectBroadcast id list, list size is : " + subjectBroadcastIdList.size());

            List<SubjectBroadcast> subjectBroadcastList = null;
            Long total = 0L;
            if (CollectionUtils.isNotEmpty(subjectBroadcastIdList)) {
                subjectBroadcastList = subjectBroadcastService.getObjectsByIds(subjectBroadcastIdList);
                log.info("get subjectBroadcast data list, list size is: " + subjectBroadcastList.size());

                Map<String, Object> countSubjectBroadcastMap = DynamicUtil.getSubjectBroadcastListParams(title,
                        subjectType, type, true);
                total = subjectBroadcastService.getIdsByDynamicCondition(SubjectBroadcast.class,
                        countSubjectBroadcastMap, 0, Integer.MAX_VALUE).get(0);
                log.info("count subjectBroadcast data total is : " + total);
            }

            model.addAttribute("code", 0);
            model.addAttribute("size", actualSize);
            model.addAttribute("total", total);
            model.addAttribute("subjectBroadcastList", subjectBroadcastList);
            return "/yi-nucleus-service/subjectBroadcast/json/subjectBroadcastListJson";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getSubjectBroadcastList() failed, parameters: title: " + title + ", subjectType: "
                    + subjectType);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 查询音频详情接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio/{id}", method = RequestMethod.GET)
    public String getSubjectBroadcastDetail(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("Interface GET /a/u/audio/{id}, Method getSubjectBroadcastDetail() parameters: id: " + id);

        if (id <= 0) {
            log.info("Error value of parameter id, id is: " + id);
            model.addAttribute("code", -1001);
            return "/common/failure";
        }

        try {
            SubjectBroadcast subjectBroadcast = subjectBroadcastService.getObjectById(id);
            log.info("get subjectBroadcast data, subjectBroadcast is:" + subjectBroadcast);

            model.addAttribute("code", 0);
            model.addAttribute("subjectBroadcast", subjectBroadcast);
            return "/yi-nucleus-service/subjectBroadcast/json/subjectBroadcastDetailJson";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getSubjectBroadcastDetail() failed, id : " + id);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 新增音频接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio", method = RequestMethod.POST)
    public String addSubjectBroadcast(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String icon, String title, String voiceUrl, Integer subjectType, Integer type) throws Exception {
        log.info("Interface POST /a/u/audio, Method addSubjectBroadcast() parameters: icon : " + icon + ", title: "
                + title + ", voiceUrl: " + voiceUrl + ", subjectType: " + subjectType);

        int validateResult = ParamsUtil.checkParametersOfModifySubjectBroadcast(icon, title, voiceUrl, subjectType,
                type);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
        SubjectBroadcast subjectBroadcast = new SubjectBroadcast();
        subjectBroadcast.setIcon(icon);
        subjectBroadcast.setTitle(title);
        subjectBroadcast.setSort(SubjectBroadcast.SORT_INIT_VALUE);
        subjectBroadcast.setType(type);
        subjectBroadcast.setSubjectType(subjectType);
        subjectBroadcast.setVoiceUrl(voiceUrl);
        subjectBroadcast.setCreateBy(managerId);
        subjectBroadcast.setUpdateBy(managerId);
        try {
            Long subjectBroadcastId = subjectBroadcastService.insert(subjectBroadcast);
            log.info("Insert subjectBroadcast success, subjectBroadcast id is : " + subjectBroadcastId);

            model.addAttribute("code", 0);
            model.addAttribute("id", subjectBroadcastId);
            return "/common/insert";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method addSubjectBroadcast() failed, parameters: icon : " + icon + ", title: " + title
                    + ", voiceUrl: " + voiceUrl);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 编辑音频接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio/{id}", method = RequestMethod.PUT)
    public String updateSubjectBroadcast(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, String icon, String title, String voiceUrl, Integer subjectType, Integer type)
            throws Exception {
        log.info("Interface PUT /a/u/audio/{id}, Method addSubjectBroadcast() parameters: icon : " + icon + ", title: "
                + title + ", voiceUrl: " + voiceUrl + ", subjectType: " + subjectType + ", type: " + type);

        int validateResult = ParamsUtil.checkParametersOfModifySubjectBroadcast(icon, title, voiceUrl, subjectType,
                type);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            SubjectBroadcast subjectBroadcast = subjectBroadcastService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(subjectBroadcast)) {
                log.info("Can not find subjectBroadcast data by id, id is : " + id);
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("get subjectBroadcast data, subjectBroadcast is : " + subjectBroadcast);

            Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
            subjectBroadcast.setIcon(icon);
            subjectBroadcast.setTitle(title);
            subjectBroadcast.setSubjectType(subjectType);
            subjectBroadcast.setType(type);
            subjectBroadcast.setVoiceUrl(voiceUrl);
            subjectBroadcast.setUpdateBy(managerId);
            subjectBroadcastService.update(subjectBroadcast);
            log.info("update subjectBroadcast success, subjectBroadcast id is : " + subjectBroadcast.getId());

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method updateSubjectBroadcast() failed, parameters: icon : " + icon + ", title: " + title
                    + ", voiceUrl: " + voiceUrl);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 删除音频列表接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio/{id}", method = RequestMethod.DELETE)
    public String deleteSubjectBroadcast(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("Interface DELETE /a/u/audio/{id}, Method deleteSubjectBroadcast() parameter: id: " + id);

        if (id <= 0) {
            log.info("Error value of parameter id: id is: " + id);
            model.addAttribute("code", -1001);
            return "/common/failure";
        }

        try {
            subjectBroadcastService.delete(id);
            log.info("delete subjectBroadcast success, id is : " + id);

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method deleteSubjectBroadcast() failed, parameter: id: " + id);
            model.addAttribute("code", -1);
            return "/common/success";
        }
    }

    /**
     * 音频拖动排序
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */

    @RequestMapping(value = "/a/u/audio/sort", method = RequestMethod.PUT)
    public String sortAudio(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @RequestParam Long[] ids) throws Exception {
        log.info("Interface PUT /a/u/audio/sort, Method sortAudio() parameters : ids : " + Arrays.toString(ids));

        if (ids == null || ids.length == 0) {
            log.info("Parameter ids can not be noll or empty!");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            List<SubjectBroadcast> subjectBroadcastList = subjectBroadcastService.getObjectsByIds(Arrays.asList(ids));
            log.info("get subjectBroadcast list by ids, list size is : " + subjectBroadcastList.size());

            if (subjectBroadcastList.size() < ids.length) {
                log.info("Error value of parameter ids, ids is : " + Arrays.toString(ids));
                model.addAttribute("code", -1001);
                return "/common/failure";
            }

            Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
            for (int i = 0; i < ids.length; i++) {
                SubjectBroadcast subjectBroadcast = subjectBroadcastList.get(i);
                subjectBroadcast.setSort((i + 1) * 10);
                subjectBroadcast.setUpdateBy(managerId);
            }
            subjectBroadcastService.updateList(subjectBroadcastList);
            log.info("update subject broadcast list field sort success");

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method sortAudio() failed, parameters : ids : " + Arrays.toString(ids));
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }
}
