package com.ptteng.controller;

import com.ptteng.util.DynamicUtil;
import com.ptteng.yi.pub.model.Info;
import com.ptteng.yi.pub.service.CitiesService;
import com.ptteng.yi.pub.service.InfoService;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Info  crud
 * 
 * @author wangmeng
 * @Date 2017-12-25 13:43
 * 
 */
@Controller
public class InfoController {
    private static final Log log = LogFactory.getLog(InfoController.class);

    @Autowired
    private InfoService infoService;

    @Autowired
    private CitiesService citiesService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 查看信息详情
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/{id}", method = RequestMethod.GET)
    public String getInfoJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("GET request url : /a/u/info/{id}, get path variable id : " + id);
        try {
            Info info = infoService.getObjectById(id);
            log.info("get info data is " + info);

            if (DataUtils.isNullOrEmpty(info)) {
                log.info("no this info");
                model.addAttribute("code", -11000);
                return "/common/failure";
            }

            model.addAttribute("code", 0);

            model.addAttribute("info", info);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get info error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/yi-pub-service/info/json/infoDetailJson";
    }

    /**
     * 上下线信息
     * @param request
     * @param response
     * @param model
     * @param id
     * @param status 状态
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/{id}/status", method = RequestMethod.PUT)
    public String updateCardTypeStatusJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Integer status) throws Exception {

        log.info("PUT request url: /a/u/info/{id}/status, get path variable : id= " + id + ", parameter : status= "
                + status);

        if (DataUtils.isNullOrEmpty(status)) {
            model.addAttribute("code", -11008);
            log.info("parameter status is null or empty");
            return "/common/failure";
        }

        if (status > Info.STATUS_ONLINE || status < Info.STATUS_OFFLINE) {
            model.addAttribute("code", -11009);
            log.info("parameter status is illegal");
            return "/common/failure";
        }

        try {
            Info info = infoService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(info)) {
                log.info("no this info");
                model.addAttribute("code", -11000);
                return "/common/failure";
            }

            info.setStatus(status);
            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            info.setUpdateBy(Long.parseLong(managerId));

            infoService.update(info);
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update info status error, id is " + id);
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 编辑信息
     * @param request
     * @param response
     * @param model
     * @param id
     * @param info
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/{id}", method = RequestMethod.PUT)
    public String updateInfoJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Info info) throws Exception {

        log.info("PUT request url : /a/u/info/{id}, get path variable id : " + id + ", info: " + info);

        if (DataUtils.isNullOrEmpty(info)) {
            model.addAttribute("code", -11001);
            log.info("parameter info is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getType())) {
            model.addAttribute("code", -11002);
            log.info("parameter type is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getContent())) {
            model.addAttribute("code", -11003);
            log.info("parameter content is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getTitle())) {
            model.addAttribute("code", -11004);
            log.info("parameter title is null or empty");
            return "/common/failure";
        }

        if (info.getType() < Info.TYPE_ENROLLMENT_NOTICE || info.getType() > Info.TYPE_AFTER_SERVICE) {
            model.addAttribute("code", -11005);
            log.info("parameter type is illegal");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getId())) {
            log.info("parameter id is null or empty");
            model.addAttribute("code", -11006);
            return "/common/failure";
        }

        if (!info.getId().equals(id)) {
            log.info("parameter id is not matching the path variable");
            model.addAttribute("code", -11007);
            return "/common/failure";
        }

        try {
            Info info1 = infoService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(info1)) {
                model.addAttribute("code", -11000);
                log.info("no this info");
                return "/common/failure";
            }

            info1.setType(info.getType());
            info1.setContent(info.getContent());
            info1.setTitle(info.getTitle());

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            info1.setUpdateBy(Long.parseLong(managerId));

            infoService.update(info1);

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update info error,id is  " + info.getId());
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 新增信息
     * @param request
     * @param response
     * @param model
     * @param info
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info", method = RequestMethod.POST)
    public String addInfoJson(HttpServletRequest request, HttpServletResponse response, ModelMap model, Info info)
            throws Exception {

        log.info("POST request url : /a/u/info, no parameter");

        if (DataUtils.isNullOrEmpty(info)) {
            model.addAttribute("code", -11001);
            log.info("parameter info is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getType())) {
            model.addAttribute("code", -11002);
            log.info("parameter type is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getContent())) {
            model.addAttribute("code", -11003);
            log.info("parameter content is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getTitle())) {
            model.addAttribute("code", -11004);
            log.info("parameter title is null or empty");
            return "/common/failure";
        }

        if (info.getType() < Info.TYPE_ENROLLMENT_NOTICE || info.getType() > Info.TYPE_AFTER_SERVICE) {
            model.addAttribute("code", -11005);
            log.info("parameter type is illegal");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(info.getCityId())) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -11013);
            return "/common/failure";
        }

        if (info.getCityId() < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -11014);
            return "/common/failure";
        }

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(info.getCityId(), 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            info.setId(null);
            info.setLevel(Info.INITIAL_VALUE);
            info.setStatus(Info.INITIAL_VALUE);

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            info.setUpdateBy(Long.parseLong(managerId));
            info.setCreateBy(Long.parseLong(managerId));

            infoService.insert(info);

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add info error ");
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 删除信息
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/{id}", method = RequestMethod.DELETE)
    public String deleteInfoJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("DELETE request url : /a/u/info/{id}, get path variable id : " + id);
        try {
            infoService.delete(id);

            log.info("delete info success");
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("delete info error, id is " + id);
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 获取信息列表
     * @param request
     * @param response
     * @param model
     * @param title 标题
     * @param type 类型
     * @param status 状态
     * @param cityId 城市id
     * @param page
     * @param size
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/list", method = RequestMethod.GET)
    public String getMultiInfoJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String title, Integer type, Integer status, Long cityId, Integer page, Integer size) throws Exception {

        log.info("GET request url: /a/u/info/list, get parameter title: " + title + ", type: " + type + ", status: "
                + status + ", cityId: " + cityId + ", page: " + page + ", size: " + size);

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -11013);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -11014);
            return "/common/failure";
        }

        if (page == null) {
            page = 1;
        }
        if (size == null || size < 0) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        log.info("pageList : page= " + start + " , size=" + size);

        try {
            Map<String, Object> map = DynamicUtil.getInfoListParams(title, type, status, cityId, false);
            log.info("query info list map : " + map);

            List<Long> infoIds = infoService.getIdsByDynamicCondition(Info.class, map, start, size);
            log.info("get info id list : " + infoIds);

            Integer total = infoService.getIdsByDynamicCondition(Info.class, map, 0, Integer.MAX_VALUE).size();

            if (CollectionUtils.isEmpty(infoIds)) {
                model.addAttribute("infoList", new ArrayList());
            } else {
                List<Info> infoList = infoService.getObjectsByIds(infoIds);
                log.info("get infoList data size is " + infoList.size());

                model.addAttribute("infoList", infoList);
            }

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);

        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("get info list error, start is  " + start + " , size " + size);
            model.addAttribute("code", -100000);
        }

        return "/yi-pub-service/info/json/infoListJson";
    }

    /**
     * 信息拖动排序
     * @param request
     * @param response
     * @param model
     * @param type 类型
     * @param cityId 城市id
     * @param ids 信息id数组
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/info/sort", method = RequestMethod.PUT)
    public String sortBanners(HttpServletRequest request, HttpServletResponse response, ModelMap model, Integer type,
            Long cityId, @RequestParam Long[] ids) throws Exception {

        log.info("PUT request url: /a/u/info/sort, get parameter type : " + type + ", cityId: " + cityId
                + ", ids array size : " + ids.length);

        if (DataUtils.isNullOrEmpty(ids)) {
            log.info("parameter ids is null or empty");
            model.addAttribute("code", -10003);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(type)) {
            log.info("parameter type is null or empty");
            model.addAttribute("code", -11002);
            return "/common/failure";
        }

        if (type < Info.TYPE_ENROLLMENT_NOTICE || type > Info.TYPE_AFTER_SERVICE) {
            model.addAttribute("code", -11005);
            log.info("parameter type is illegal");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        List<Long> idList = Arrays.asList(ids);

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(cityId, 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            Map<String, Object> map = DynamicUtil.getInfoListParams(null, type, null, cityId, false);
            log.info("query info list map : " + map);

            List<Long> onlineIds = infoService.getIdsByDynamicCondition(Info.class, map, 0, Integer.MAX_VALUE);
            log.info("get info id list : " + onlineIds);

            if (CollectionUtils.isEmpty(onlineIds)) {
                log.info("there are no online info");
                model.addAttribute("code", -11012);
                return "/common/failure";
            } else if (!onlineIds.containsAll(idList)) {
                log.info("the available info ids didn't contain all ids in parameter id list");
                model.addAttribute("code", -11011);
                return "/common/failure";
            } else {
                List<Info> infoList = infoService.getObjectsByIds(idList);
                String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
                int index = 0;
                for (Info info : infoList) {
                    info.setLevel(index);
                    info.setUpdateBy(Long.parseLong(managerId));
                    index = index + 10;
                }
                infoService.updateList(infoList);
                model.addAttribute("code", 0);
            }
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("sort info error, ids is  " + idList);
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }

}
