package com.ptteng.controller;

import com.ptteng.model.CenterFinance;
import com.ptteng.model.SchoolFinance;
import com.ptteng.util.*;
import com.ptteng.yi.nucleus.model.EnrollmentDetail;
import com.ptteng.yi.nucleus.service.EnrollmentDetailService;
import com.ptteng.yi.nucleus.service.SchoolService;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Map;

/**
 * @Description: 包含报名学员相关接口
 * @author: zhangxin
 * @Date: 2017-10-12
 */
@Controller
public class EnrollmentDetailController {
    private static final Log log = LogFactory.getLog(EnrollmentDetailController.class);

    @Resource
    private EnrollmentDetailService enrollmentDetailService;

    @Resource
    private SchoolService schoolService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 驾校端查询报名学员接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = " /a/u/school/enrollment/list", method = RequestMethod.GET)
    public String getEnrollmentListBySchool(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                            Integer page, Integer size, Long startTime, Long endTime, Integer type, Integer status, Long schoolId)
            throws Exception {
        log.info("Interface GET /a/u/school/enrollment/list,Method getEnrollmentListBySchool() parameters, page: "
                + page + ", size: " + size + ", startTime: " + startTime + ", endTime: " + endTime + ", type: " + type
                + ", status: " + status + ", schoolId: " + schoolId);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int startPosition = ParamsUtil.calculateStartPosition(page, actualSize);
        log.info("page : " + startPosition + ", size: " + actualSize);

        if (DataUtils.isNullOrEmpty(schoolId)) {
            log.info("Parameter schoolId can not be null!");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            Map<String, Object> queryEnrollmentDetailMap = DynamicUtil.getEnrollmentListBySchool(startTime, endTime,
                    type, status, schoolId, false);
            List<Long> enrollmentDetailIdList = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                    queryEnrollmentDetailMap, startPosition, actualSize);
            log.info("get enrollmentDetail id list, list is: " + enrollmentDetailIdList);
            List<EnrollmentDetail> enrollmentDetailList = null;
            Long total = 0L;
            if (CollectionUtils.isNotEmpty(enrollmentDetailIdList)) {
                enrollmentDetailList = enrollmentDetailService.getObjectsByIds(enrollmentDetailIdList);
                log.info("get enrollmentDetail data list, list size is: " + enrollmentDetailList.size());

                Map<String, Object> countEnrollmentDetailMap = DynamicUtil.getEnrollmentListBySchool(startTime, endTime,
                        type, status, schoolId, true);
                total = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                        countEnrollmentDetailMap, 0, Integer.MAX_VALUE).get(0);
                log.info("get enrollmentDetail data total, total is: " + total);
            }

            model.addAttribute("enrollmentDetailListSchool", enrollmentDetailList);
            model.addAttribute("code", 0);
            model.addAttribute("size", actualSize);
            model.addAttribute("total", total);
            return "yi-nucleus-service/enrollmentDetail/json/enrollmentListJsonSchool";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getEnrollmentList() error, page is: " + page + ", size is: " + size);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 运营中心端查询报名学员接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/center/enrollment/list", method = RequestMethod.GET)
    public String getEnrollmentListByCenter(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                            Integer page, Integer size, Long startTime, Long endTime, Integer type, Integer status, Long cityId)
            throws Exception {
        log.info("Interface GET /a/u/center/enrollment/list,Method getEnrollmentListByCenter() parameters, page: "
                + page + ", size: " + size + ", startTime: " + startTime + ", endTime: " + endTime + ", type: " + type
                + ", status: " + status + ", cityId: " + cityId);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int startPosition = ParamsUtil.calculateStartPosition(page, actualSize);
        log.info("page : " + startPosition + ", size: " + actualSize);

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("Parameter cityId can not be null!");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            Map<String, Object> queryEnrollmentDetailMap = DynamicUtil.getEnrollmentListByCenter(startTime, endTime,
                    type, status, cityId, false);
            List<Long> enrollmentDetailIdList = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                    queryEnrollmentDetailMap, startPosition, actualSize);
            log.info("get enrollmentDetail id list, list is: " + enrollmentDetailIdList);
            List<EnrollmentDetail> enrollmentDetailList = null;
            Long total = 0L;
            if (CollectionUtils.isNotEmpty(enrollmentDetailIdList)) {
                enrollmentDetailList = enrollmentDetailService.getObjectsByIds(enrollmentDetailIdList);
                log.info("get enrollmentDetail data list, list size is: " + enrollmentDetailList.size());

                Map<String, Object> countEnrollmentDetailMap = DynamicUtil.getEnrollmentListByCenter(startTime, endTime,
                        type, status, cityId, true);
                total = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                        countEnrollmentDetailMap, 0, Integer.MAX_VALUE).get(0);
                log.info("get enrollmentDetail data total, total is: " + total);
            }

            model.addAttribute("enrollmentDetailListCenter", enrollmentDetailList);
            model.addAttribute("code", 0);
            model.addAttribute("size", actualSize);
            model.addAttribute("total", total);
            return "yi-nucleus-service/enrollmentDetail/json/enrollmentListJsonCenter";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getEnrollmentList() error, page is: " + page + ", size is: " + size);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 更新报名学员接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/enrollment/{id}", method = RequestMethod.PUT)
    public String updateEnrollmentDetail(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                         @PathVariable Long id) throws Exception {
        log.info("Interface PUT /a/u/enrollment/{id},Method updateEnrollmentDetail() parameter id: " + id);

        try {
            EnrollmentDetail enrollmentDetail = enrollmentDetailService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(enrollmentDetail)) {
                log.info("Can not find enrollmentDetail Object by id, id is : " + id);
                model.addAttribute("code", -1001);
                return "/common/failure";
            }
            log.info("get enrollmentDetail object by id, id is : " + enrollmentDetail.getId());

            Long managerId = ManagerUtil.getManagerId(cookieUtil, request);
            enrollmentDetail.setUpdateBy(managerId);
            enrollmentDetail.setStatus(EnrollmentDetail.HAS_RECEIVED);
            enrollmentDetail.setReceiveTime(System.currentTimeMillis());
            enrollmentDetailService.update(enrollmentDetail);
            log.info("update enrollmentDetail status success, status is: " + enrollmentDetail.getStatus());

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method updateEnrollmentDetail() failed, id is : " + id);
            model.addAttribute("code", -1);
            return "/common/failed";
        }
    }

    /**
     * 查询驾校财务接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/finance/school", method = RequestMethod.GET)
    public String getFinanceOfSchool(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                     Long schoolId) throws Exception {
        log.info("Interface GET /a/u/finance/school, Method getFinanceOfSchool() parameter: schoolId: " + schoolId);

        if (DataUtils.isNullOrEmpty(schoolId)) {
            log.info("Parameter schoolId can not be null, schoolId is: " + schoolId);
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            List<EnrollmentDetail> prepareReceiveList = enrollmentDetailService
                    .getEnrollmentDetailListBySchoolIdAndStatus(schoolId, EnrollmentDetail.NOT_RECEIVED, 0,
                            Integer.MAX_VALUE);
            log.info("get prepareReceiveEnrollmentDetailList data list, list size is:" + prepareReceiveList.size());

            List<EnrollmentDetail> receivedList = enrollmentDetailService.getEnrollmentDetailListBySchoolIdAndStatus(
                    schoolId, EnrollmentDetail.HAS_RECEIVED, 0, Integer.MAX_VALUE);
            log.info("get receivedEnrollmentDetailList data list, list size is:" + receivedList.size());

            SchoolFinance schoolFinance = SchoolFinance.getSchoolFinanceRecord(prepareReceiveList, receivedList);

            model.addAttribute("code", 0);
            model.addAttribute("schoolFinance", schoolFinance);
            return "/yi-nucleus-service/enrollmentDetail/json/financeOfSchool";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getFinanceOfSchool() failed, schoolId is : " + schoolId);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 查询收支/结算/订单明细接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/finance/record/detail/list", method = RequestMethod.GET)
    public String getFinanceRecordDetailList(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                             Integer page, Integer size, Long id, Integer type) throws Exception {
        log.info("Interface /a/u/finance/record/detail/list, Method getFinanceRecordDetailList() parameters: " + "id: "
                + id + ", type: " + type);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int startPosition = ParamsUtil.calculateStartPosition(page, actualSize);

        int validateResult = ParamsUtil.checkParametersOfGetFinanceRecordDetailList(id, type);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            Map<String, Object> queryFinanceRecordDetailMap = DynamicUtil.getFinanceRecordDetailListParams(id, type,
                    false);
            List<Long> enrollmentDetailIdList = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                    queryFinanceRecordDetailMap, startPosition, actualSize);
            log.info("get enrollment id list, list is : " + enrollmentDetailIdList);

            List<EnrollmentDetail> enrollmentDetailList = null;
            Long total = 0L;
            if (CollectionUtils.isNotEmpty(enrollmentDetailIdList)) {
                enrollmentDetailList = enrollmentDetailService.getObjectsByIds(enrollmentDetailIdList);
                log.info("get enrollmentDetailList data, list size is : " + enrollmentDetailList.size());

                Map<String, Object> countFinanceRecordDetailMap = DynamicUtil.getFinanceRecordDetailListParams(id, type,
                        true);
                total = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                        countFinanceRecordDetailMap, 0, Integer.MAX_VALUE).get(0);
                log.info("count enrollmentDetailList total is:" + total);
            }

            String[] viewKeywords = {"enrollmentDetailListRecord", "enrollmentDetailListRecord",
                    "enrollmentDetailListSettlement", "enrollmentDetailListWithdraw"};
            model.addAttribute(viewKeywords[type], enrollmentDetailList);
            model.addAttribute("code", 0);
            model.addAttribute("size", actualSize);
            model.addAttribute("total", total);
            return "/yi-nucleus-service/enrollmentDetail/json/financeRecordDetailList";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getFinanceRecordDetailList() failed, page is:" + page + ", size: " + size);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 查询驾校可提现/提现中报名学员列表接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/school/enrollment", method = RequestMethod.GET)
    public String getSchoolEnrollmentIdList(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                            Long schoolId, Integer withdrawStatus) throws Exception {
        log.info("Interface /a/u/school/enrollment, Method getSchoolEnrollmentIdList() parameters: " + "schoolId: "
                + schoolId + ", withdrawStatus: " + withdrawStatus);

        int validateResult = ParamsUtil.checkParametersOfGetSchoolEnrollmentIdList(schoolId, withdrawStatus);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            List<Long> idList = enrollmentDetailService.getEnrollmentDetailIdsBySchoolIdAndStatusAndWithdrawStatus(
                    schoolId, EnrollmentDetail.HAS_RECEIVED, withdrawStatus, 0, Integer.MAX_VALUE);
            log.info("get idList by schoolId and withdrawStatus, idList is : " + idList);

            if (withdrawStatus.equals(EnrollmentDetail.NOT_WITHDRAW) && CollectionUtils.isNotEmpty(idList)) {
                List<EnrollmentDetail> enrollmentDetailList = enrollmentDetailService.getObjectsByIds(idList);
                log.info("get enrollmentDetailList by id list, list size is : " + enrollmentDetailList.size());

                for (int i = 0; i < enrollmentDetailList.size(); i++) {
                    EnrollmentDetail enrollmentDetail = enrollmentDetailList.get(i);
                    if (enrollmentDetail.getReceiveTime() > TimeUtil.getTimeStampOfToday()) {
                        idList.remove(enrollmentDetail.getId());
                    }
                }
            }

            model.addAttribute("code", 0);
            model.addAttribute("idList", idList);
            return "/yi-nucleus-service/enrollmentDetail/json/schoolEnrollment";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getSchoolEnrollmentIdList() failed, schoolId is:" + schoolId + ", withdrawStatus: "
                    + withdrawStatus);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 运营中心财务统计接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/finance/center/total", method = RequestMethod.GET)
    public String getFinanceOfCenterTotal(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                          Integer type, Integer status, Long startTime, Long endTime, Long cityId) throws Exception {

        log.info("Interface GET /a/u/finance/center/total, Method getFinanceOfCenterTotal() parameters:" + " type: "
                + type + ", status: " + status + ", startTime: " + startTime + ", endTime: " + endTime + ", cityId: "
                + cityId);

        int validateResult = ParamsUtil.checkParametersOfGetFinanceOfCenter(type, status, cityId);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            Map<String, Object> queryEnrollmentDetailMap = DynamicUtil.getEnrollmentDetailParams(startTime, endTime,
                    type, status, cityId, null);
            List<Long> enrollmentDetailIdList = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                    queryEnrollmentDetailMap, 0, Integer.MAX_VALUE);
            log.info("get enrollment detail id list, list is : " + enrollmentDetailIdList);

            List<EnrollmentDetail> enrollmentDetailList = null;
            if (CollectionUtils.isNotEmpty(enrollmentDetailIdList)) {
                enrollmentDetailList = enrollmentDetailService.getObjectsByIds(enrollmentDetailIdList);
                log.info("get enrollment detail data list, list size is : " + enrollmentDetailList.size());
            }

            CenterFinance centerFinance = CenterFinance.countAmountOfCenter(enrollmentDetailList);
            log.info("get finance data, finance is : " + centerFinance);

            model.addAttribute("finance", centerFinance);
            model.addAttribute("code", 0);
            return "yi-nucleus-service/enrollmentDetail/json/financeOfCenterTotal";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getFinanceOfCenterTotal() parameters failed, startTime is :" + startTime + ", endTime:"
                    + endTime + ", status:" + status + ", type:" + type);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 运营中心财务统计驾校明细接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/finance/center/school", method = RequestMethod.GET)
    public String getFinanceOfCenterBySchool(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                             Integer page, Integer size, Integer type, Integer status, Long startTime, Long endTime, Long cityId)
            throws Exception {
        log.info("Interface /a/u/finance/center/school,Method getFinanceOfCenterBySchool() parameters: " + "type: "
                + type + ", status: " + status + ", startTime: " + startTime + ", endTime: " + endTime + ", cityId: "
                + cityId);
        int actualSize = ParamsUtil.calculateActualSize(size);
        int startPosition = ParamsUtil.calculateStartPosition(page, actualSize);
        log.info("page: " + startPosition + ", size: " + actualSize);

        int validateResult = ParamsUtil.checkParametersOfGetFinanceOfCenter(type, status, cityId);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            List<Long> schoolIdList = schoolService.getSchoolIds(0, Integer.MAX_VALUE);
            log.info("get school id list, list is : " + schoolIdList);
            if (CollectionUtils.isEmpty(schoolIdList)) {
                model.addAttribute("code", 0);
                model.addAttribute("size", 0);
                model.addAttribute("total", 0);
                return "yi-nucleus-service/enrollmentDetail/json/financeOfCenterSchool";
            }

            Map<String, Object> queryEnrollmentDetailMap = DynamicUtil.getEnrollmentDetailParams(startTime, endTime,
                    type, status, cityId, schoolIdList);
            List<Long> enrollmentDetailIdList = enrollmentDetailService.getIdsByDynamicCondition(EnrollmentDetail.class,
                    queryEnrollmentDetailMap, 0, Integer.MAX_VALUE);
            log.info("get enrollment detail id list, list is : " + enrollmentDetailIdList);

            if (CollectionUtils.isNotEmpty(enrollmentDetailIdList)) {
                List<EnrollmentDetail> enrollmentDetailList = enrollmentDetailService
                        .getObjectsByIds(enrollmentDetailIdList);
                log.info("get enrollment detail data list, list size is : " + enrollmentDetailList.size());

                Map<Long, CenterFinance> financeMap = CenterFinance.generateSchoolFinanceMap(enrollmentDetailList);
                log.info("get school finance map, map size is : " + financeMap.size());

                List<CenterFinance> centerFinanceList = CenterFinance.countAmountOfSchools(enrollmentDetailList,
                        financeMap);
                log.info("count amount of school list total, total is : " + centerFinanceList.size());

                List<CenterFinance> resultCenterFinanceList = ListUtil.getPartElementsFromList(centerFinanceList,
                        startPosition, actualSize);

                model.addAttribute("total", centerFinanceList.size());
                model.addAttribute("financeList", resultCenterFinanceList);
            }
            model.addAttribute("code", 0);
            model.addAttribute("size", size);
            return "yi-nucleus-service/enrollmentDetail/json/financeOfCenterSchool";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method getFinanceOfCenterBySchool() parameters failed, startTime is :" + startTime + ", endTime:"
                    + endTime + ", status:" + status + ", type:" + type + ", page:" + page + ", size: " + size);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }
}
