package com.ptteng.controller;

import com.ptteng.common.yi.service.ManagerService;
import com.ptteng.util.DynamicUtil;
import com.ptteng.yi.pub.model.Content;
import com.ptteng.yi.pub.service.CitiesService;
import com.ptteng.yi.pub.service.ContentService;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * Content  crud
 * 
 * @author wangmeng
 * @Date 2017-12-20 13:43
 * 
 */
@Controller
public class ContentController {
    private static final Log log = LogFactory.getLog(ContentController.class);

    @Autowired
    private ContentService contentService;

    @Autowired
    private CitiesService citiesService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 查看内容详情
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/content/{id}", method = RequestMethod.GET)
    public String getContentJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("GET request url : /a/u/content/{id}, get path variable id : " + id);
        try {
            Content content = contentService.getObjectById(id);
            log.info("get content data is " + content);

            if (DataUtils.isNullOrEmpty(content)) {
                log.info("no this content");
                model.addAttribute("code", -10018);
                return "/common/failure";
            }

            model.addAttribute("code", 0);

            model.addAttribute("content", content);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get content error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/yi-pub-service/content/json/contentDetailJson";
    }

    /**
     * 删除内容
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/content/{id}", method = RequestMethod.DELETE)
    public String deleteContentJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("DELETE request url : /a/u/content/{id}, get path variable id : " + id);
        try {
            Content content = contentService.getObjectById(id);

            if (content.getType() == Content.TYPE_EXPERIENCE) {
                log.info("this content type is not allowed deleting");
                model.addAttribute("code", -10023);
                return "/common/failure";
            }

            contentService.delete(id);
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("delete content error,id is  " + id);
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 上下线内容
     * @param request
     * @param response
     * @param model
     * @param id
     * @param status
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/content/{id}/status", method = RequestMethod.PUT)
    public String updateCardTypeStatusJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Integer status) throws Exception {

        log.info("PUT request url: /a/u/content/{id}/status, get path variable : id= " + id + ", parameter : status= "
                + status);

        if (DataUtils.isNullOrEmpty(status)) {
            model.addAttribute("code", -10001);
            log.info("parameter status is null or empty");
            return "/common/failure";
        }

        if (status > Content.STATUS_ONLINE || status < Content.STATUS_OFFLINE) {
            model.addAttribute("code", -10002);
            log.info("parameter status is illegal");
            return "/common/failure";
        }

        try {
            Content content = contentService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(content)) {
                log.info("no this content");
                model.addAttribute("code", -10000);
                return "/common/failure";
            }

            if (content.getType() == Content.TYPE_EXPERIENCE) {
                log.info("this content type is not allowed changing status");
                model.addAttribute("code", -10026);
                return "/common/failure";
            }

            content.setStatus(status);

            if (status == Content.STATUS_ONLINE) {
                content.setPublishAt(System.currentTimeMillis());
            } else {
                content.setPublishAt(Content.LONG_NULL_VALUE);
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            content.setUpdateBy(Long.parseLong(managerId));

            contentService.update(content);
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update content status error,id is  " + id);
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 编辑banner
     * @param request
     * @param response
     * @param model
     * @param id
     * @param content
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/banner/{id}", method = RequestMethod.PUT)
    public String updateBannerJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Content content) throws Exception {

        log.info("PUT request url : /a/u/banner/{id}, get path variable id : " + id + ", parameter content : "
                + content);

        if (DataUtils.isNullOrEmpty(content)) {
            log.info("no this content");
            model.addAttribute("code", -10000);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getStatus())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10001);
            return "/common/failure";
        }

        if (content.getStatus() < Content.STATUS_OFFLINE || content.getStatus() > Content.STATUS_ONLINE) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10002);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getApp())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (content.getApp() < Content.APP_COACH || content.getApp() > Content.APP_STUDENT) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getBannerType())) {
            log.info("parameter bannerType is null or empty");
            model.addAttribute("code", -10011);
            return "/common/failure";
        }

        if (content.getBannerType() < Content.INTEGER_NULL_VALUE
                || content.getBannerType() > Content.BANNER_TYPE_LINK) {
            log.info("parameter bannerType is illegal");
            model.addAttribute("code", -10012);
            return "/common/failure";
        }

        if (content.getApp() == Content.APP_COACH) {
            if (content.getBannerType() == Content.BANNER_TYPE_CLASS) {
                log.info("there is not class type in banner of coach");
                model.addAttribute("code", -10020);
                return "/common/failure";
            }
        }

        if (content.getTitle() == null) {
            log.info("parameter title is null or empty");
            model.addAttribute("code", -10013);
            return "/common/failure";
        }

        if (content.getImg() == null) {
            log.info("parameter img is null or empty");
            model.addAttribute("code", -10014);
            return "/common/failure";
        }

        String regex = "^([hH][tT]{2}[pP]://|[hH][tT]{2}[pP][sS]://)(([A-Za-z0-9-~]+).)+([A-Za-z0-9-~\\\\/])+$";
        Pattern pattern = Pattern.compile(regex);

        if (StringUtils.isEmpty(content.getImg()) || !pattern.matcher(content.getImg()).matches()) {
            log.info("parameter img is illegal");
            model.addAttribute("code", -10024);
            return "/common/failure";
        }

        if (content.getContent() == null) {
            log.info("parameter content is null or empty");
            model.addAttribute("code", -10015);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getId())) {
            log.info("parameter id is null or empty");
            model.addAttribute("code", -10016);
            return "/common/failure";
        }

        if (!content.getId().equals(id)) {
            log.info("parameter id is not matching the path variable");
            model.addAttribute("code", -10017);
            return "/common/failure";
        }

        try {
            Content content1 = contentService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(content1)) {
                log.info("there is not this content in DB");
                model.addAttribute("code", -10018);
                return "/common/failure";
            }

            if (content1.getType() != Content.TYPE_BANNER) {
                log.info("content type is not banner");
                model.addAttribute("code", -10019);
                return "/common/failure";
            }

            content1.setStatus(content.getStatus());
            content1.setApp(content.getApp());
            content1.setBannerType(content.getBannerType());
            content1.setTitle(content.getTitle());
            content1.setImg(content.getImg());
            content1.setContent(content.getContent());

            if (content.getStatus() == Content.STATUS_ONLINE) {
                content1.setPublishAt(System.currentTimeMillis());
            } else {
                content1.setPublishAt(Content.LONG_NULL_VALUE);
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            content1.setUpdateBy(Long.parseLong(managerId));

            contentService.update(content1);

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update content error,id is  " + content.getId());
            model.addAttribute("code", -100000);

        }

        return "/common/success";
    }

    /**
     * 新增banner
     * @param request
     * @param response
     * @param model
     * @param content
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/banner", method = RequestMethod.POST)
    public String addBannerJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Content content) throws Exception {

        log.info("POST request url : /a/u/banner, get parameter content : " + content);

        if (DataUtils.isNullOrEmpty(content)) {
            log.info("no this content");
            model.addAttribute("code", -10000);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getStatus())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10001);
            return "/common/failure";
        }

        if (content.getStatus() < Content.STATUS_OFFLINE || content.getStatus() > Content.STATUS_ONLINE) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10002);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getApp())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (content.getApp() < Content.APP_COACH || content.getApp() > Content.APP_STUDENT) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getBannerType())) {
            log.info("parameter bannerType is null or empty");
            model.addAttribute("code", -10011);
            return "/common/failure";
        }

        if (content.getBannerType() < Content.INTEGER_NULL_VALUE
                || content.getBannerType() > Content.BANNER_TYPE_LINK) {
            log.info("parameter bannerType is illegal");
            model.addAttribute("code", -10012);
            return "/common/failure";
        }

        if (content.getApp() == Content.APP_COACH) {
            if (content.getBannerType() == Content.BANNER_TYPE_CLASS) {
                log.info("there is not class type in banner of coach");
                model.addAttribute("code", -10020);
                return "/common/failure";
            }
        }

        if (content.getTitle() == null) {
            log.info("parameter title is null or empty");
            model.addAttribute("code", -10013);
            return "/common/failure";
        }

        if (content.getImg() == null) {
            log.info("parameter img is null or empty");
            model.addAttribute("code", -10014);
            return "/common/failure";
        }

        String regex = "^([hH][tT]{2}[pP]://|[hH][tT]{2}[pP][sS]://)(([A-Za-z0-9-~]+).)+([A-Za-z0-9-~\\\\/])+$";
        Pattern pattern = Pattern.compile(regex);

        if (StringUtils.isEmpty(content.getImg()) || !pattern.matcher(content.getImg()).matches()) {
            log.info("parameter img is illegal");
            model.addAttribute("code", -10024);
            return "/common/failure";
        }

        if (content.getContent() == null) {
            log.info("parameter content is null or empty");
            model.addAttribute("code", -10015);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getCityId())) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (content.getCityId() < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(content.getCityId(), 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            content.setId(null);
            content.setType(Content.TYPE_BANNER);
            content.setCountdown(Content.INTEGER_NULL_VALUE);
            content.setLevel(Content.INITIAL_VALUE);
            content.setUrl(Content.STRING_NULL_VALUE);
            content.setSummary(Content.STRING_NULL_VALUE);
            content.setReading(Content.INITIAL_VALUE);
            content.setCountAt(Content.LONG_NULL_VALUE);

            if (content.getStatus() == Content.STATUS_ONLINE) {
                content.setPublishAt(System.currentTimeMillis());
            } else {
                content.setPublishAt(Content.LONG_NULL_VALUE);
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            content.setUpdateBy(Long.parseLong(managerId));
            content.setCreateBy(Long.parseLong(managerId));

            contentService.insert(content);

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add banner error ");
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 获取banner列表
     * @param request
     * @param response
     * @param model
     * @param title 标题名称
     * @param status 状态
     * @param app app类型
     * @param updateAtStart 更新时间起
     * @param updateAtEnd 更新时间止
     * @param publishAtStart 发布时间起
     * @param publishAtEnd 发布时间止
     * @param cityId 城市id
     * @param page
     * @param size
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/banner/list", method = RequestMethod.GET)
    public String getMultiBannerJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String title, Integer status, Integer app, Long updateAtStart, Long updateAtEnd, Long publishAtStart,
            Long publishAtEnd, Long cityId, Integer page, Integer size) throws Exception {

        log.info("GET request url: /a/u/banner/list, get parameter title: " + title + ", status: " + status + ", app: "
                + app + ", updateAtStart: " + updateAtStart + ", updateAtEnd: " + updateAtEnd + ", publishAtStart: "
                + publishAtStart + ", publishAtEnd: " + publishAtEnd + ", cityId: " + cityId + ", page: " + page
                + ", size: " + size);

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        if (page == null) {
            page = 1;
        }
        if (size == null || size < 0) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        log.info("pageList : page= " + start + " , size=" + size);

        try {
            Map<String, Object> map = DynamicUtil.getContentListParams(Content.TYPE_BANNER, title, status, app,
                    updateAtStart, updateAtEnd, publishAtStart, publishAtEnd, cityId, false);
            log.info("query banner map : " + map);

            List<Long> bannerIds = contentService.getIdsByDynamicCondition(Content.class, map, start, size);
            log.info("get banner id list : " + bannerIds);
            Integer total = contentService.getIdsByDynamicCondition(Content.class, map, 0, Integer.MAX_VALUE).size();

            if (CollectionUtils.isEmpty(bannerIds)) {
                model.addAttribute("contentList", new ArrayList());
            } else {
                List<Content> bannerList = contentService.getObjectsByIds(bannerIds);
                log.info("get bannerList data size is " + bannerList.size());

                model.addAttribute("contentList", bannerList);
            }

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);

        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("get banner list error, start is  " + start + " , size " + size);
            model.addAttribute("code", -100000);
        }

        return "/yi-pub-service/content/json/contentListJson";
    }

    /**
     * banner拖动排序
     * @param request
     * @param response
     * @param model
     * @param app app类型
     * @param cityId 城市id
     * @param ids banner id数组
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/banner/sort", method = RequestMethod.PUT)
    public String sortBanners(HttpServletRequest request, HttpServletResponse response, ModelMap model, Integer app,
            Long cityId, @RequestParam Long[] ids) throws Exception {

        log.info("PUT request url: /a/u/banner/sort, get parameter app : " + app + ", array ids size : " + ids.length);

        if (DataUtils.isNullOrEmpty(ids)) {
            log.info("parameter ids is null or empty");
            model.addAttribute("code", -10003);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(app)) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (app < Content.APP_COACH || app > Content.APP_STUDENT) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        List<Long> idList = Arrays.asList(ids);

        try {
            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);

            List<Long> cityIds = citiesService.getCitiesIdsByCityid(cityId, 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            Map<String, Object> map = DynamicUtil.getContentListParams(Content.TYPE_BANNER, null, null, app, null, null,
                    null, null, cityId, false);
            log.info("query banner map : " + map);

            List<Long> onlineIds = contentService.getIdsByDynamicCondition(Content.class, map, 0, Integer.MAX_VALUE);
            log.info("get banner id list : " + onlineIds);

            if (CollectionUtils.isEmpty(onlineIds)) {
                log.info("there are no online banners");
                model.addAttribute("code", -10006);
                return "/common/failure";
            } else if (!onlineIds.containsAll(idList)) {
                log.info("the available banner ids didn't contain all ids in parameter id list");
                model.addAttribute("code", -10007);
                return "/common/failure";
            } else {
                List<Content> contentList = contentService.getObjectsByIds(idList);
                int index = 0;
                for (Content content : contentList) {
                    content.setLevel(index);
                    content.setUpdateBy(Long.parseLong(managerId));
                    index = index + 10;
                }
                contentService.updateList(contentList);
                model.addAttribute("code", 0);
            }
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("sort banner error,ids is  " + idList);
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }

    /**
     * 编辑资讯
     * @param request
     * @param response
     * @param model
     * @param id
     * @param content
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/news/{id}", method = RequestMethod.PUT)
    public String updateNewsJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Content content) throws Exception {

        log.info("PUT request url : /a/u/news/{id}, get path variable id : " + id + ", parameter content : " + content);

        if (DataUtils.isNullOrEmpty(content)) {
            log.info("no this news");
            model.addAttribute("code", -10000);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getApp())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (content.getApp() < Content.APP_COACH || content.getApp() > Content.APP_GENERAL) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (content.getTitle() == null) {
            log.info("parameter title is null or empty");
            model.addAttribute("code", -10013);
            return "/common/failure";
        }

        if (content.getImg() == null) {
            log.info("parameter img is null or empty");
            model.addAttribute("code", -10014);
            return "/common/failure";
        }

        String regex = "^([hH][tT]{2}[pP]://|[hH][tT]{2}[pP][sS]://)(([A-Za-z0-9-~]+).)+([A-Za-z0-9-~\\\\/])+$";
        Pattern pattern = Pattern.compile(regex);

        if (StringUtils.isEmpty(content.getImg()) || !pattern.matcher(content.getImg()).matches()) {
            log.info("parameter img is illegal");
            model.addAttribute("code", -10024);
            return "/common/failure";
        }

        if (content.getContent() == null) {
            log.info("parameter content is null or empty");
            model.addAttribute("code", -10015);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getId())) {
            log.info("parameter id is null or empty");
            model.addAttribute("code", -10016);
            return "/common/failure";
        }

        if (!content.getId().equals(id)) {
            log.info("parameter id is not matching the path variable");
            model.addAttribute("code", -10017);
            return "/common/failure";
        }

        try {
            Content content1 = contentService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(content1)) {
                log.info("there is not this content in DB");
                model.addAttribute("code", -10018);
                return "/common/failure";
            }

            if (content1.getType() != Content.TYPE_NEWS) {
                log.info("content type is not news");
                model.addAttribute("code", -10019);
                return "/common/failure";
            }

            content1.setApp(content.getApp());
            content1.setTitle(content.getTitle());
            content1.setImg(content.getImg());
            content1.setContent(content.getContent());

            if (content1.getStatus() == Content.STATUS_ONLINE) {
                content1.setPublishAt(System.currentTimeMillis());
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            content1.setUpdateBy(Long.parseLong(managerId));

            contentService.update(content1);

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update content error, id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 新增资讯
     * @param request
     * @param response
     * @param model
     * @param content
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/news", method = RequestMethod.POST)
    public String addNewsJson(HttpServletRequest request, HttpServletResponse response, ModelMap model, Content content)
            throws Exception {

        log.info("POST request url : /a/u/news/{id}, get parameter content : " + content);

        if (DataUtils.isNullOrEmpty(content)) {
            log.info("no this content");
            model.addAttribute("code", -10000);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getApp())) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (content.getApp() < Content.STATUS_OFFLINE || content.getApp() > Content.APP_GENERAL) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (content.getTitle() == null) {
            log.info("parameter title is null or empty");
            model.addAttribute("code", -10013);
            return "/common/failure";
        }

        if (content.getImg() == null) {
            log.info("parameter img is null or empty");
            model.addAttribute("code", -10014);
            return "/common/failure";
        }

        String regex = "^([hH][tT]{2}[pP]://|[hH][tT]{2}[pP][sS]://)(([A-Za-z0-9-~]+).)+([A-Za-z0-9-~\\\\/])+$";
        Pattern pattern = Pattern.compile(regex);

        if (StringUtils.isEmpty(content.getImg()) || !pattern.matcher(content.getImg()).matches()) {
            log.info("parameter img is illegal");
            model.addAttribute("code", -10024);
            return "/common/failure";
        }

        if (content.getContent() == null) {
            log.info("parameter content is null or empty");
            model.addAttribute("code", -10015);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getCityId())) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (content.getCityId() < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(content.getCityId(), 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            content.setId(null);
            content.setType(Content.TYPE_NEWS);
            content.setBannerType(Content.INTEGER_NULL_VALUE);
            content.setCountdown(Content.INTEGER_NULL_VALUE);
            content.setStatus(Content.INITIAL_VALUE);
            content.setLevel(Content.INITIAL_VALUE);
            content.setUrl(Content.STRING_NULL_VALUE);
            content.setSummary(Content.STRING_NULL_VALUE);
            content.setReading(Content.INITIAL_VALUE);
            content.setCountAt(Content.LONG_NULL_VALUE);
            content.setPublishAt(Content.LONG_NULL_VALUE);

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            content.setUpdateBy(Long.parseLong(managerId));
            content.setCreateBy(Long.parseLong(managerId));

            contentService.insert(content);

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add banner error");
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 获取资讯列表
     * @param request
     * @param response
     * @param model
     * @param title 标题
     * @param status 状态
     * @param app app类型
     * @param cityId 城市id
     * @param page
     * @param size
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/news/list", method = RequestMethod.GET)
    public String getMultiNewsJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String title, Integer status, Integer app, Long cityId, Integer page, Integer size) throws Exception {

        log.info("GET request url: /a/u/new/list, get parameter title: " + title + ", status: " + status + ", app: "
                + app + ", cityId: " + cityId + ", page: " + page + ", size: " + size);

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        if (page == null) {
            page = 1;
        }
        if (size == null || size < 0) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        log.info("pageList : page= " + start + " , size=" + size);

        try {
            Map<String, Object> map = DynamicUtil.getContentListParams(Content.TYPE_NEWS, title, status, app, null,
                    null, null, null, cityId, false);
            log.info("query news map : " + map);

            List<Long> newsIds = contentService.getIdsByDynamicCondition(Content.class, map, start, size);
            log.info("get news id list : " + newsIds);
            Integer total = contentService.getIdsByDynamicCondition(Content.class, map, 0, Integer.MAX_VALUE).size();

            if (CollectionUtils.isEmpty(newsIds)) {
                model.addAttribute("contentList", new ArrayList());
            } else {
                List<Content> newsList = contentService.getObjectsByIds(newsIds);
                log.info("get newsList data size is " + newsList.size());

                model.addAttribute("contentList", newsList);
            }

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);

        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("get news list error, start is  " + start + " , size " + size);
            model.addAttribute("code", -100000);
        }

        return "/yi-pub-service/content/json/contentListJson";
    }

    /**
     * 资讯拖动排序
     * @param request
     * @param response
     * @param model
     * @param app app类型
     * @param cityId 城市id
     * @param ids 资讯id数组
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/news/sort", method = RequestMethod.PUT)
    public String sortNews(HttpServletRequest request, HttpServletResponse response, ModelMap model, Integer app,
            Long cityId, @RequestParam Long[] ids) throws Exception {

        log.info("PUT request url: /a/u/news/sort, get parameter app : " + app + ", array ids size : " + ids.length);

        if (DataUtils.isNullOrEmpty(ids)) {
            log.info("parameter ids is null or empty");
            model.addAttribute("code", -10003);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(app)) {
            log.info("parameter app is null or empty");
            model.addAttribute("code", -10004);
            return "/common/failure";
        }

        if (app < Content.APP_COACH || app > Content.APP_STUDENT) {
            log.info("parameter app is illegal");
            model.addAttribute("code", -10005);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        List<Long> idList = Arrays.asList(ids);

        try {
            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);

            List<Long> cityIds = citiesService.getCitiesIdsByCityid(cityId, 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            Map<String, Object> map = DynamicUtil.getContentListParams(Content.TYPE_NEWS, null, null, app, null, null,
                    null, null, cityId, false);
            log.info("query news map : " + map);

            List<Long> onlineIds = contentService.getIdsByDynamicCondition(Content.class, map, 0, Integer.MAX_VALUE);
            log.info("get news id list : " + onlineIds);

            if (CollectionUtils.isEmpty(onlineIds)) {
                log.info("there are no online news");
                model.addAttribute("code", -10006);
                return "/common/failure";
            } else if (!onlineIds.containsAll(idList)) {
                log.info("the available news ids didn't contain all ids in parameter id list");
                model.addAttribute("code", -10007);
                return "/common/failure";
            } else {
                List<Content> contentList = contentService.getObjectsByIds(idList);
                int index = 0;
                for (Content content : contentList) {
                    content.setLevel(index);
                    content.setUpdateBy(Long.parseLong(managerId));
                    index = index + 10;
                }
                contentService.updateList(contentList);
                model.addAttribute("code", 0);
            }
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("sort news error,ids is  " + idList);
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }

    /**
     * 查看免费试学
     * @param request
     * @param response
     * @param model
     * @param cityId 城市id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/experience", method = RequestMethod.GET)
    public String getExperienceJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Long cityId) throws Exception {

        log.info("GET request url : /a/u/experience, get parameter cityId: " + cityId);

        if (DataUtils.isNullOrEmpty(cityId)) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (cityId < 1L) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(cityId, 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            List<Long> ids = contentService.getContentIdsByTypeAndStatusAndAppAndCityId(Content.TYPE_EXPERIENCE,
                    Content.STATUS_ONLINE, Content.APP_STUDENT, cityId, 0, Integer.MAX_VALUE);

            if (!CollectionUtils.isEmpty(ids)) {
                Content content = contentService.getObjectById(ids.get(0));
                log.info("get experience content data is " + content);
                model.addAttribute("content", content);
            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get experience content error");
            model.addAttribute("code", -100000);

        }

        return "yi-pub-service/content/json/experienceJson";
    }

    /**
     * 编辑免费试学
     * @param request
     * @param response
     * @param model
     * @param content
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/experience", method = RequestMethod.PUT)
    public String updateExperienceJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Content content) throws Exception {

        log.info("PUT request url : /a/u/experience, get parameter content: " + content);

        if (DataUtils.isNullOrEmpty(content)) {
            log.info("no this experience content");
            model.addAttribute("code", -10000);
            return "/common/failure";
        }

        if (content.getImg() == null) {
            log.info("parameter img is null or empty");
            model.addAttribute("code", -10014);
            return "/common/failure";
        }

        String regex = "^([hH][tT]{2}[pP]://|[hH][tT]{2}[pP][sS]://)(([A-Za-z0-9-~]+).)+([A-Za-z0-9-~\\\\/])+$";
        Pattern pattern = Pattern.compile(regex);

        if (StringUtils.isEmpty(content.getImg()) || !pattern.matcher(content.getImg()).matches()) {
            log.info("parameter img is illegal");
            model.addAttribute("code", -10024);
            return "/common/failure";
        }

        if (content.getContent() == null) {
            log.info("parameter content is null or empty");
            model.addAttribute("code", -10015);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getCountdown())) {
            log.info("parameter countdown is null or empty");
            model.addAttribute("code", -10021);
            return "/common/failure";
        }

        if (content.getCountdown() < Content.INITIAL_VALUE) {
            log.info("parameter countdown is illegal");
            model.addAttribute("code", -10022);
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(content.getCityId())) {
            log.info("parameter cityId is null or empty");
            model.addAttribute("code", -10027);
            return "/common/failure";
        }

        if (content.getCityId() < Content.LONG_NULL_VALUE) {
            log.info("parameter cityId is illegal");
            model.addAttribute("code", -10028);
            return "/common/failure";
        }

        try {
            List<Long> cityIds = citiesService.getCitiesIdsByCityid(content.getCityId(), 0, Integer.MAX_VALUE);

            if (CollectionUtils.isEmpty(cityIds)) {
                log.info("there is no this city in DB");
                model.addAttribute("code", -10030);
                return "/common/failure";
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);

            Map<String, Object> map = DynamicUtil.getContentListParams(Content.TYPE_EXPERIENCE, null, null, null, null,
                    null, null, null, content.getCityId(), false);
            log.info("query experience map : " + map);

            List<Long> ids = contentService.getIdsByDynamicCondition(Content.class, map, 0, Integer.MAX_VALUE);
            log.info("get experience id list : " + ids);

            if (CollectionUtils.isEmpty(ids)) {
                //没有就插入一条数据
                content.setId(null);
                content.setType(Content.TYPE_EXPERIENCE);
                content.setBannerType(Content.INTEGER_NULL_VALUE);
                content.setApp(Content.APP_STUDENT);
                content.setTitle(Content.STRING_NULL_VALUE);
                content.setLevel(Content.INTEGER_NULL_VALUE);
                content.setSummary(Content.STRING_NULL_VALUE);
                content.setUrl(Content.STRING_NULL_VALUE);
                content.setPublishAt(Content.LONG_NULL_VALUE);
                content.setStatus(Content.STATUS_ONLINE);
                content.setReading(Content.INITIAL_VALUE);
                content.setCountdown(content.getCountdown());
                content.setCountAt(System.currentTimeMillis());
                content.setCityId(content.getCityId());
                content.setCreateBy(Long.parseLong(managerId));
                content.setUpdateBy(Long.parseLong(managerId));

                contentService.insert(content);
            } else {
                //有的话取出来编辑
                Content content1 = contentService.getObjectById(ids.get(0));
                log.info("get experience content data is " + content1);

                content1.setImg(content.getImg());
                content1.setContent(content.getContent());
                content1.setCountdown(content.getCountdown());
                content1.setCountAt(System.currentTimeMillis());
                content1.setUpdateBy(Long.parseLong(managerId));

                contentService.update(content1);
            }
            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update experience content error");
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

}