package com.ptteng.controller;

import com.gemantic.common.util.MyListUtil;
import com.gemantic.commons.code.annotation.FileTemplate;
import com.ptteng.common.yi.model.Manager;
import com.ptteng.common.yi.service.ManagerService;
import com.ptteng.model.CardExport;
import com.ptteng.util.CardUtil;
import com.ptteng.util.DynamicUtil;
import com.ptteng.util.ListUtil;
import com.ptteng.yi.nucleus.model.Card;
import com.ptteng.yi.nucleus.model.CardType;
import com.ptteng.yi.nucleus.service.CardService;
import com.ptteng.yi.nucleus.service.CardTypeService;
import com.ptteng.yi.nucleus.service.DrivingClassService;
import com.ptteng.yi.nucleus.service.SchoolService;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Card  crud
 * 
 * @author wangmeng
 * @Date 2017-12-23 13:43
 * 
 */
@Controller
public class CardController {
    private static final Log log = LogFactory.getLog(CardController.class);

    @Autowired
    private CardService cardService;

    @Autowired
    private CardTypeService cardTypeService;

    @Autowired
    private SchoolService schoolService;

    @Autowired
    private DrivingClassService drivingClassService;

    @Autowired
    private ManagerService managerService;

    @Autowired
    private CookieUtil cookieUtil;

    /**
     * 查看卡券详情
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/card/{id}", method = RequestMethod.GET)
    public String getCardJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {

        log.info("GET request url: /a/u/card/{id}, get parameter : id= " + id);
        try {
            Card card = cardService.getObjectById(id);
            log.info("get card data is :" + card);

            if (DataUtils.isNullOrEmpty(card)) {
                log.info("no this card");
                model.addAttribute("code", -9015);
                return "/common/failure";
            }

            CardType cardType = cardTypeService.getObjectById(card.getTid());
            log.info("get cardType data is :" + cardType);

            model.addAttribute("code", 0);

            model.addAttribute("card", card);
            model.addAttribute("cardType", cardType);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get card error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/yi-nucleus-service/card/json/cardDetailJson";
    }

    /**
     * 启用/禁用/核销卡券
     * @param request
     * @param response
     * @param model
     * @param id
     * @param status
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/card/{id}", method = RequestMethod.PUT)
    public String updateCardJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id, Integer status) throws Exception {

        log.info("GET request url: /a/u/card/{id}, get path variable : id= " + id + ", parameter status : " + status);

        if (DataUtils.isNullOrEmpty(status)) {
            model.addAttribute("code", -9003);
            log.info("parameter status is null or empty");
            return "/common/failure";
        }

        if (status > Card.STATUS_VERIFIED || status < Card.STATUS_BANNED) {
            model.addAttribute("code", -9016);
            log.info("parameter status is illegal");
            return "/common/failure";
        }

        try {

            Card card = cardService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(card)) {
                log.info("no this card");
                model.addAttribute("code", -9015);
                return "/common/failure";
            }

            //判断status和卡券状态进行操作
            if (card.getStatus().equals(status)) {
                log.info("there is no need changing the card status : " + card.getStatus());
                model.addAttribute("code", -9019);
                return "/common/failure";
            } else {
                //如果是禁用操作
                if (status == Card.OPERATION_BAN) {
                    if (card.getStatus() != Card.STATUS_UNEXCHANGED && card.getStatus() != Card.STATUS_UNUSED) {
                        log.info("there is not allowed to change the card status : " + card.getStatus());
                        model.addAttribute("code", -9020);
                        return "/common/failure";
                    }
                    card.setStatus(status);
                }
                //如果是启用操作
                else if (status == Card.OPERATION_USE) {
                    if (card.getStatus() != Card.STATUS_BANNED) {
                        log.info("there is not allowed to change the card status : " + card.getStatus());
                        model.addAttribute("code", -9020);
                        return "/common/failure";
                    }
                    if (!card.getExchangeAt().equals(Card.LONG_NULL_VALUE)) {
                        card.setStatus(Card.STATUS_UNUSED);
                    } else {
                        card.setStatus(Card.STATUS_UNEXCHANGED);
                    }
                }
                //如果是核销操作
                else {
                    if (card.getStatus() != Card.STATUS_USED) {
                        log.info("there is not allowed to change the card status : " + card.getStatus());
                        model.addAttribute("code", -9020);
                        return "/common/failure";
                    }
                    card.setStatus(status);
                }
            }

            String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            card.setUpdateBy(Long.parseLong(managerId));

            cardService.update(card);

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update card error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 新增卡券
     * @param request
     * @param response
     * @param model
     * @param tid 卡券类型id
     * @param amount 卡券数量
     * @param schools 可用驾校
     * @param classes 可用班型
     * @return
     * @throws Exception*/

    @RequestMapping(value = "/a/u/card", method = RequestMethod.POST)
    public String addCardJson(HttpServletRequest request, HttpServletResponse response, ModelMap model, Long tid,
            Integer amount, @RequestParam(value = "schools") Long[] schools,
            @RequestParam(value = "classes") Long[] classes) throws Exception {

        log.info("POST request url: /a/u/card, get parameter tid : " + tid + ", amount : " + amount
                + ", array schools size : " + schools.length + ", array classes size : " + classes.length);

        if (DataUtils.isNullOrEmpty(tid)) {
            model.addAttribute("code", -9021);
            log.info("parameter tid is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(amount)) {
            model.addAttribute("code", -9022);
            log.info("parameter amount is null or empty");
            return "/common/failure";
        }

        if (amount < 1) {
            model.addAttribute("code", -9023);
            log.info("parameter amount is illegal");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(schools)) {
            model.addAttribute("code", -9024);
            log.info("parameter schools is null or empty");
            return "/common/failure";
        }

        if (DataUtils.isNullOrEmpty(classes)) {
            model.addAttribute("code", -9025);
            log.info("parameter classes is null or empty");
            return "/common/failure";
        }

        try {
            List<Long> schoolIdList = Arrays.asList(schools);
            List<Long> classIdList = Arrays.asList(classes);
            //查询可用驾校和班型校验入参
            List<Long> schoolIds = schoolService.getSchoolIdsByStatusOrderByCreateAt(1, 0, Integer.MAX_VALUE);
            log.info("get available school id list : " + schoolIds);

            List<Long> drivingClassIds = drivingClassService.getDrivingClassIdsByStatusOrderByCreateAt(1, 0,
                    Integer.MAX_VALUE);
            log.info("get available drivingClass id list : " + drivingClassIds);

            if (CollectionUtils.isEmpty(schoolIds) || CollectionUtils.isEmpty(drivingClassIds)) {
                log.info("there is no schools or drivingClasses available");
                model.addAttribute("code", -9026);
                return "/common/failure";
            } else {
                if (!schoolIds.containsAll(schoolIdList) || !drivingClassIds.containsAll(classIdList)) {
                    log.info("the available ids didn't contain all ids in parameter id list");
                    model.addAttribute("code", -9027);
                    return "/common/failure";
                } else {
                    CardType cardType = cardTypeService.getObjectById(tid);

                    if (DataUtils.isNullOrEmpty(cardType)) {
                        log.info("no this cardType");
                        model.addAttribute("code", -9004);
                        return "/common/failure";
                    }

                    //通过manager获取运营中心id，即cityId
                    String managerId = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
                    Manager manager = managerService.getObjectById(Long.parseLong(managerId));

                    String school = "[" + StringUtils.join(schools, ",") + "]";
                    String clas = "[" + StringUtils.join(classes, ",") + "]";

                    List<Card> cardList = CardUtil.createCardList(tid, cardType.getExpire(), school, clas,
                            Long.parseLong(managerId), manager.getCityId(), amount);

                    log.info("batch generate card list size is : " + cardList.size());

                    if (CollectionUtils.isEmpty(cardList)) {
                        model.addAttribute("code", -9030);
                    } else {
                        List<Card> insertResults = cardService.insertList(cardList);
                        log.info("batch insert card list size is : " + insertResults.size());

                        cardType.setAmount(cardType.getAmount() + amount);
                        boolean result = cardTypeService.update(cardType);
                        log.info("update cardType result: " + result);
                    }
                }
            }

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("create card list error ");
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 获取卡券列表，导出卡券
     * @param request
     * @param response
     * @param model
     * @param number 卡券编号
     * @param creator 创建人角色名
     * @param ownerName 持有人姓名
     * @param verifier 核销人角色名
     * @param tid 卡券类型id
     * @param status 卡券状态
     * @param createAtStart 创建时间起
     * @param createAtEnd 创建时间止
     * @param exchangeAtStart 兑换时间起
     * @param exchangeAtEnd 兑换时间止
     * @param verifyAtStart 核销时间起
     * @param verifyAtEnd 核销时间止
     * @param export 是否导出excel
     * @param page
     * @param size
     * @return
     * @throws Exception*/

    @FileTemplate(templatePath = "excel/excel.vm", filePath = "/data/webs/yi-admin-web/excel/", fileName = "")
    @RequestMapping(value = "/a/u/card/list", method = RequestMethod.GET)
    public String getMultiCardJson(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            String number, String creator, String ownerName, String verifier, Long tid, Integer status,
            Long createAtStart, Long createAtEnd, Long exchangeAtStart, Long exchangeAtEnd, Long verifyAtStart,
            Long verifyAtEnd, String export, Integer page, Integer size) throws Exception {

        log.info("GET request url: /a/u/card/list, get parameter number: " + number + ", creator: " + creator
                + ", ownerName: " + ownerName + ", verifier: " + verifier + ", tid: " + tid + ", status: " + status
                + ", createAtStart: " + createAtStart + ", createAtEnd: " + createAtEnd + ", exchangeAtStart: "
                + exchangeAtStart + ", exchangeAtEnd: " + exchangeAtEnd + ", verifyAtStart: " + verifyAtStart
                + ", verifyAtEnd: " + verifyAtEnd + ", export: " + export + ", page: " + page + ", size: " + size);

        if (page == null) {
            page = 1;
        }
        if (size == null || size < 0) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        if (DataUtils.isNotNullOrEmpty(export)) {
            if (export.equals(Card.EXPORT_TYPE)) {
                log.info("parameter export = " + export + ", excel export is enabled");
                start = 0;
                size = Integer.MAX_VALUE;
            }
        }

        log.info("pageList : page= " + start + " , size=" + size);

        try {
            //先查满足条件的manager id
            List<Long> creatorIdList = null;
            List<Long> verifierIdList = null;
            if (StringUtils.isNotBlank(creator)) {
                Map<String, Object> creatorMap = DynamicUtil.getManagerListParams(creator, null, false);
                log.info("query creator map = " + creatorMap);
                creatorIdList = managerService.getIdsByDynamicCondition(Manager.class, creatorMap, 0,
                        Integer.MAX_VALUE);
            }
            if (StringUtils.isNotBlank(verifier)) {
                Map<String, Object> verifierMap = DynamicUtil.getManagerListParams(null, verifier, false);
                log.info("query verifier map = " + verifierMap);
                verifierIdList = managerService.getIdsByDynamicCondition(Manager.class, verifierMap, 0,
                        Integer.MAX_VALUE);
            }

            Map<String, Object> map = DynamicUtil.getCardListParams(number, creatorIdList, ownerName, verifierIdList,
                    tid, status, createAtStart, createAtEnd, exchangeAtStart, exchangeAtEnd, verifyAtStart, verifyAtEnd,
                    false);
            log.info("query card map = " + map);

            List<Long> cids = cardService.getIdsByDynamicCondition(Card.class, map, start, size);
            log.info("get dynamic query card list ids is: " + cids);

            Integer total = cardService.getIdsByDynamicCondition(Card.class, map, 0, Integer.MAX_VALUE).size();
            log.info("get dynamic query card total is: " + total);

            if (CollectionUtils.isEmpty(cids)) {
                model.addAttribute("cardList", new ArrayList());
                model.addAttribute("cidCreatorName", new HashMap());
                model.addAttribute("cidVerifierName", new HashMap());
                model.addAttribute("code", 0);
            } else {
                List<Card> cardList = cardService.getObjectsByIds(cids);
                log.info("get card list size is " + cardList.size());

                List<Long> cardTypeIds = MyListUtil.getFieldValueListFromModelList(cardList, false,
                        Card.class.getDeclaredField("tid"));
                log.info("get cardType id list is : " + cardTypeIds);
                List<CardType> cardTypeList = cardTypeService.getObjectsByIds(cardTypeIds);
                log.info("get cardType list size is : " + cardTypeList.size());

                List<Integer> expireList = MyListUtil.getFieldValueListFromModelList(cardTypeList, false,
                        CardType.class.getDeclaredField("expire"));
                List<String> titleList = MyListUtil.getFieldValueListFromModelList(cardTypeList, false,
                        CardType.class.getDeclaredField("title"));
                List<Integer> valueList = MyListUtil.getFieldValueListFromModelList(cardTypeList, false,
                        CardType.class.getDeclaredField("value"));

                List<Long> creatorIds = MyListUtil.getFieldValueListFromModelList(cardList, false,
                        Card.class.getDeclaredField("createBy"));
                log.info("get creator id list is : " + creatorIds);
                List<Manager> creatorList = managerService.getObjectsByIds(creatorIds);
                log.info("get creator list size is : " + creatorList.size());
                List<String> creatorNames = MyListUtil.getFieldValueListFromModelList(creatorList, false,
                        Manager.class.getDeclaredField("name"));

                // verifiedBy有可能为-1，所以数据不一定和card相等，需要处理
                List<Long> verifierIds = MyListUtil.getFieldValueListFromModelList(cardList, false,
                        Card.class.getDeclaredField("verifiedBy"));
                log.info("get verifier id list is : " + verifierIds);
                List<Manager> verifierList = managerService.getObjectsByIds(verifierIds);
                log.info("get verifier list size is : " + verifierList.size());
                List<String> verifierNames = MyListUtil.getFieldValueListFromModelList(verifierList, false,
                        Manager.class.getDeclaredField("name"));
                log.info("===========" + verifierNames);

                for (int i = 0; i < verifierIds.size(); i++) {
                    if (verifierIds.get(i).equals(Card.LONG_NULL_VALUE)) {
                        //核销人没有的话用空字符串代替
                        verifierNames.add(i, "");
                    }
                }

                if (!StringUtils.isEmpty(export) && export.equals(Card.EXPORT_TYPE)) {
                    setIndentName(cardList);
                    CardExport cardExport = new CardExport();
                    Map<String, String> config = cardExport.getConfig();
                    model.put("config", config);
                    String[] columns = cardExport.getKeys();
                    model.put("configKeys", columns);

                    for (int i = 0; i < cardList.size(); i++) {
                        Card card = cardList.get(i);
                        //卡券编号前加一个单引号，避免excel打印成科学计数格式
                        card.setNumber("'" + card.getNumber());
                        card.setExpire(expireList.get(i));
                        card.setTitle(titleList.get(i));
                        card.setValue(valueList.get(i));
                        card.setCreator(creatorNames.get(i));
                        card.setVerifier(verifierNames.get(i));
                    }

                    model.addAttribute("list", cardList);

                } else {
                    //构建失效天数Map
                    Map<Long, Integer> cidExpire = ListUtil.convert2Map(cids, expireList);
                    //构建标题名称Map
                    Map<Long, String> cidTitle = ListUtil.convert2Map(cids, titleList);
                    //构建价值Map
                    Map<Long, Integer> cidValue = ListUtil.convert2Map(cids, valueList);
                    //构建创建人名称Map
                    Map<Long, String> cidCreatorName = ListUtil.convert2Map(cids, creatorNames);
                    //构建核销人名称Map
                    Map<Long, String> cidVerifierName = ListUtil.convert2Map(cids, verifierNames);

                    model.addAttribute("page", page);
                    model.addAttribute("size", size);
                    model.addAttribute("total", total);

                    model.addAttribute("cardList", cardList);
                    model.addAttribute("cidCreatorName", cidCreatorName);
                    model.addAttribute("cidVerifierName", cidVerifierName);
                    model.addAttribute("cidExpire", cidExpire);
                    model.addAttribute("cidTitle", cidTitle);
                    model.addAttribute("cidValue", cidValue);
                }

                model.addAttribute("code", 0);
            }

        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("get card list error, start is  " + start + " , size " + size);
            model.addAttribute("code", -100000);
        }

        return "/yi-nucleus-service/card/json/cardListJson";
    }

    public static void setIndentName(List<Card> list) {
        log.info(" setIndentName start ... ");
        for (Card card : list) {
            Integer status = card.getStatus();
            if (null != status) {
                if (Card.STATUS_BANNED == status) {
                    card.setStatusStr("已禁用");
                } else if (Card.STATUS_UNEXCHANGED == status) {
                    card.setStatusStr("未兑换");
                } else if (Card.STATUS_UNUSED == status) {
                    card.setStatusStr("未使用");
                } else if (Card.STATUS_USED == status) {
                    card.setStatusStr("已使用");
                } else if (Card.STATUS_VERIFIED == status) {
                    card.setStatusStr("已核销");
                } else if (Card.STATUS_EXPIRED == status) {
                    card.setStatusStr("已过期");
                } else {
                    card.setStatusStr("");
                }
            }

            card.setCreateDate(getDate(card.getCreateAt()));
            card.setExpireDate(getDate(card.getExpireAt()));
            if (card.getExchangeAt() != -1) {
                card.setExchangeDate(getDate(card.getExchangeAt()));
            } else {
                card.setExchangeDate("");
            }

            if (card.getVerifyAt() != -1) {
                card.setVerifyDate(getDate(card.getVerifyAt()));
            } else {
                card.setVerifyDate("");
            }
        }
    }

    public static String getDate(Long time) {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String date = "";

        if (null != time) {
            date = dateFormat.format(new Date(time));
        }
        return date;
    }

}
