package com.ptteng.rent.etl;

import com.alibaba.fastjson.JSON;
import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.rent.etl.constant.WxConfig;
import com.ptteng.rent.etl.util.DynamicUtil;
import com.ptteng.rent.etl.util.wx.model.JsonResult;
import com.ptteng.rent.etl.util.wx.model.ResponseData;
import com.ptteng.rent.etl.util.wx.transfer.*;
import com.ptteng.rent.user.model.Orders;
import com.ptteng.rent.user.service.OrdersService;
import com.qding.community.common.weixin.service.WeiXinService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author cjsff
 * 检测需要给承租人退款的订单
 */
public class LesseeRefundOrdersEtl {

    private static final Log log = LogFactory.getLog(LesseeRefundOrdersEtl.class);

    private OrdersService ordersService;

    @Autowired
    private WeiXinService weixinService;

    /**
     * 一次任务检测多少条数据
     */
    private static final int TASK_LEN = 1;

    private static final String REFUND = "https://api.mch.weixin.qq.com/secapi/pay/refund";

    private static final String APP_ID = ConfigUtil.getProperty("wx.appid");

    private static final String MCH_ID = ConfigUtil.getProperty("wx.mchid");

    private static final String API_SECRET = ConfigUtil.getProperty("wx.api.secret");

    /**
     * 空转任务间隔休息毫秒数
     */
    private static final long SLEEP_MILLISECOND = 10000;

    private Long interval = 2000L;

    public LesseeRefundOrdersEtl() {
        super();
    }

    public void process() throws InterruptedException {

        while (true) {

            try {

                log.info("--------------ETL 6-----------------");

                log.info("find lessee refund orders etl is start!");

                // 查询正常结束的订单
                List<Orders> normalEndOrderList = findNormalEndOrders();

                log.info("normal end order list size :" + normalEndOrderList.size());

                if (CollectionUtils.isEmpty(normalEndOrderList)) {

                    log.info("orders push etl not get any id ,sleep " + SLEEP_MILLISECOND + " ms ");

//                    Thread.sleep(SLEEP_MILLISECOND);

                }else {
                    processOrder(normalEndOrderList);
                }

                Thread.sleep(SLEEP_MILLISECOND);

            } catch (Throwable t) {

                t.printStackTrace();

                Thread.sleep(SLEEP_MILLISECOND);

                log.error("process goods bytime status error ,sleep " + t.getMessage());

            }

        }

    }


    /**
     * 寻找需要给承租人退款的订单
     *
     * @return
     */
    private List<Orders> findNormalEndOrders() {

        log.info("============------------===============");

        log.info("find normal end order method start!!!");

        List<Orders> normalEndOrderList = new ArrayList();

        Map<String, Object> conditions = DynamicUtil.getLesseeRefundOrderList();

        log.info("get normal end order params ；" + conditions);

        try {

            List<Long> normalEndOrderIds = this.ordersService.getIdsByDynamicCondition(Orders.class, conditions,
                    0, TASK_LEN);

            if (CollectionUtils.isNotEmpty(normalEndOrderIds)) {

                normalEndOrderList = this.ordersService.getObjectsByIds(normalEndOrderIds);

            }


        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t);
        }

        return normalEndOrderList;
    }


    /**
     * 给出租人转账
     *
     * @param normaoEndOrderList
     * @throws ServiceException
     * @throws ServiceDaoException
     * @throws InterruptedException
     * @throws IOException
     */
    private void processOrder(List<Orders> normaoEndOrderList) throws ServiceException, ServiceDaoException,
            InterruptedException,
            IOException, XmlPullParserException {

        log.info("***********************************");

        log.info("Transfer to lessor");

        if (CollectionUtils.isNotEmpty(normaoEndOrderList)) {

            Map<String, String> restmap = null;
            for (Orders orders : normaoEndOrderList) {
                String refundOrderNo = generateCode();
                Integer orderStatus = orders.getStatus();
                log.info("order status is :" + orderStatus);

                String refund = "";

                // 退全款（订单金额+服务费）
                if (isRetirement(orderStatus)) {

                    refund = orders.getTotalPrice().multiply(new BigDecimal("100")).stripTrailingZeros().toPlainString().toString();

                    log.info("refund is :" + refund);

                // 退订单金额
                } else if (isOrderFee(orderStatus)) {

                    refund = orders.getPrice().multiply(new BigDecimal("100")).stripTrailingZeros().toPlainString().toString();

                    log.info("order fee refund is :" + refund);

                }

                try {
                    log.info("refund order number is :" + refundOrderNo);
                    Map<String, String> parm = new HashMap<String, String>();
                    parm.put("appid", APP_ID);
                    parm.put("mch_id", MCH_ID); //商户号
                    parm.put("nonce_str", PayUtil.getNonceStr()); //随机字符串
                    parm.put("out_trade_no", orders.getOrderNo()); //商户订单号
                    parm.put("out_refund_no", refundOrderNo); //退款订单号
                    parm.put("total_fee", orders.getTotalPrice().multiply(new BigDecimal("100")).stripTrailingZeros().toPlainString().toString()); //订单金额
                    parm.put("refund_fee", refund); //退款金额
                    log.info("------------------" + PayUtil.getSign(parm, API_SECRET));
                    parm.put("sign", PayUtil.getSign(parm, API_SECRET));
                    log.info("parm is :" + parm);
                    String restxml = HttpUtils.posts(REFUND, XmlUtil.xmlFormat(parm, false));
                    restmap = XmlUtil.xmlParse(restxml);
                    log.info("rest map is :" + restmap);
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }

                if (CollectionUtil.isNotEmpty(restmap) && "SUCCESS".equals(restmap.get("result_code"))) {

                    log.info("转账成功：" + restmap.get("err_code") + ":" + restmap.get("err_code_des"));
                    orders.setStatus(Orders.REFUNDED);
                    orders.setRefundOrderNo(refundOrderNo);

                } else {
                    if (CollectionUtil.isNotEmpty(restmap)) {
                        log.info("转账失败：" + restmap.get("err_code") + ":" + restmap.get("err_code_des"));
                    }

                }

            }

            boolean boo = ordersService.updateList(normaoEndOrderList);

            if (boo) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                log.info("-------------------------");

            }

        }

    }

    private static String generateCode() {
        // 14位日期
        Long now = Long.parseLong(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
        // 随机码3位
        int num = (int) (Math.random() * 900) + 100;

        return now.toString() + Integer.toString(num);
    }

    public OrdersService getOrdersService() {
        return ordersService;
    }

    public void setOrdersService(OrdersService ordersService) {
        this.ordersService = ordersService;
    }

    public Long getInterval() {
        return interval;
    }

    public void setInterval(Long interval) {
        this.interval = interval;
    }


    /**
     * 3={退全款}
     * 8={退全款}
     * 14={退全款}
     * @param orderStatus
     * @return
     */
    private static Boolean isRetirement(Integer orderStatus) {

        if (Orders.LESSOR_IS_CONFIRMED_LESSEE_REFUND.equals(orderStatus)
                || Orders.ORDER_TIME_OUT_CANCEL.equals(orderStatus)
                || Orders.REFUSE_TO_TAKE_ORDERS.equals(orderStatus)) {

            return true;

        }

        return false;

    }

    /**
     * 5={不退还服务费}
     * 6={不退还服务费}
     * 7={不退还服务费}
     * 10=
     */
    private static Boolean isOrderFee(Integer orderStatus) {

        if (Orders.PROCESSING_LESSEE_REFUND_LESSOR_IS_CONFIRMED.equals(orderStatus)
                || Orders.PROCESSING_LESSEE_REFUND_SYSTEM_REFUND.equals(orderStatus)
                || Orders.PROCESSING_LESSOR_TERMINATION_ORDER.equals(orderStatus)) {

            return true;

        }

        return false;

    }

}
