package com.ptteng.keeper.admin.controller;

import com.ptteng.keeper.admin.model.Manager;
import com.ptteng.keeper.admin.utils.DynamicUtil;
import com.ptteng.keeper.admin.utils.FormValidationUtil;
import com.ptteng.keeper.admin.utils.LogUtil;
import com.ptteng.keeper.common.model.Product;
import com.ptteng.keeper.common.service.ProductService;
import com.qding.common.util.DataUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


/**
 * Product  crud
 *
 * @author magenm
 * @Date 2014-4-16 13:43
 */
@Controller
public class ProductController {
    private static final Log log = LogFactory.getLog(ProductController.class);

    @Autowired
    private ProductService productService;

    /**
     * show 产品详情
     *
     * @param id 产品id
     * @return product json
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/{id}", method = RequestMethod.GET)
    public String getProductDetail(HttpServletRequest request,
                                   HttpServletResponse response, ModelMap model,
                                   @PathVariable Long id) throws Exception {

        log.info(" getProductDetail: id= " + id);
        if (DataUtils.isNullOrEmpty(id)) {
            model.addAttribute("code", -1100);
            return "/common/success";
        }
        try {
            Product product = productService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(product)) {
                model.addAttribute("code", -1005);
                log.info("Data does not exist,id is " + id);
                return "/common/success";
            }
            model.addAttribute("code", 0);

            model.addAttribute("product", product);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get product list error");
            // for test
            model.addAttribute("code", -100000);
        }

        return "/keeper/product/json/productDetailJson";
    }


    /**
     * show 产品列表
     *
     * @param type         产品类型 1贷款 2信用卡
     * @param name         产品名
     * @param belongModule 所属模块
     * @param settleMode   结算类型
     * @param status       状态
     * @param amountEnd    金额上限
     * @return
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/search", method = RequestMethod.GET)
    public String getProductList(HttpServletRequest request,
                                 HttpServletResponse response, ModelMap model, Integer page,
                                 Integer size, Integer type, String name, Integer belongModule, Integer settleMode,
                                 Integer status, BigDecimal amountEnd
    ) throws Exception {

        if (page == null) {
            page = 1;
        }
        if (size == null) {
            size = 10;
        }
        if (type == null) {
            type = 1;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        log.info("pageList : page= " + start + " , size=" + size);
        log.info("/a/u/product/search: " + LogUtil.convertParam("type", type, "name", name, "belongModule", belongModule,
                "settleMode", settleMode, "status", status, "amountEnd", amountEnd));
        try {
            Map<String, Object> params = DynamicUtil.getProductList(type, name, belongModule, settleMode, status, amountEnd, false);
            List<Long> ids = productService.getIdsByDynamicCondition(Product.class, params, start, size);
            List<Product> productList = new ArrayList<>();
            if (ids.size() > 0) {
                productList = productService.getObjectsByIds(ids);
            }
            Map<String, Object> totalParams = DynamicUtil.getProductList(type, name, belongModule, settleMode, status, amountEnd, true);
            List<Long> totalIds = productService.getIdsByDynamicCondition(Product.class, totalParams, 0, Integer.MAX_VALUE);
            Long total = totalIds.get(0);

            log.info("get product total= " + total);

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);

            model.addAttribute("total", total);

            model.addAttribute("productList", productList);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get product list error,page is  " + start + " , size "
                    + size);
            // for test
            model.addAttribute("code", -100000);
        }

        return "/keeper/product/json/productListJson";
    }


    /**
     * show 产品上下架
     *
     * @param id     产品id
     * @param status 状态 0下架 1上架
     * @return 操作结果
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/{id}/status", method = RequestMethod.PUT)
    public String updateProductStatus(HttpServletRequest request,
                                      HttpServletResponse response, ModelMap model, @PathVariable Long id, Integer status)
            throws Exception {

        log.info("update product : id= " + id + " ,status=" + status);
        if (DataUtils.isNullOrEmpty(id)) {
            model.addAttribute("code", -1100);
            return "/common/success";
        }
        if (DataUtils.isNullOrEmpty(status)) {
            model.addAttribute("code", -2002);
            return "/common/success";
        } else if (Product.STATUS_UP != status && Product.STATUS_DOWN != status) {
            model.addAttribute("code", -2002);
            return "/common/success";
        }
        try {
            Manager manager = (Manager) request.getAttribute("manager");
            Product product = productService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(product)) {
                model.addAttribute("code", -1005);
                log.info("Data does not exist,id is " + id);
                return "/common/success";
            }
            if (!product.getStatus().equals(status)) {
                product.setStatus(status);
                product.setUpdateBy(manager.getId());
                product.setStatusChangeAt(System.currentTimeMillis());
                productService.update(product);
                log.info("update:product status success ,manager is " + manager.getId());
            }
            model.addAttribute("code", 0);

            model.addAttribute("product", product);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get product error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/keeper/product/json/productDetailJson";
    }

    /**
     * show 产品修改
     *
     * @param product product 对象
     * @return 操作结果
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/{id}", method = RequestMethod.PUT)
    public String updateProductJson(HttpServletRequest request,
                                    HttpServletResponse response, ModelMap model, Product product, @PathVariable Long id) throws Exception {

        log.info("update product : product= " + product);
        if (DataUtils.isNullOrEmpty(id)) {
            model.addAttribute("code", -1100);
            return "/common/success";
        }
        try {
            if (!FormValidationUtil.createProduct(product, model)) {
                return "/common/success";
            }
            Manager manager = (Manager) request.getAttribute("manager");
            Product oldProduct = productService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(oldProduct)) {
                model.addAttribute("code", -1005);
                log.info("Data does not exist,id is " + id);
                return "/common/success";
            }
            //接口不可修改数据
            product.setUpdateBy(manager.getId());
            product.setCreateBy(manager.getCreateBy());
            product.setCreateAt(oldProduct.getCreateAt());
            product.setStatusChangeAt(oldProduct.getStatusChangeAt());
            product.setApplyCount(oldProduct.getApplyCount());
            product.setStatus(oldProduct.getStatus());
            product.setSort(oldProduct.getSort());
            productService.update(product);
            log.info("update product success,manager is " + manager.getId());
            model.addAttribute("code", 0);

            model.addAttribute("product", product);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("update product error,id is  " + product.getId());
            model.addAttribute("code", -6003);

        }

        return "/data/json";
    }

    /**
     * show 产品新增
     *
     * @param product product 对象
     * @return 操作结果
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product", method = RequestMethod.POST)
    public String addProductJson(HttpServletRequest request,
                                 HttpServletResponse response, ModelMap model, Product product) throws Exception {

        log.info("create product : product= " + product);
        try {
            if (!FormValidationUtil.createProduct(product, model)) {
                return "/common/success";
            }

            Manager manager = (Manager) request.getAttribute("manager");
            product.setId(null);
            product.setStatus(Product.STATUS_UP);
            product.setCreateBy(manager.getId());
            product.setUpdateBy(manager.getId());
            Map<String, Object> map = DynamicUtil.getProductMaxSort(product.getType());
            List<Long> ids = productService.getIdsByDynamicCondition(Product.class, map, 0, Integer.MAX_VALUE);
            product.setSort(1);
            if (ids.size()>0) {
                Integer sort = productService.getObjectById(ids.get(0)).getSort();
                product.setSort(sort + 1);
            }
            log.info("product is " + product);
            Long insertResult = productService.insert(product);
            log.info("create product success ,id is " + insertResult);
            model.addAttribute("code", 0);
            model.addAttribute("id", insertResult);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add product error ");
            model.addAttribute("code", -6002);
        }

        return "/common/insert";
    }

    /**
     * show 产品删除
     *
     * @param id 产品id
     * @return 操作结果
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/{id}", method = RequestMethod.DELETE)
    public String deleteProductJson(HttpServletRequest request,
                                    HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("delete product : id= " + id);
        try {
            if (DataUtils.isNullOrEmpty(id)) {
                model.addAttribute("code", -1100);
                return "/common/success";
            }
            Manager manager = (Manager) request.getAttribute("manager");
            Product product = productService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(product)) {
                model.addAttribute("code", -1005);
                log.info("Data does not exist,id is " + id);
                return "/common/success";
            }

            //申请过就不能删
            if(product.getApplyCount()>0){
                model.addAttribute("code", -4002);
                return "common/success";
            }
            productService.delete(id);
            log.info("delete product success,product id=" + id + " ,manager=" + manager.getId());
            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("delete product error,id is  " + id);
            model.addAttribute("code", -6004);

        }
        return "/data/json";
    }

    /**
     * show 产品拖动排序
     *
     * @param ids 需要排序的列表
     * @return java.lang.String
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/10
     */

    @RequestMapping(value = "/a/u/product/sort", method = RequestMethod.PUT)
    public String sortProduct(HttpServletRequest request,
                              HttpServletResponse response, ModelMap model, @RequestBody List<Long> ids)
            throws Exception {
        log.info("level product:ids=" + ids);
        try {
            List<Product> products = this.productService.getObjectsByIds(ids);
            int index = 1;
            for (Product product : products) {
                product.setSort(index);
                index++;
            }
            this.productService.updateList(products);
            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("order products error ");
            model.addAttribute("code", -1);
        }
        return "/common/success";
    }

}

