package com.ptteng.common.skill.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.common.skill.model.*;
import com.ptteng.common.skill.service.*;
import com.ptteng.common.skill.util.DataValidatorUtil;
import com.ptteng.common.skill.util.DynamicUtil;
import com.ptteng.common.skill.util.GetUserFromCookieUtil;
import com.ptteng.common.skill.util.UserUtil;
import com.qding.common.util.DataUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Record  crud
 *
 * @author magenm
 * @Date 2014-4-16 13:43
 */

@Controller
public class RecordController {

    private static final Log log = LogFactory.getLog(RecordController.class);

    @Autowired
    private RecordService recordService;

    @Autowired
    private RecordLogService recordLogService;

    @Autowired
    private UserService userService;

    @Autowired
    private ProvincesService provincesService;

    @Autowired
    private CitiesService citiesService;

    @Autowired
    private CountiesService countiesService;

    @Autowired
    private SchoolService schoolService;

    @Autowired
    private MajorService majorService;

    @Autowired
    private TemplateTestService templateTestService;

    @Resource
    private GetUserFromCookieUtil getUserFromCookieUtil;


    /**
     * 新建档案
     *
     * @param: request 请求对象
     * @param: model 模型对象
     * @param: record 要创建的档案详情
     */
    @RequestMapping(value = "/a/u/record", method = RequestMethod.POST)
    public String createNewRecord(HttpServletRequest request, ModelMap model, Record record) {

        log.info("POST /a/u/record: prepare to create new record: " + record);

        //参数校验
        if (!DataValidatorUtil.checkRecord(record, model)) {
            log.info("params error");
            return "/common/failure";
        }

        try {
            Long uid = record.getUid();
            Long userId = getUserFromCookieUtil.getUserFromCookie(request);
            if (!ObjectUtils.equals(userId, uid)) { // 不允许为他人创建档案
                log.info("no permission: can't create record for other");
                model.addAttribute("code", -38001);
                return "/common/failure";
            }

            User u = userService.getObjectById(uid);
            if (DataUtils.isNullOrEmpty(u)) { // 新建档案失败，该学员不存在
                log.info("create record failure, because user in record is not exist, uid = " + record.getUid());
                model.addAttribute("code", -38002);
                return "/common/failure";
            }

            // 个人档案功能只对内门弟子、首席师兄、荣耀师兄开放
            if (!UserUtil.isOffline(u.getIdentity())) {
                log.info("current user has no permission to create record: user's identity: " + u.getIdentity());
                model.addAttribute("code", -38020);
                return "/common/failure";
            }

            Long rid = recordService.getRecordIdByUidOrderByCreateAt(uid);
            if (DataUtils.isNotNullOrEmpty(rid)) { // 不允许重复创建档案
                log.info("can not create record repeatedly，current user's recordId = " + rid);
                model.addAttribute("code", -38003);
                return "/common/failure";
            }

            record.setSchoolType(getSchoolType(record.getSchool()));
            record.setStatus(Record.STATUS_APPLYING);
            record.setCreateBy(uid);
            record.setUpdateBy(uid);
            Long recordId = recordService.insert(record);
            log.info("insert new record success, recordId = " + recordId);

            //内门弟子新建档案之后，更新user表的record_status字段
            User user = userService.getObjectById(userId);
            user.setRecordStatus(Record.STATUS_APPLYING);
            boolean update = userService.update(user);
            log.info("update user's filed record_status result: " + update);

            model.addAttribute("code", 0);
            model.addAttribute("id", recordId);

        } catch (Exception e) {

            e.printStackTrace();
            e.getMessage();
            log.info("insert new record failure");
            model.addAttribute("code", -38000); //新建档案失败，请稍后再尝试
        }
        return "/common/insert";
    }


    /**
     * 查看本人档案
     *
     * @param: request 请求对象
     * @param: model 模型对象
     */
    @RequestMapping(value = "/a/u/record", method = RequestMethod.GET)
    public String getMyRecord(HttpServletRequest request, ModelMap model) {

        User user = (User) request.getAttribute("user");
        log.info("GET /a/u/record: user want to get his record, uid = " + user.getId());

        //个人档案功能只对内门弟子、首席师兄、荣耀师兄开放
        if (!UserUtil.isOffline(user.getIdentity())) {
            log.info("current user has no permission to check record: user's identity: " + user.getIdentity());
            model.addAttribute("code", -38020);
            return "/common/failure";
        }

        try {
            Long rid = recordService.getRecordIdByUidOrderByCreateAt(user.getId());
            if (DataUtils.isNullOrEmpty(rid)) { // 尚未提交档案
                log.info("current user never commit his record");
                model.addAttribute("code", -38004);
                return "common-skill-service/record/json/areaNumberJson";
            }

            Record record = recordService.getObjectById(rid);
            log.info("current user's record = " + record);

            model.addAttribute("code", 0);
            model.addAttribute("record", record);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get record failure, try again later");
            model.addAttribute("code", -38005);
        }
        return "/common-skill-service/record/json/recordDetailJson";
    }

    /**
     * 查看审核记录
     */
    @RequestMapping(value = "/a/u/recordlog/{recordLogId}", method = RequestMethod.GET)
    public String getNewestRecordLog(ModelMap model, @PathVariable Long recordLogId) {

        log.info("GET /a/u/record/recordLog: get user's newest record log by recordLogId = " + recordLogId);

        try {
            RecordLog recordLog = recordLogService.getObjectById(recordLogId);
            if (DataUtils.isNotNullOrEmpty(recordLog)) {
                model.addAttribute("recordLog", recordLog);
            }

            model.addAttribute("code", 0);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get newest record log failure, try again later");
            model.addAttribute("code", -38034);
        }
        return "/common-skill-service/recordLog/json/recordLogDetailJson";
    }

    /**
     * 获取省市县编号
     *
     * @param province 省名称
     * @param city     城市名称
     * @param county   区县名称
     */
    @RequestMapping(value = "/a/area/number", method = RequestMethod.GET)
    public String getAreaNumberByName(ModelMap model, String province, String city, String county) {

        log.info("GET /a/u/area/number: get provinceID && cityID && countyID by params: province = " + province
                + ", city = " + city + ", county = " + county);

        try {
            Long provinceId = -1L; //若是通过后台新增的档案，因为省市县都是-1，所以省市县编号的默认值也设为-1
            Long cityId = -1L;
            Long countyId = -1L;

            if (StringUtils.isNotEmpty(province)) {
                provinceId = templateTestService.getProvinceIdByProvince(province);
                log.info("get provinceID : " + provinceId);
            }

            if (StringUtils.isNotEmpty(city)) {
                cityId = templateTestService.getCityIdByCity(city);
                log.info("get cityID: " + cityId);
            }

            if (StringUtils.isNotEmpty(county) && ObjectUtils.notEqual(cityId, -1L)) {
                countyId = templateTestService.getCountyIdByCounty(county, cityId);
                log.info("get countyID: " + countyId);
            }

            model.addAttribute("code", 0);
            model.addAttribute("provinceId", provinceId);
            model.addAttribute("cityId", cityId);
            model.addAttribute("countyId", countyId);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get area number failure, try again later");
            model.addAttribute("code", -38035);
        }
        return "/common-skill-service/record/json/areaNumberJson";
    }


    /**
     * 修改档案
     *
     * @param record  档案要修改的信息
     * @param request 请求对象
     */
    @RequestMapping(value = "/a/u/record", method = RequestMethod.PUT)
    public String modifyMyRecord(HttpServletRequest request, ModelMap model, Record record) {

        log.info("PUT /a/u/record: prepare to set record like:" + record);

        //参数校验
        if (!DataValidatorUtil.checkRecord(record, model)) {
            log.info("params error");
            return "/common/failure";
        }

        try {
            User user = (User) request.getAttribute("user");
            // 个人档案功能只对内门弟子开放
            if (!UserUtil.isOffline(user.getIdentity())) {
                log.info("current user has no permission to modify record: user's identity: " + user.getIdentity());
                model.addAttribute("code", -38020);
                return "/common/failure";
            }

            //先判断该档案是否存在
            Long rid = record.getId();
            if (DataUtils.isNullOrEmpty(rid)) {
                log.info("no such record which id = " + rid);
                model.addAttribute("code", -38007);
                return "/common/failure";
            }

            Record realRecord = recordService.getObjectById(rid);
            log.info("record before modify: " + realRecord);
            if (DataUtils.isNullOrEmpty(realRecord)) {
                log.info("no such record which id = " + rid);
                model.addAttribute("code", -38007);
                return "/common/failure";
            }

            //再判断是不是本人的档案 && 不可以修改他人的档案
            if (!ObjectUtils.equals(user.getId(), record.getUid()) || !ObjectUtils.equals(user.getId(), realRecord.getUid())
                    || !ObjectUtils.equals(realRecord.getUid(), record.getUid())) {
                log.info("can not modify other's record: current userId = " + user.getId() + ", uid-in-record = " + record.getUid()
                        + "uid-in-realRecord = " + realRecord.getUid());
                model.addAttribute("code", -38008);
                return "/common/failure";
            }

            //档案的状态为“申请中”或“已存档”时，不可修改。状态为 未通过 才可以修改
            if (!ObjectUtils.equals(realRecord.getStatus(), Record.STATUS_NO_PASS)) {
                log.info("cant not modify, record status before modify: " + realRecord.getStatus() + " not equals" + Record.STATUS_NO_PASS);
                model.addAttribute("code", -38009);
                return "/common/failure";
            }

            // 并且，还要统一设置编辑后的status为“申请中”
            realRecord.setAvatar(record.getAvatar());
            realRecord.setName(record.getName());
            realRecord.setSex(record.getSex());
            realRecord.setIdentityNum(record.getIdentityNum());
            realRecord.setBornAt(record.getBornAt());
            realRecord.setProvince(record.getProvince());
            realRecord.setCity(record.getCity());
            realRecord.setCounty(record.getCounty());
            realRecord.setEducatedStartAt(record.getEducatedStartAt());
            realRecord.setEducatedEndAt(record.getEducatedEndAt());
            realRecord.setGrade(record.getGrade());
            realRecord.setSchool(record.getSchool());
            realRecord.setSchoolType(getSchoolType(record.getSchool()));
            realRecord.setMajor(record.getMajor());
            realRecord.setIsWork(record.getIsWork());
            realRecord.setWorkExp(record.getWorkExp());
            realRecord.setContactName(record.getContactName());
            realRecord.setContactRelation(record.getContactRelation());
            realRecord.setContactMobile(record.getContactMobile());
            realRecord.setCardNum(record.getCardNum());
            realRecord.setBank(record.getBank());
            realRecord.setSource(record.getSource());
            realRecord.setIdentityFrontUrl(record.getIdentityFrontUrl());
            realRecord.setIdentityReverseUrl(record.getIdentityReverseUrl());
            realRecord.setIsLoan(record.getIsLoan());
            realRecord.setHouseholdRegisterUrl(record.getHouseholdRegisterUrl());
            realRecord.setStatus(Record.STATUS_APPLYING);
            realRecord.setUpdateBy(user.getId());
            recordService.update(realRecord);

            user.setRecordStatus(Record.STATUS_APPLYING);
            userService.update(user);

            model.addAttribute("code", 0);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("modify record failure");
            model.addAttribute("code", -38006); //修改档案失败，请稍后再尝试
        }
        return "/common/success";
    }


    /**
     * 获取全国各省市区县数据
     */
    @RequestMapping(value = "/a/u/area/list", method = RequestMethod.GET)
    public String getAreaList(ModelMap model) {

        log.info("GET /a/u/area/list");

        try {
            List<Long> provincesIds = provincesService.getProvincesIds(0, Integer.MAX_VALUE);
            log.info("get provincesIds size = " + provincesIds.size());

            List<Long> citiesIds = citiesService.getCitiesIds(0, Integer.MAX_VALUE);
            log.info("get citiesIds  size = " + citiesIds.size());

            List<Long> countiesIds = countiesService.getCountiesIds(0, Integer.MAX_VALUE);
            log.info("get countiesIds size = " + countiesIds.size());

            if (CollectionUtils.isEmpty(provincesIds) || CollectionUtils.isEmpty(citiesIds) || CollectionUtils.isEmpty(countiesIds)) {
                log.info("get area failure");
                model.addAttribute("code", -38010); //省市县数据不存在
                return "/common/failure";
            }

            List<Provinces> provincesList = provincesService.getObjectsByIds(provincesIds);
            log.info("get provincesList success, size = " + provincesList.size());

            List<Cities> citiesList = citiesService.getObjectsByIds(citiesIds);
            log.info("get citiesList success, size = " + citiesList.size());

            List<Counties> countiesList = countiesService.getObjectsByIds(countiesIds);
            log.info("get countiesList success, size = " + countiesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("provinceList", provincesList);
            model.addAttribute("cityList", citiesList);
            model.addAttribute("countyList", countiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get areas failure, try again later ");
            model.addAttribute("code", -38011);
        }
        return "/common-skill-service/provinces/json/areaListJson";
    }


    /**
     * 获取全国各省数据
     */
    @RequestMapping(value = "/a/u/province/list", method = RequestMethod.GET)
    public String getProvinceList(ModelMap model) {

        log.info("GET /a/u/province");

        try {
            List<Long> provincesIds = provincesService.getProvincesIds(0, Integer.MAX_VALUE);
            log.info("get provincesIds size = " + provincesIds.size());

            if (CollectionUtils.isEmpty(provincesIds)) {
                log.info("get provinceList failure");
                model.addAttribute("code", -38012); //省数据不存在
                return "/common/failure";
            }

            List<Provinces> provincesList = provincesService.getObjectsByIds(provincesIds);
            log.info("get provincesList success, size = " + provincesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", provincesList.size());
            model.addAttribute("provincesList", provincesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get provinceList failure, try again later");
            model.addAttribute("code", -38013);
        }
        return "/common-skill-service/provinces/json/provincesListJson";
    }


    /**
     * 获取某省的市数据
     *
     * @param provinceId 某省ID
     */
    @RequestMapping(value = "/a/u/city/list", method = RequestMethod.GET)
    public String getCityList(ModelMap model, Long provinceId) {

        log.info("GET /a/u/city/list: which provinceId = " + provinceId);

        try {
            List<Long> citiesIds = citiesService.getCitiesIdsByProvinceIdOrderByCreateAt(provinceId, 0, Integer.MAX_VALUE);
            log.info("get citiesIds  size = " + citiesIds.size());

            if (CollectionUtils.isEmpty(citiesIds)) {
                log.info("get cityList failure");
                model.addAttribute("code", -38014); //市数据不存在
                return "/common/failure";
            }

            List<Cities> citiesList = citiesService.getObjectsByIds(citiesIds);
            log.info("get citiesList success, size = " + citiesList.size());


            model.addAttribute("code", 0);
            model.addAttribute("total", citiesList.size());
            model.addAttribute("citiesList", citiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get cityList failure, ty again later");
            model.addAttribute("code", -38015);
        }
        return "/common-skill-service/cities/json/citiesListJson";
    }


    /**
     * 获取某市的区县数据
     *
     * @param cityId 某市ID
     */
    @RequestMapping(value = "/a/u/county/list", method = RequestMethod.GET)
    public String getCountyList(ModelMap model, Long cityId) {

        log.info("GET /a/u/county/list: which cityId = " + cityId);

        try {

            List<Long> countiesIds = countiesService.getCountiesIdsByCityIdOrderByCreateAt(cityId, 0, Integer.MAX_VALUE);
            log.info("get countiesIds size = " + countiesIds.size());

            if (CollectionUtils.isEmpty(countiesIds)) {
                log.info("get countyList failure");
                model.addAttribute("code", -38016); //县数据不存在
                return "/common/failure";
            }

            List<Counties> countiesList = countiesService.getObjectsByIds(countiesIds);
            log.info("get countiesList success, size = " + countiesList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", countiesList.size());
            model.addAttribute("countiesList", countiesList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get cityList failure, ty again later");
            model.addAttribute("code", -38017);
        }
        return "/common-skill-service/counties/json/countiesListJson";
    }


    /**
     * 按名称查找学校-模糊搜索
     *
     * @param name 学校名称
     */
    @RequestMapping(value = "/a/u/school/search", method = RequestMethod.GET)
    public String getSchoolByName(ModelMap model, String name) {

        log.info("GET /a/u/school/search: prepare to get school list by name: " + name);

        if (StringUtils.isEmpty(name)) {
            log.info("params are null: name = " + name);
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {

            Map<String, Object> params = DynamicUtil.getSchoolByName(name);
            log.info("get params:" + params);
            //默认size=10
            int size = 10;
            List<Long> schoolIds = schoolService.getIdsByDynamicCondition(School.class, params, 0, size);
            log.info("get schoolIds size = " + schoolIds.size());
            if (CollectionUtils.isEmpty(schoolIds)) {
                model.addAttribute("code", -38018);
                return "/common/failure";
            }

            List<School> schoolList = schoolService.getObjectsByIds(schoolIds);
            log.info("get schoolList size = " + schoolList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", schoolList.size());
            model.addAttribute("schoolList", schoolList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get school failure, try again later");
            model.addAttribute("code", -38019);
        }
        return "/common-skill-service/school/json/schoolListJson";
    }


    /**
     * 搜索专业列表 - 按第一级专业名搜索
     *
     * @param major 完整精确的一级专业名
     */
    @RequestMapping(value = "/a/u/major/primary/search", method = RequestMethod.GET)
    public String getSecondaryMajorByPrimaryMajor(ModelMap model, String major) {

        log.info("GET /a/u/major/primary/search: get secondaryMajorList by primaryMajor: " + major);

        if (StringUtils.isEmpty(major)) {
            log.info("param is null");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        try {
            List<Long> majorIds = majorService.getMajorIdsByPrimaryMajor(major, 0, Integer.MAX_VALUE);
            log.info("get majorIds size: " + majorIds.size());
            if (CollectionUtils.isEmpty(majorIds)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -38022);
                return "/common-skill-service/major/json/majorListJson";
            }

            List<Major> majorList = majorService.getObjectsByIds(majorIds);
            log.info("get majorList size: " + majorList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", majorList.size());
            model.addAttribute("majorList", majorList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get majorList failure, try again later");
            model.addAttribute("code", -38024);
        }
        return "/common-skill-service/major/json/majorListJson";
    }


    //搜索专业列表
    @RequestMapping(value = "/a/u/major/{type}", method = RequestMethod.GET)
    public String getMajorList(ModelMap model, @PathVariable Integer type, String major) {

        log.info("Get /a/u/major/{type}: get majorList by type: " + type + " & major: " + major);

        if (StringUtils.isEmpty(major)) {
            log.info("param is null");
            model.addAttribute("code", -1000);
            return "/common/failure";
        }

        if (type < 1 && type > 3) {
            log.info("unsupported this value of type: " + type);
            model.addAttribute("code", -38025);
            return "/common/failure";
        }

        try {
            List<Long> majorIds = new ArrayList<>();

            if (ObjectUtils.equals(1, type)) {
                log.info("get majorList by major: " + major);
                majorIds = majorService.getMajorIdsByPrimaryMajor(major, 0, Integer.MAX_VALUE);
            } else if (ObjectUtils.equals(2, type)) {
                log.info("get majorList by major: " + major);
                majorIds = majorService.getMajorIdsBySecondryMajor(major, 0, Integer.MAX_VALUE);
            } else if (ObjectUtils.equals(3, type)) {
                Map<String, Object> params = DynamicUtil.getMajorByName(major);
                log.info("get majorList by params: " + params);
                majorIds = majorService.getIdsByDynamicCondition(Major.class, params, 0, Integer.MAX_VALUE);
            }

            log.info("get majorIds size: " + majorIds.size());

            if (CollectionUtils.isEmpty(majorIds)) {
                log.info("there are no qualified results");
                model.addAttribute("code", -38022);
                return "/common-skill-service/major/json/majorListJson";
            }

            List<Major> majorList = majorService.getObjectsByIds(majorIds);
            log.info("get majorList size: " + majorList.size());

            model.addAttribute("code", 0);
            model.addAttribute("total", majorList.size());
            model.addAttribute("majorList", majorList);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get majorList failure, try again later");
            model.addAttribute("code", -38024);
        }
        return "/common-skill-service/major/json/majorListJson";
    }


    /**
     * 根据学校名返回学校的类型
     *
     * @param schoolName 学校名称
     */
    private Integer getSchoolType(String schoolName) throws ServiceException, ServiceDaoException {
        if (StringUtils.isNotEmpty(schoolName)) {
            Long schoolId = schoolService.getSchoolIdByName(schoolName);
            if (DataUtils.isNotNullOrEmpty(schoolId)) {
                School school = schoolService.getObjectById(schoolId);
                return school.getSchoolType();
            } else {
                return School.SCHOOL_TYPE_NORMAL;
            }
        }
        return -1;
    }

}

