package com.ptteng.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.common.skill.model.*;
import com.ptteng.common.skill.service.*;
import com.ptteng.util.DynamicUtil;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Array;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.ptteng.common.skill.model.User.*;

/**
 * 课堂小组管理
 *
 * @Author ruanshaonan
 * @Description
 * @Date 9:04 2018/6/27
 */
@Controller
public class ClassTeamController {
    private static final Log log = LogFactory.getLog(ClassTeamController.class);

    @Autowired
    private ClassTeamService classTeamService;

    @Autowired
    private ClassTimetableService classTimetableService;

    @Autowired
    private UserService userService;

    @Autowired
    private OccupationService occupationService;

    @Autowired
    private BranchInstituteService branchInstituteService;

    @Autowired
    private MessageService messageService;

    @Autowired
    private CookieUtil cookieUtil;


    /**
     * 新增课题小组
     *
     * @Author ruanshaonan
     * @Description
     * @Date 9:07 2018/6/27
     * @Param oid：职业id
     * @Param groupName：小组名称
     * @Param liveNumber：直播群号
     */
    @RequestMapping(value = "/a/u/class/team", method = RequestMethod.POST)
    public String addClassTeam(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                               Long oid, String groupName, String liveNumber) throws Exception {

        log.info("addClassTeam= oid" + oid + "=groupName=" + groupName + "=liveNumber=" + liveNumber);

        if (DataUtils.isNullOrEmpty(oid)
                || DataUtils.isNullOrEmpty(groupName)
                || DataUtils.isNullOrEmpty(liveNumber)) {
            model.addAttribute("code", -1001);
            return "/common/success";
        }
        try {
            //判断该职业下是否已有该组名
            Map<String, Object> sqlMap = DynamicUtil.getClassTeamByOidAndName(oid, groupName);
            log.info("sqlMap=" + sqlMap);
            List<Long> ids = classTeamService.getIdsByDynamicCondition(ClassTeam.class, sqlMap, 0, Integer.MAX_VALUE);
            if (CollectionUtils.isNotEmpty(ids)) {
                model.addAttribute("code", -70004);
                return "/common/success";
            }

            ClassTeam classTeam = new ClassTeam();
            classTeam.setOid(oid);
            classTeam.setGroupName(groupName);
            classTeam.setLiveNumber(liveNumber);
            classTeam.setCreateBy(Long.valueOf(cookieUtil.getKeyIdentity(request,
                    com.qding.common.util.http.cookie.CookieUtil.USER_ID)));
            Long id = classTeamService.insert(classTeam);
            model.addAttribute("code", 0);
            model.addAttribute("id", id);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("addClassTeam error ");
            model.addAttribute("code", -100000);
        }
        return "/data/json";
    }

    /**
     * 修改课堂小组
     *
     * @Author ruanshaonan
     * @Description
     * @Date 10:02 2018/6/27
     * @Param id：小组id
     * @Param oid：职业id
     * @Param groupName：小组名称
     * @Param liveNumber：小组直播群号
     */
    @RequestMapping(value = "/a/u/class/team/{id}", method = RequestMethod.PUT)
    public String updateClassTeam(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                  @PathVariable Long id, Long oid, String groupName, String liveNumber)
            throws Exception {

        log.info("updateClassTeam = id=" + id + "=oid=" + oid + "=groupName=" + groupName + "=liveNumber=" + liveNumber);

        try {
            //获取小组信息
            ClassTeam classTeam = classTeamService.getObjectById(id);
            log.info("classTeam=" + classTeam.getId());
            if (DataUtils.isNullOrEmpty(classTeam)) {
                model.addAttribute("code", -70003);
                return "/common/success";
            }
            if (DataUtils.isNullOrEmpty(oid)
                    || DataUtils.isNullOrEmpty(groupName)
                    || DataUtils.isNullOrEmpty(liveNumber)) {
                model.addAttribute("code", -1001);
                return "/common/success";
            }

            //判断该职业下是否已有该组名
            Map<String, Object> sqlMap = DynamicUtil.getClassTeamByOidAndName(oid, groupName);
            log.info("sqlMap=" + sqlMap);
            List<Long> ids = classTeamService.getIdsByDynamicCondition(ClassTeam.class, sqlMap, 0, Integer.MAX_VALUE);
            log.info("ids==" + ids.size());
            if (CollectionUtils.isNotEmpty(ids)) {

                if (!ids.contains(id)){
                    model.addAttribute("code", -70004);
                    return "/common/success";
                }
            }

            classTeam.setGroupName(groupName);
            classTeam.setOid(oid);
            classTeam.setLiveNumber(liveNumber);
            classTeamService.update(classTeam);
            log.info("update classTeam success=");
            model.addAttribute("code", 0);
        } catch (Exception e) {
            log.info("updateClassTeam error=" + e.getMessage());
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }


    /**
     * 对小组成员进行操作  新增、修改身份、删除
     *
     * @Author ruanshaonan
     * @Description
     * @Date 16:05 2018/6/28
     * @Param id：小组id
     * @Param uid：用户id
     * @Param behavior：操作：1：加入小组  2：删除  3：任命为组长
     */
    @RequestMapping(value = "/a/u/class/team/user/{id}", method = RequestMethod.PUT)
    public String operateTeamUser(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                  @PathVariable Long id, Long uid, Integer behavior)
            throws Exception {

        log.info("operateTeamUser = id=" + id + "=uid=" + uid + "=behavior=" + behavior);

        try {
            ClassTeam classTeam = classTeamService.getObjectById(id);
            log.info("get classTeam success==" + classTeam.getId());
            if (DataUtils.isNullOrEmpty(classTeam)) {
                model.addAttribute("code", -70003);
                return "/common/success";
            }

            User user = userService.getObjectById(uid);
            log.info("get user success==" + uid);
            if (DataUtils.isNullOrEmpty(user)) {
                model.addAttribute("code", -5003);
                return "/common/success";
            }

            switch (behavior) {
                //加入小组  先判断是否加入了其他小组
                case ClassTeam.JOIN:
                    if (user.getGroupId().longValue() != User.GROUP_DEFAULT_ID) {
                        if (user.getGroupId().intValue() != id) {
                            model.addAttribute("code", -70005);
                        } else {
                            model.addAttribute("code", -70018);
                        }
                        return "/common/success";
                    }

                    //如果用户身份不是 40、内门弟子  50、首席弟子  则不能加入课堂小组
                    if (user.getIdentity().longValue() != User.IDENTIFY_40
                            && user.getIdentity().longValue() != User.IDENTIFY_50 ) {
                        model.addAttribute("code", -70011);
                        return "/common/success";
                    }
                    //没有班级（职业）不能加入
                    if (user.getOid() == -1 ){
                        model.addAttribute("code", -70011);
                        return "/common/success";
                    }
                    user.setGroupId(id);
                    //加入小组时，默认身份为普通组员
                    user.setIdentityInclass(User.IDENTITY_NORMAL_MEMBER);
                    user.setJoinTime(System.currentTimeMillis());
                    userService.update(user);

                    //给该组员发消息
                    log.info("send message start====");
                    sendMsgToUser(ClassTeam.JOIN, user, classTeam);
                    break;
                //删除
                case ClassTeam.DELETE:
                    Map<String, Object> sqlMap = DynamicUtil.getMemberByGroupId(id);
                    List<Long> uids = userService.getIdsByDynamicCondition(User.class, sqlMap, 0, Integer.MAX_VALUE);
                    if (CollectionUtils.isEmpty(uids)) {
                        log.info("uids is empty==");
                        model.addAttribute("code", -70008);
                        return "/common/success";
                    }
                    log.info("uids==" + uids.size());
                    //判断用户是否在当前小组里
                    if (!uids.contains(uid)) {
                        log.info("uids not contain uid=");
                        model.addAttribute("code", -70009);
                        return "/common/success";
                    }

                    //当日0点的时间戳
                    long currentTimeZero = System.currentTimeMillis() / (1000 * 3600 * 24) * (1000 * 3600 * 24)
                            - TimeZone.getDefault().getRawOffset();
                    //当日0点的时间戳+7天
                    long currentTimeZeroAdd7Day = currentTimeZero + 24 * 60 * 60 * 1000 * 7 - 1;
                    User speaker = userService.getObjectById(uid);
                    //判断7天以内是否有排课，如果有，则要先删除排课
                    Map<String, Object> in7DayClassTimetableListParam = DynamicUtil.getClassTimetableParam(null, null, null, id,
                            currentTimeZero, currentTimeZeroAdd7Day, speaker.getNick(), null, false);
                    log.info("in7DayClassTimetableListParam=" + in7DayClassTimetableListParam);
                    List<Long> in7DayClassArrangeIds = classTimetableService.getIdsByDynamicCondition(ClassTimetable.class,
                            in7DayClassTimetableListParam, 0, Integer.MAX_VALUE);
                    log.info("in7DayClassArrangeIds==" + in7DayClassArrangeIds.size());
                    if (CollectionUtils.isNotEmpty(in7DayClassArrangeIds)) {
                        model.addAttribute("code", -70022);
                        return "/common/success";
                    }

                    //判断7天以外是否有排课，如果有，则要先修改排课计划
                    Map<String, Object> out7DayClassTimetableListParam = DynamicUtil.getClassTimetableParam(null, null, null, id,
                            currentTimeZeroAdd7Day, null, speaker.getNick(), null, false);
                    log.info("out7DayClassTimetableListParam=" + out7DayClassTimetableListParam);
                    List<Long> out7DayClassArrangeIds = classTimetableService.getIdsByDynamicCondition(ClassTimetable.class,
                            out7DayClassTimetableListParam, 0, Integer.MAX_VALUE);
                    log.info("out7DayClassArrangeIds==" + out7DayClassArrangeIds.size());
                    if (CollectionUtils.isNotEmpty(out7DayClassArrangeIds)) {
                        model.addAttribute("code", -70023);
                        return "/common/success";
                    }

                    user.setGroupId(GROUP_DEFAULT_ID);
                    user.setJoinTime(GROUP_DEFAULT_JOINTIME);
                    user.setIdentityInclass(GROUP_DEFAULT_IDENTITY);
                    userService.update(user);

                    //给该组员发消息
                    sendMsgToUser(ClassTeam.DELETE, user, classTeam);
                    break;
                //任命为组长  将原来组长改为普通组员身份
                case ClassTeam.LEADER:
                    //原来就是组长
                    if (user.getIdentityInclass().intValue() == User.IDENTITY_LEADER) {
                        model.addAttribute("code", -70007);
                        return "/common/success";
                    }
                    //获取小组组员信息
                    Map<String, Object> groupUsersqlMap = DynamicUtil.getMemberByGroupId(id);
                    List<Long> groupUids = userService.getIdsByDynamicCondition(User.class, groupUsersqlMap, 0, Integer.MAX_VALUE);
                    //如果组员信息为空，则不能任命为组长
                    if (CollectionUtils.isEmpty(groupUids)) {
                        log.info("groupUids is empty==");
                        model.addAttribute("code", -70008);
                        return "/common/success";
                    }
                    log.info("groupUids==" + groupUids.size());
                    //如果该成员不在小组内，则不能任命为组长
                    if (!groupUids.contains(uid)) {
                        log.info("uids not contain uid=");
                        model.addAttribute("code", -70009);
                        return "/common/success";
                    }
                    List<User> groupUsers = userService.getObjectsByIds(groupUids);
                    log.info("groupUsers=" + groupUsers.size());
                    User leader = null;
                    //将组长身份改为普通组员
                    for (User tempUser : groupUsers) {
                        if (tempUser.getIdentityInclass().intValue() == User.IDENTITY_LEADER) {
                            leader = tempUser;
                            leader.setIdentityInclass(User.IDENTITY_NORMAL_MEMBER);
                            break;
                        }
                    }
                    //将该成员身份改为组长
                    user.setIdentityInclass(User.IDENTITY_LEADER);
                    List<User> tempList = new ArrayList<>();
                    if (leader != null) {
                        tempList.add(leader);
                    }
                    tempList.add(user);
                    userService.updateList(tempList);
                    log.info("change leader success");
                    break;
            }
            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.info("updateClassTeam error=" + t.getMessage());
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }

    //加入、退出 小组后，给该组员发消息
    private void sendMsgToUser(int join, User user, ClassTeam classTeam) throws ServiceException, ServiceDaoException, IOException {
        Message message = new Message();
        message.setReceiveId(user.getId());
        message.setType(Message.CLASS_TYPE);
        message.setUnread(Message.Status_Unread);
        message.setStatus(Message.STATUS_UNDELETE);
        message.setTemplate(Message.TEMPLATE_5);
        message.setCreateAt(System.currentTimeMillis());
        message.setUpdateAt(System.currentTimeMillis());
        message.setSendId(0L);
        message.setRelationId(classTeam.getId());
        log.info("set message already");
        //获取职业名称
        Occupation occupation = occupationService.getObjectById(classTeam.getOid());
        log.info("set message already");
        String oname = occupation.getName();
        log.info(oname);
        List<Map> list = new ArrayList<>();
        log.info("set message already");
        Map<String, Object> map = new HashMap<>();
        String content;
        switch (join) {
            case ClassTeam.JOIN:
                content = "您已加入小课堂" + oname + " " + classTeam.getGroupName() + ",点击XXX课表";
                map.put("content", content);
                Map<String, Object> map1 = new HashMap<>();
                map1.put("childContent", "查看");
                map1.put("skip", true);
                map1.put("teamId", classTeam.getGroupName());
                map1.put("oid", occupation.getId());

                list.add(map);
                list.add(map1);
                break;
            case ClassTeam.DELETE:
                content = "您已经从小课堂" + oname + " " + classTeam.getGroupName() + "移除";
                map.put("content", content);

                list.add(map);
                break;
        }
        //把list转成json格式的String字符串
        ObjectMapper json = new ObjectMapper();
        String contentStr = json.writeValueAsString(list);
        message.setContent(contentStr);
        log.info(contentStr);
        messageService.insert(message);
        log.info("insert success==" + message);

        //更新用户的未读消息数量和最后一条消息类型
        user.setIsLook(user.getIsLook() + 1);
        user.setLatestMsg(Message.CLASS_TYPE);
        userService.update(user);
    }


    /**
     * 删除课堂小组
     *
     * @Author ruanshaonan
     * @Description
     * @Date 10:08 2018/6/27
     * @Param id：小组id
     */
    @RequestMapping(value = "/a/u/class/team/{id}", method = RequestMethod.DELETE)
    public String deleteClassTeam(HttpServletRequest request,
                                  HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("deleteClassTeam : id= " + id);
        try {
            ClassTeam classTeam = classTeamService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(classTeam)) {
                model.addAttribute("code", -70003);
            }
            log.info("get classTeam data is " + classTeam.getId());
            if (DataUtils.isNullOrEmpty(classTeam)) {
                model.addAttribute("code", -70003);
                return "/common/success";
            }

            //如果小组从当前时间往后有排课，则不能删除
            Map<String, Object> classTimetableIdsSql = DynamicUtil.getTimeTables(id, null, System.currentTimeMillis(), null);
            log.info("classTimetableIdsSql==" + classTimetableIdsSql);
            List<Long> classTimetableIds = classTimetableService.getIdsByDynamicCondition(ClassTimetable.class, classTimetableIdsSql, 0, Integer.MAX_VALUE);
            log.info("classTimetableIds==" + classTimetableIds.size());
            if (CollectionUtils.isNotEmpty(classTimetableIds)) {
                log.info("team has class timetable ,can not delete=");
                model.addAttribute("code", -70020);
                return "/common/success";
            }

            //如果小组成员不为空，则不能删除
            Map<String, Object> groupUserSqlMap = DynamicUtil.getMemberByGroupId(id);
            List<Long> groupUserIds = userService.getIdsByDynamicCondition(User.class, groupUserSqlMap, 0, Integer.MAX_VALUE);
            log.info("groupUserIds=" + groupUserIds.size());
            if (CollectionUtils.isNotEmpty(groupUserIds)) {
                log.info("team has member ,can not delete=");
                model.addAttribute("code", -70021);
                return "/common/success";
            }

            //删除课堂小组
            classTeamService.delete(id);
            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get classTeam error,id is  " + id);
            model.addAttribute("code", -100000);
        }
        return "/common/success";
    }

    /**
     * 获取小组列表
     *
     * @return
     * @Author ruanshaonan
     * @Description
     * @Date 10:16 2018/6/27
     * @Param oid：职业id
     * @Param groupId：小组id
     * @Param groupLeaderName：小组长姓名
     */
    @RequestMapping(value = "/a/u/class/team/list", method = RequestMethod.GET)
    public String getClassTeamList(HttpServletRequest request, HttpServletResponse response, ModelMap model,
                                   Long oid, String teamName, String leaderName, Integer page, Integer size)
            throws Exception {

        log.info("getClassTeamList= oid=" + oid + "=teamName=" + teamName + "=leaderName=" + leaderName
                + "=page=" + page + "=size=" + size);
        if (page == null) {
            page = 1;
        }
        if (size == null) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }
        int total = 0;

        try {
            List<Integer> groupUserCounts = new ArrayList<>();
            List<String> leaderNames = new ArrayList<>();

            Map<String, Object> sqlMap = DynamicUtil.getClassTeamByOidAndNameAndLeaderName(oid, teamName, leaderName);
            log.info("sqlMap=" + sqlMap);
            //小组id
            List<Long> ids = classTeamService.getIdsByDynamicCondition(ClassTeam.class, sqlMap, start, size);
            log.info("ids=" + ids.size());
            if (CollectionUtils.isEmpty(ids)) {
                model.addAttribute("code", -70010);
                return "/common/success";
            }

            Map<String, Object> sqlMap1 = DynamicUtil.getClassTeamUser();
            log.info("sqlMap1==" + sqlMap1);
            //所有加入小组的用户
            List<Long> uids = userService.getIdsByDynamicCondition(User.class, sqlMap1, 0, Integer.MAX_VALUE);
            log.info("uids=" + uids.size());

            if (CollectionUtils.isNotEmpty(uids)) {
                groupUserCounts = new ArrayList<>();
                leaderNames = new ArrayList<>();
                Map<Long, String> leaderNamesMap = new HashMap<>();

                List<User> groupUsers = userService.getObjectsByIds(uids);
                log.info("get user success== groupUsers==" + groupUsers.size());

                //统计各个小组的人数和组长姓名
                for (Long teamId : ids) {
                    log.info(teamId);
                    leaderNamesMap.put(teamId, "");
                    int i = 0;
                    for (User groupUser : groupUsers) {
                        //这里直接用包装类型对比的时候，返回的是不相等；用值对比的时候，值可能是null；所以转成字符串对比
                        if (groupUser.getGroupId().intValue() == teamId) {
                            i++;
                            if (groupUser.getIdentityInclass().intValue() == User.IDENTITY_LEADER) {
                                leaderNamesMap.put(teamId, groupUser.getNick());
                            }
                        }
                    }
                    groupUserCounts.add(i);
                }
                log.info("leaderNamesMap==" + leaderNamesMap.values().size());
                model.addAttribute("leaderNamesMap", leaderNamesMap);
                log.info("leaderNamesMap==" + leaderNamesMap);
                for (String value : leaderNamesMap.values()) {
                    leaderNames.add(value);
                }

                log.info("groupUserCounts=" + groupUserCounts);
                log.info("leaderNames=" + leaderNames);
            }
            else{
                groupUserCounts = new ArrayList<>();
                leaderNames = new ArrayList<>();
                for (Long teamId : ids){
                    groupUserCounts.add(0);
                    leaderNames.add("");
                }
            }

            model.addAttribute("groupUserCounts", groupUserCounts);
            model.addAttribute("leaderNames", leaderNames);
            List<ClassTeam> classTeams = classTeamService.getObjectsByIds(ids);
            model.addAttribute("classTeamList", classTeams);


            if (CollectionUtils.isNotEmpty(classTeams)) {
                List<Long> oids = new ArrayList<>();
                for (ClassTeam classTeam : classTeams) {
                    oids.add(classTeam.getOid());
                }
                List<Occupation> occupationList = occupationService.getObjectsByIds(oids);
                model.addAttribute("occupationList", occupationList);
                log.info("occupationList====" + occupationList);
            }

            total = classTeamService.getIdsByDynamicCondition(ClassTeam.class, sqlMap, 0, Integer.MAX_VALUE).size();
            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);
            log.info("classTeamList====" + classTeams);
            log.info("leaderNames====" + leaderNames);
            log.info("groupUserCounts====" + groupUserCounts);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage(), t);
            log.error("getClassTeamList error=" + t.getMessage());
            model.addAttribute("code", -100000);
            return "/common/success";
        }
        return "/common-skill-service/classTeam/json/classTeamListJson";
    }


    /**
     * 获取小组成员列表
     *
     * @Author ruanshaonan
     * @Description
     * @Date 12:20 2018/6/27
     * @Param id：小组id
     */
    @RequestMapping(value = "/a/u/class/team/users/{id}", method = RequestMethod.GET)
    public String getGroupUsers(HttpServletRequest request,
                                HttpServletResponse response, ModelMap model, @PathVariable("id") Long id)
            throws Exception {

        log.info("getGroupUsers==id==" + id);

        List<User> userList = new ArrayList<>();
        try {
            Map<String, Object> sqlMap = DynamicUtil.getMemberByGroupId(id);
            List<Long> groupUserIds = userService.getIdsByDynamicCondition(User.class, sqlMap, 0, Integer.MAX_VALUE);
            log.info("groupUserIds=" + groupUserIds.size());
            if (CollectionUtils.isEmpty(groupUserIds)) {
                model.addAttribute("code", -70008);
                return "/common/success";
            }
            userList = userService.getObjectsByIds(groupUserIds);
            log.info("get group user success==userList=" + userList.size());
            List<Long> branchIds = new ArrayList<>();
            if (CollectionUtils.isEmpty(userList)) {
                model.addAttribute("code", -70008);
                return "/common/success";
            }
            log.info("userList====" + userList);
            for (User user : userList) {
                log.info("====" + user);
                branchIds.add(user.getBranch().longValue());
            }
            log.info("branchIds==" + branchIds.size());
            if (CollectionUtils.isNotEmpty(branchIds)) {
                List<BranchInstitute> branchList = branchInstituteService.getObjectsByIds(branchIds);
                log.info("get branch success==" + branchList.size());
                model.addAttribute("branchList", branchList);
            }
        } catch (Throwable throwable) {
            log.info("get group user fail=" + id + "=msg=" + throwable.getMessage());
            model.addAttribute("code", -100000);
            return "/common/success";
        }
        model.addAttribute("code", 0);
        model.addAttribute("userList", userList);
        return "/common-skill-service/classTeam/json/classTeamUserListJson";
    }

}

