package com.ptteng.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.common.skill.model.*;
import com.ptteng.common.skill.service.*;
import com.ptteng.util.DataValidatorUtil;
import com.ptteng.util.DynamicUtil;
import com.qding.common.util.DataUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Controller
public class RecordController {

    public static final Log log = LogFactory.getLog(RecordController.class);

    @Autowired
    private UserService userService;

    @Autowired
    private RecordService recordService;

    @Autowired
    private ManagerService managerService;

    @Autowired
    private RecordLogService recordLogService;

    @Autowired
    private BranchInstituteService branchInstituteService;

    @Autowired
    private OccupationService occupationService;

    @Autowired
    private SchoolService schoolService;

    @Autowired
    private MessageService messageService;

    /**
     * @Description: 对应档案模块下面的用户列表，实际是以user表为基础进行动态查询
     * @param:
     */
    @RequestMapping(value = "/a/u/record/search", method = RequestMethod.GET)
    public String getRecordList(ModelMap model, Long uid, Long branchId, String name, String school, Integer grade,
                                Integer schoolType, Long minStartAt, Long maxStartAt, Long minGraduateAt, Long maxGraduateAt,
                                String major, String place, Integer status, Integer page, Integer size) {

        log.info("GET /a/u/record/search: by params: uid = " + uid + ", branchId = " + branchId + ", name = " + name
                + ", school = " + school + ", grade = " + grade + ", schoolType = " + schoolType + ", minStartAt = "
                + ", maxStartAt = " + ", minGraduateAt = " + ", maxGraduateAt = " + ". major = " + major + ". place = "
                + ", status = " + status + ", page = " + page + ", size = " + size);

        if (page == null) {
            page = 1;
        }
        if (size == null) {
            size = 10;
        }
        int start = (page - 1) * size;
        if (start < 0) {
            start = 0;
        }

        try {
            Map<String, Object> params = DynamicUtil.getUserListParam(uid, branchId, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, status, false);
            log.info("search userIds by params: " + params);

            List<Long> userIds = userService.getIdsByDynamicCondition(User.class, params, start, size);
            log.info("get userIds size: " + userIds.size());
            if (CollectionUtils.isEmpty(userIds)) {
                model.addAttribute("code", -39002);
                return "/common/failure";
            }

            List<User> userList = userService.getObjectsByIds(userIds);
            log.info("get userList size: " + userList.size());

            Map<String, Object> params2 = DynamicUtil.getUserListParam(uid, branchId, name, minStartAt, maxStartAt, minGraduateAt,
                    maxGraduateAt, school, grade, schoolType, major, place, status, true);
            log.info("count userIds by params:" + params2);

            List<Long> countUserIds = userService.getIdsByDynamicCondition(User.class, params, 0, Integer.MAX_VALUE);
            log.info("get countUserIds: " + countUserIds.get(0));

            int total = 0;
            if (!org.springframework.util.CollectionUtils.isEmpty(countUserIds)) {
                total = countUserIds.get(0).intValue();
            }


            Map<String, Object> params3 = DynamicUtil.getRecordListByUserIdsParam(userIds, false);
            log.info("search recordIds by params: " + params3);

            List<Long> recordIds = recordService.getIdsByDynamicCondition(Record.class, params3, start, size);
            log.info("get recordIds size: " + recordIds.size() + ", detail: " + recordIds);
            if (CollectionUtils.isEmpty(recordIds)) {
                model.addAttribute("code", -39002);
                return "/common/failure";
            }

            List<Record> recordList = recordService.getObjectsByIds(recordIds);
            log.info("get recordList size: " + recordList.size());

            Map<Long, Record> recordMap = new HashMap<>();
            for (Record record : recordList) {
                recordMap.put(record.getUid(), record);
            }

            List<Long> occupationIds = new ArrayList<>();
            List<Long> branchIds = new ArrayList<>();
            for (User user : userList) {
                if (DataUtils.isNotNullOrEmpty(user.getOid())) {
                    occupationIds.add(user.getOid());
                }
                if (DataUtils.isNotNullOrEmpty(user.getBranch())) {
                    branchIds.add(Long.valueOf(user.getBranch()));
                }
            }

            List<Occupation> occupationList = occupationService.getObjectsByIds(occupationIds);
            List<BranchInstitute> branchInstituteList = branchInstituteService.getObjectsByIds(branchIds);

            Map<Long, Occupation> occupationMap = new HashMap<>();
            Map<Long, BranchInstitute> branchInstituteMap = new HashMap<>();
            for (Occupation o : occupationList) {
                occupationMap.put(o.getId(), o);
            }
            for (BranchInstitute b : branchInstituteList) {
                branchInstituteMap.put(b.getId(), b);
            }

            model.addAttribute("code", 0);
            model.addAttribute("page", page);
            model.addAttribute("size", size);
            model.addAttribute("total", total);
            model.addAttribute("userList", userList);
            model.addAttribute("recordMap", recordMap);
            model.addAttribute("occupationMap", occupationMap);
            model.addAttribute("branchMap", branchInstituteMap);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("search record failure, try again later");
            model.addAttribute("code", -39003);
        }
        return "/common-skill-service/record/json/userListAndRecordMapJson";
    }

    /**
     * @Description: 查看档案详情
     * @param: recordId 档案ID
     */
    @RequestMapping(value = "/a/u/record/{recordId}", method = RequestMethod.GET)
    public String getRecordDetail(ModelMap model, @PathVariable Long recordId) {

        log.info("GET /a/u/record/{recordId}: by recordId: " + recordId);

        try {
            Record record = recordService.getObjectById(recordId);
            log.info("get record: " + record);
            if (DataUtils.isNullOrEmpty(record)) {
                log.info("the record is not exist, recordId = " + recordId);
                model.addAttribute("code", -39004);
                return "/common/failure";
            }

            User user = userService.getObjectById(record.getUid());
            log.info("get user: " + user);

            Occupation occupation = occupationService.getObjectById(user.getOid());
            log.info("get occupation: " + occupation.getName());

            BranchInstitute branchInstitute = branchInstituteService.getObjectById(Long.valueOf(user.getBranch()));
            log.info("get branchInstitute: " + branchInstitute.getName());

            model.addAttribute("code", 0);
            model.addAttribute("record", record);
            model.addAttribute("user", user);
            model.addAttribute("occupation", occupation);
            model.addAttribute("branch", branchInstitute);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("get record detail failure, try again later");
            model.addAttribute("code", -39005);
        }
        return "/common-skill-service/record/json/recordDetailJson";
    }

    /**
     * @param record 要修改的档案信息
     * @Description: 编辑档案
     * @param: recordId 档案ID
     */
    @RequestMapping(value = "/a/u/record/{recordId}", method = RequestMethod.PUT)
    public String modifyRecord(HttpServletRequest request, ModelMap model, @PathVariable Long recordId, Record record) {

        log.info("PUT /a/u/record/{recordId}: modify record files to: " + record);

        if (!DataValidatorUtil.checkRecord(record)) {
            log.info("params error");
            model.addAttribute("code", -1001);
            return "/common/failure";
        }

        try {
            Record realRecord = recordService.getObjectById(recordId);
            log.info("get record before modify: " + realRecord);
            if (DataUtils.isNullOrEmpty(realRecord)) {
                log.info("the record is not exist, recordId = " + recordId);
                model.addAttribute("code", -39004);
                return "/common/failure";
            }

            if (!realRecord.getUid().equals(record.getUid())) {
                log.info("record's uid do not support modify: " + realRecord.getUid() + " to " + record.getUid());
                model.addAttribute("code", -39006);
                return "/common/failure";
            }

            //更新record, 档案的状态status修改为申请中
            Manager manager = (Manager) request.getAttribute("manager");
            log.info("get manager: " + manager);

            realRecord.setAvatar(record.getAvatar());
            realRecord.setName(record.getName());
            realRecord.setSex(record.getSex());
            realRecord.setIdentityNum(record.getIdentityNum());
            realRecord.setBornAt(record.getBornAt());
            realRecord.setProvince(record.getProvince());
            realRecord.setCity(record.getCity());
            realRecord.setCounty(record.getCounty());
            realRecord.setEducatedStartAt(record.getEducatedStartAt());
            realRecord.setEducatedEndAt(record.getEducatedEndAt());
            realRecord.setGrade(record.getGrade());
            realRecord.setSchool(record.getSchool());
            realRecord.setSchoolType(getSchoolType(record.getSchool()));
            realRecord.setMajor(record.getMajor());
            realRecord.setIsWork(record.getIsWork());
            realRecord.setWorkExp(record.getWorkExp());
            realRecord.setContactName(record.getContactName());
            realRecord.setContactRelation(record.getContactRelation());
            realRecord.setContactMobile(record.getContactMobile());
            realRecord.setCardNum(record.getCardNum());
            realRecord.setBank(record.getBank());
            realRecord.setSource(record.getSource());
            realRecord.setIdentityFrontUrl(record.getIdentityFrontUrl());
            realRecord.setIdentityReverseUrl(record.getIdentityReverseUrl());
            realRecord.setIsLoan(record.getIsLoan());
            realRecord.setStatus(Record.STATUS_APPLYING); //状态为申请中
            realRecord.setHouseholdRegisterUrl(record.getHouseholdRegisterUrl());
            realRecord.setUpdateBy(manager.getId());
            recordService.update(realRecord);

            //新增审核记录
            RecordLog recordLog = new RecordLog();
            recordLog.setUid(realRecord.getUid());
            recordLog.setOperate(RecordLog.OPERATE_EDIT);
            recordLog.setCreateBy(manager.getId());
            recordLog.setUpdateBy(manager.getId());
            Long recordLogId = recordLogService.insert(recordLog);
            log.info("insert new recordLog: recordLogId = " + recordLogId + ", operate = \"edit\"");

            model.addAttribute("code", 0);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("modify record failure, try again later");
            model.addAttribute("code", -39007);
        }
        return "/common/success";
    }

    /**
     * @Description: 审核档案
     * @param recordId     档案ID
     * @param operate      审核操作（1-存档，2-驳回）
     * @param refuseReason 驳回理由
     * @return
     */
    @RequestMapping(value = "/a/u/record/check/{recordId}/{operate}", method = RequestMethod.POST)
    public String checkRecord(HttpServletRequest request, ModelMap model, @PathVariable Long recordId, @PathVariable Integer operate, String refuseReason) {

        log.info("POST /a/u/record/check/{recordId}/{operate}: recordId = " + recordId + ", operate = " + operate + ", refuseReason =" + refuseReason);

        if (operate > RecordLog.OPERATE_REFUSE || operate < RecordLog.OPERATE_SAVE) {
            log.info("not supported this operate: " + operate);
            model.addAttribute("code", -39008);
            return "/common/failure";
        }

        if (operate.equals(RecordLog.OPERATE_REFUSE) && StringUtils.isEmpty(refuseReason)) {
            log.info("operate equals \"refuse\", refuseReason cant not be empty");
            model.addAttribute("code", -39009);
            return "/common/failure";
        }

        if (operate.equals(RecordLog.OPERATE_SAVE) && StringUtils.isNotEmpty(refuseReason)) {
            log.info("operate equals \"save\", don't need refuseReason");
            model.addAttribute("code", -39010);
            return "/common/failure";
        }

        try {
            Record record = recordService.getObjectById(recordId);
            log.info("get record by recordId: " + record);
            if (DataUtils.isNullOrEmpty(record)) {
                log.info("the record is not exist, recordId = " + recordId);
                model.addAttribute("code", -39004);
                return "/common/failure";
            }

            Manager manager = (Manager) request.getAttribute("manager");
            log.info("get manager: " + manager);

            //新增审核记录
            RecordLog recordLog = new RecordLog();
            recordLog.setUid(record.getUid());
            recordLog.setOperate(operate);
            recordLog.setRefuseReason(refuseReason);
            recordLog.setCreateBy(manager.getId());
            recordLog.setUpdateBy(manager.getId());
            Long recordLogId = recordLogService.insert(recordLog);
            log.info("insert new recordLog: recordLogId = " + recordLogId);

            User user = userService.getObjectById(record.getUid());
            if (operate.equals(RecordLog.OPERATE_SAVE)) { //审核结果为存档的
                record.setStatus(Record.STATUS_SAVE);
                user.setRecordStatus(Record.STATUS_SAVE);

                //存档后，还要以档案表信息覆盖个人信息
                user.setThumb(record.getAvatar());
                user.setNick(record.getName());
                user.setSex(Long.valueOf(record.getSex()));
                user.setBirthday(record.getBornAt());
                user.setSchool(record.getSchool());
                user.setProvince(record.getProvince());
                user.setCity(record.getCity());

            } else if (operate.equals(RecordLog.OPERATE_REFUSE)) { //审核结果为驳回
                record.setStatus(Record.STATUS_REFUSE);
                user.setRecordStatus(Record.STATUS_REFUSE);

                //还要给用户发送一条类型为“class”的消息，更新未读消息数，最新的消息类型
                Long messageId = sendRefuseMessageToStudent(manager, user, record);
                log.info("send refuse message to student: messageId = " + messageId);

                user.setIsLook(user.getIsLook() + 1);
                user.setLatestMsg(Message.CLASS_TYPE);
            }

            recordService.update(record);
            userService.update(user);

            model.addAttribute("code", 0);
            model.addAttribute("id", recordLogId);

        } catch (Exception e) {
            e.printStackTrace();
            e.getMessage();
            log.info("check record failure, try again later");
            model.addAttribute("code", -39011);
        }
        return "/common/insert";
    }

    /**
     * @return 学校类型（985、211、非重点）
     * @Description: 根据学校名返回学校的类型
     * @param: schoolName 学校名称
     */
    private Integer getSchoolType(String schoolName) throws ServiceException, ServiceDaoException {
        Long schoolId = schoolService.getSchoolIdByName(schoolName);
        if (DataUtils.isNotNullOrEmpty(schoolId)) {
            School school = schoolService.getObjectById(schoolId);
            return school.getSchoolType();
        } else {
            return School.SCHOOL_TYPE_NORMAL;
        }
    }

    /**
     * @param user   接收消息的学员
     * @param record 被驳回的档案
     * @return 新消息的ID
     * @Description: 档案被驳回时，发送一条消息给学员
     * @param: manager 进行审核操作的后台管理员
     */
    private Long sendRefuseMessageToStudent(Manager manager, User user, Record record) throws ServiceException, ServiceDaoException {
        String content = "<span class=\"font-16px system-alert-outline  ng-scope\">你提交的档案表被驳回，请 <a data-mid=\"96745\" ng-click=\"sendAlertMsg(li.id)\" target=\"_blank\" href=\"/school/10799/class\"><span class=\"font-green\">“查看原因”</span> </a> 并重新填写提交</span>";

        Message message = new Message();
        message.setSendId(manager.getId());
        message.setReceiveId(user.getId());
        message.setSendName(manager.getName());
        message.setReceiveName(user.getNick());
        message.setContent(content);
        message.setStatus(Message.STATUS_UNDELETE);
        message.setUnread(Message.Status_Unread);
        message.setType(Message.RECORD_TYPE);
        message.setTemplate(-1);
        message.setRelationId(record.getId());
        message.setCreateBy(manager.getId());
        message.setUpdateBy(manager.getId());

        return messageService.insert(message);
    }

}
