package com.qding.common.util;

import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class FileUploadUtil {
    private static final Logger log = LoggerFactory.getLogger("FileUploadUtil");
    /**
     * 已上传切片初始容量
     */
    private static final int UPLOAD_SLICE_INITIAL_CAPACITY = 128;
    private static final String SAMPLING_MD5_FILE_NAME = "JOY-LEARNING-SAMPLING-FILE-NAME";

    /**
     * 通过切片文件根路径获取已上传切片名称列表
     *
     * @param sliceFileRootFolderPath 切片文件根文件夹路径
     * @return 已上传切片名称列表
     */
    public static List<String> uploadedList(String sliceFileRootFolderPath) {
        // 1.查看上传切片根路径是否存在fileHash匹配的文件夹
        List<String> uploadSliceList = new ArrayList<>(UPLOAD_SLICE_INITIAL_CAPACITY);
        File sliceFileRootFolder = new File(sliceFileRootFolderPath);
        if (sliceFileRootFolder.exists() && sliceFileRootFolder.isDirectory()) {
            log.info("uploadedList->sliceFileRootFolderPath={},get sliceFileRootFolder success",
                    sliceFileRootFolderPath);
            // 如果校验的文件夹存在,返回已上传切片名称列表
            File[] sliceFilePathArray = sliceFileRootFolder.listFiles();

            if (null != sliceFilePathArray) {
                log.info("uploadedList->sliceFileRootFolderPath={},get sliceFilePathArray success,size={}",
                        sliceFileRootFolderPath, sliceFilePathArray.length);

                for (File sliceFile : sliceFilePathArray) {
                    if (sliceFile.isFile()) {
                        log.info("uploadedList->sliceFileRootFolderPath={},slice:[{}] is file",
                                sliceFileRootFolderPath, sliceFile.getName());
                        uploadSliceList.add(sliceFile.getName());
                    } else {
                        log.info("uploadedList->sliceFileRootFolderPath={}", sliceFileRootFolderPath);
                    }
                }

            } else {
                log.info("uploadedList->sliceFileRootFolderPath={},get sliceFilePathArray is null",
                        sliceFileRootFolderPath);
            }

        } else {
            log.info("uploadedList->sliceFileRootFolderPath={},get sliceFileRootFolder fail",
                    sliceFileRootFolderPath);
        }
        return uploadSliceList;
    }

    /**
     * 检查文件夹路径是否存在，如果不存在则创建
     *
     * @param fileFolderPath 文件夹路径
     * @return true=存在(包括不存在时创建成功)，false=不存在(创建文件夹失败)
     */
    private static boolean isFolderExists(String fileFolderPath) {
        File fileRootFolder = new File(fileFolderPath);
        if (fileRootFolder.exists() && fileRootFolder.isDirectory()) {
            log.info("isFolderExists->fileFolderPath={},fileRootFolder[exists=true,isDirectory=true]",
                    fileFolderPath);
        } else {
            // 创建文件夹
            boolean hasMkdirs = fileRootFolder.mkdirs();
            if (hasMkdirs) {
                log.info("isFolderExists->fileFolderPath={},[hasMkdirs]create dir success", fileFolderPath);
                return true;
            } else {
                log.info("isFolderExists->fileFolderPath={},[hasMkdirs]create dir fail", fileFolderPath);
                return false;
            }
        }
        return true;
    }

    /**
     * 把输入流读取到字节数组中
     *
     * @param inputStream 输入流
     * @return 输入流字节数组
     */
    private static byte[] readInputStream(InputStream inputStream) throws IOException {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        //创建一个Buffer
        byte[] buffer = new byte[1024];
        //每次读取的字符串长度，如果为-1，代表全部读取完毕
        int len;
        //使用一个输入流从buffer里把数据读取出来
        while ((len = inputStream.read(buffer)) != -1) {
            //用输出流往buffer里写入数据，中间参数代表从哪个位置开始读，len代表读取的长度
            byteArrayOutputStream.write(buffer, 0, len);
        }
        //关闭输入流
        inputStream.close();
        //把outStream里的数据写入内存
        return byteArrayOutputStream.toByteArray();
    }

    /**
     * 保存切片文件
     *
     * @param inputStream           切片文件输入流
     * @param fileMD5RootFolderPath 切片根文件夹路径
     * @param sliceFileMD5Path      切片文件路径
     * @return true=切片保存成功，false=切片保存失败
     */
    public static boolean saveSliceFile(InputStream inputStream,
                                        String fileMD5RootFolderPath,
                                        String sliceFileMD5Path) {

        boolean isFolderExists = isFolderExists(fileMD5RootFolderPath);

        if (isFolderExists) {
            log.info("saveSliceFile->fileMD5RootFolderPath={},folder exists", fileMD5RootFolderPath);
        } else {
            log.error("saveSliceFile->fileMD5RootFolderPath={},folder not exists,folder create fail.",
                    fileMD5RootFolderPath);
            return false;
        }

        File uploadFile = new File(sliceFileMD5Path);
        try {

            byte[] data = readInputStream(inputStream);

            try (FileOutputStream outputStream = new FileOutputStream(uploadFile)) {
                outputStream.write(data);
                outputStream.flush();
            } catch (Exception e) {
                log.error("saveSliceFile->fileMD5RootFolderPath={},uploadFile [outputStream] write data fail",
                        fileMD5RootFolderPath, e);
                e.printStackTrace();
            }

        } catch (IOException e) {
            log.error("saveSliceFile->fileMD5RootFolderPath={},uploadFile [readInputStream] fail",
                    fileMD5RootFolderPath, e);
            e.printStackTrace();
        }

        if (uploadFile.exists()) {
            log.info("saveSliceFile->fileMD5RootFolderPath={},uploadFile[{}] success",
                    fileMD5RootFolderPath, sliceFileMD5Path);
            return true;
        } else {
            log.info("saveSliceFile->fileMD5RootFolderPath={},uploadFile[{}] fail",
                    fileMD5RootFolderPath, sliceFileMD5Path);
            return false;
        }
    }


    public static boolean mergeFiles(String[] sliceFilePaths, String mergeResultPath) {
        log.info("mergeFiles->sliceFilePaths size={},mergeResultPath={}", sliceFilePaths.length, mergeResultPath);
        File[] files = new File[sliceFilePaths.length];
        for (int i = 0; i < sliceFilePaths.length; i++) {
            files[i] = new File(sliceFilePaths[i]);
            if (!files[i].exists() || !files[i].isFile()) {
                return false;
            }
        }

        File mergeResultFile = combineMultipleFiles(mergeResultPath, sliceFilePaths);

        if (null != mergeResultFile && mergeResultFile.length() > 0) {
            log.info("mergeFiles->sliceFilePaths size={},mergeResultPath={},mergeResultFile success",
                    sliceFilePaths.length, mergeResultPath);
        } else {
            log.info("mergeFiles->sliceFilePaths size={},mergeResultPath={},mergeResultFile fail",
                    sliceFilePaths.length, mergeResultPath);
            return false;
        }

        for (int i = 0; i < sliceFilePaths.length; i++) {
            boolean hasDel = files[i].delete();
            if (hasDel) {
                log.info("mergeFiles->files[{}]，name={} del success", i, files[i].getName());
            } else {
                log.info("mergeFiles->files[{}]，name={} del fail", i, files[i].getName());
            }
        }

        return true;
    }

    /**
     * 获取排好序的切片数组
     *
     * @param sliceFileRootFolderPath 切片跟路径
     */
    public static String[] sliceSortArray(String sliceFileRootFolderPath, String fileMD5) {

        String slicePath = sliceFileRootFolderPath + "/" + fileMD5 + "-";
        // 获取所有切片名称list
        List<String> sliceList = uploadedList(sliceFileRootFolderPath);
        if (CollectionUtils.isEmpty(sliceList)) {
            log.info("samplingMD5->sliceFileRootFolderPath={},file does not exist", sliceFileRootFolderPath);
            return null;
        }

        int sliceSize = 0;
        if (CollectionUtils.isNotEmpty(sliceList)) {
            sliceSize = sliceList.size();
        }
        String[] slicePathArray = new String[sliceSize];
        for (int i = 0; i < sliceSize; i++) {
            slicePathArray[i] = slicePath + i;
        }

        return slicePathArray;
    }


    /**
     * 多个文件合并为一个文件
     *
     * @param outputPath    合并文件输出路径
     * @param filePathArray 多个要合并文件的路径数组
     */
    public static File combineMultipleFiles(String outputPath, String[] filePathArray) {
        File[] files = new File[filePathArray.length];
        for (int i = 0; i < filePathArray.length; i++) {
            files[i] = new File(filePathArray[i]);
            if (!files[i].exists() || !files[i].isFile()) {
                return null;
            }
        }

        File outputFile = new File(outputPath);

        try {
            FileChannel mergeOutputFileChannel = new FileOutputStream(outputFile, true).getChannel();
            for (int i = 0; i < filePathArray.length; i++) {
                FileChannel sliceFileChannel = new FileInputStream(files[i]).getChannel();
                mergeOutputFileChannel.transferFrom(sliceFileChannel,
                        mergeOutputFileChannel.size(), sliceFileChannel.size());
                sliceFileChannel.close();
            }
            mergeOutputFileChannel.close();
        } catch (IOException e) {
            log.error("combineMultipleFiles->filePathArray={},combineMultipleFiles append file fail.error={}",
                    filePathArray, e);
            e.printStackTrace();
        }

        if (outputFile.exists() && outputFile.length() > 0) {
            log.info("combineMultipleFiles->outputPath={} merge output file success.", outputPath);
            return outputFile;
        }
        log.info("combineMultipleFiles->outputPath={} merge output file fail.", outputPath);
        return null;
    }


    /**
     * 文件抽样MD5值校验
     *
     * @param sliceFileRootFolderPath 切片根文件夹路径
     * @param fileMD5                 抽样MD5值
     * @return true: 校验成功  false：校验失败
     */
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static boolean samplingMD5(String sliceFileRootFolderPath, String fileMD5) {
        //第一个和最后一个切片全部内容，其他切片的取 首中尾三个地方各2个字节
        //合并后的内容，计算md5

        // 获取所有切片名称数组
        String[] slicePathArray = sliceSortArray(sliceFileRootFolderPath, fileMD5);

        if (slicePathArray == null) {
            return false;
        }

        // 获取第一个切片
        // 创建抽样文件
        int slicePathArrayLength = slicePathArray.length;
        log.info("samplingMD5->fileMD5={},slicePathArray length={}", fileMD5,
                slicePathArrayLength);

        // 计算全量切片内容MD5的切片路径数组
        String[] fullAmountSlicePathArray = new String[1];
        fullAmountSlicePathArray[0] = slicePathArray[0];
        if (slicePathArray.length > 1) {
            fullAmountSlicePathArray =  Arrays.copyOf(fullAmountSlicePathArray,
                    fullAmountSlicePathArray.length + 1);
            fullAmountSlicePathArray[1] = slicePathArray[slicePathArrayLength - 1];
        }

        for (String fullAmountSlicePath : fullAmountSlicePathArray) {
            log.info("samplingMD5->fileMD5={},slicePathArray fullAmountSlicePath={}", fileMD5,
                    fullAmountSlicePath);
        }
        int middleSliceLength = Math.max((slicePathArray.length - 2), 0);
        String[] middleSlicePathArray = new String[middleSliceLength];
        if (slicePathArray.length > 2) {
            middleSlicePathArray   =
                    Arrays.copyOfRange(slicePathArray, 1, slicePathArrayLength - 1);
        }



        if (middleSlicePathArray.length > 0) {
            for (String middleSlicePath : middleSlicePathArray) {
                log.info("samplingMD5->fileMD5={},slicePathArray middleSlicePath={}", fileMD5,
                        middleSlicePath);
            }
        }


        boolean hasFolderExists = isFolderExists(sliceFileRootFolderPath + "/SAMPLING_MD5_FILE_FOLDER");
        log.info("samplingMD5-> slice file root folder path hasFolderExists={}", hasFolderExists);
        File samplingMD5File = new File(sliceFileRootFolderPath + "/SAMPLING_MD5_FILE_FOLDER/" + SAMPLING_MD5_FILE_NAME);

        try {
            // 创建抽样MD5 file channel
            FileChannel samplingMD5FileChannel = new FileOutputStream(samplingMD5File, true).getChannel();
            FileChannel firstSliceFileChannel = new FileInputStream(fullAmountSlicePathArray[0]).getChannel();
            // 把第一个切片拷贝进 samplingMD5FileChannel
            samplingMD5FileChannel.transferFrom(
                    firstSliceFileChannel,
                    samplingMD5FileChannel.size(),
                    firstSliceFileChannel.size());

            if (middleSlicePathArray.length > 0) {
                for (String middleSlicePath : middleSlicePathArray) {
                    FileInputStream in = new FileInputStream(middleSlicePath);
                    byte[] firstBytes = new byte[2];
                    in.read(firstBytes);
                    byte[] middleBytes = new byte[2];
                    // 切片中间2字节偏移量(之前已经读了2字节所以这里要减去)
                    int middleOffset = Math.toIntExact(in.getChannel().size() / 2 - 2);
                    in.skip(middleOffset);
                    in.read(middleBytes);
                    // 切片末尾2字节偏移量(剩余未读字节减去2字节)
                    in.skip(in.available() - 2);
                    byte[] endOffset = new byte[2];
                    in.read(endOffset);
                    samplingMD5FileChannel.write(ByteBuffer.wrap(firstBytes));
                    samplingMD5FileChannel.write(ByteBuffer.wrap(middleBytes));
                    samplingMD5FileChannel.write(ByteBuffer.wrap(endOffset));
                }
            }

            if (slicePathArrayLength > 1) {
                FileChannel lastSliceFileChannel = new FileInputStream(fullAmountSlicePathArray[1]).getChannel();
                samplingMD5FileChannel.transferFrom(
                        lastSliceFileChannel,
                        samplingMD5FileChannel.size(),
                        lastSliceFileChannel.size());
            }

            samplingMD5FileChannel.close();
        } catch (IOException e) {
            log.error("combineMultipleFiles->filePathArray={},combineMultipleFiles append file fail.error={}",
                    middleSlicePathArray, e);
            e.printStackTrace();
        }

        boolean samplingMD5Flag = false;

        try {
            log.info("samplingMD5->fileMD5={},samplingMD5={}", fileMD5,getFileMD5(samplingMD5File));
            String samplingMD5 = getFileMD5(samplingMD5File);
            if (fileMD5.equals(samplingMD5)) {
                log.info("samplingMD5->fileMD5={},samplingMD5 is same", fileMD5);
                samplingMD5Flag = true;
            } else {
                log.info("samplingMD5->fileMD5={},samplingMD5 is different", fileMD5);
            }
        } catch (Exception e) {
            log.error("calculation samplingMD5 error,fileMD5={}", fileMD5, e);
            e.printStackTrace();
        }

        return samplingMD5Flag;
    }

    @SuppressWarnings("UnstableApiUsage")
    public static String getFileMD5(File file) {
        if (!file.isFile()) {
            return null;
        }
        String md5 = null;
        try {
            HashCode hashCode = Files.hash(file, Hashing.md5());
            md5 =  hashCode.toString();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return md5;
    }

    public static void main(String[] args) throws Exception {
        File file = new File("/Users/rick/IdeaProjects/old/Joylearning-apiserver/tmp4upload/伊藤賢治 - メインテーマ「永遠の一瞬」.mp3");
        System.out.println(getFileMD5(file));
        FileInputStream stream = new FileInputStream(file);
    }
}
