/*
 * Copyright 2012-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.client.hypermedia;

import java.util.List;

import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;

/**
 * {@link ServiceInstanceProvider} to work with a {@link DiscoveryClient} to look up a
 * service by name. Picks the first one returned by the configured
 * {@link DiscoveryClient}.
 *
 * @author Oliver Gierke
 */
public class DynamicServiceInstanceProvider implements ServiceInstanceProvider {

	private final DiscoveryClient client;

	private final String serviceName;

	public DynamicServiceInstanceProvider(DiscoveryClient client, String serviceName) {
		this.client = client;
		this.serviceName = serviceName;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see example.customers.integration.ServiceInstanceProvider#getServiceInstance()
	 */
	@Override
	public ServiceInstance getServiceInstance() {

		try {

			List<ServiceInstance> instances = this.client.getInstances(this.serviceName);
			return instances.isEmpty() ? null : instances.get(0);

		}
		catch (RuntimeException o_O) {
			return null;
		}
	}

}
