/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients;

import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.internals.ClusterResourceListeners;
import org.apache.kafka.test.MockClusterResourceListener;
import org.apache.kafka.test.TestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;

public class MetadataTest {
    private long refreshBackoffMs = 100L;
    private long metadataExpireMs = 1000L;
    private Metadata metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true);
    private AtomicReference<Exception> backgroundError = new AtomicReference();

    @After
    public void tearDown() {
        Assert.assertNull((String)("Exception in background thread : " + this.backgroundError.get()), (Object)this.backgroundError.get());
    }

    @Test
    public void testMetadata() throws Exception {
        long time = 0L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        this.metadata.requestUpdate();
        Assert.assertFalse((String)"Still no updated needed due to backoff", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed now that backoff time expired", (this.metadata.timeToNextUpdate(time += this.refreshBackoffMs) == 0L ? 1 : 0) != 0);
        String topic = "my-topic";
        Thread t1 = this.asyncFetch(topic, 500L);
        Thread t2 = this.asyncFetch(topic, 500L);
        Assert.assertTrue((String)"Awaiting update", (boolean)t1.isAlive());
        Assert.assertTrue((String)"Awaiting update", (boolean)t2.isAlive());
        while (t1.isAlive() || t2.isAlive()) {
            if (this.metadata.timeToNextUpdate(time) == 0L) {
                this.metadata.update(TestUtils.singletonCluster(topic, 1), Collections.emptySet(), time);
                time += this.refreshBackoffMs;
            }
            Thread.sleep(1L);
        }
        t1.join();
        t2.join();
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed due to stale metadata.", (this.metadata.timeToNextUpdate(time += this.metadataExpireMs) == 0L ? 1 : 0) != 0);
    }

    @Test
    public void testMetadataAwaitAfterClose() throws InterruptedException {
        long time = 0L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        this.metadata.requestUpdate();
        Assert.assertFalse((String)"Still no updated needed due to backoff", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"Update needed now that backoff time expired", (this.metadata.timeToNextUpdate(time += this.refreshBackoffMs) == 0L ? 1 : 0) != 0);
        String topic = "my-topic";
        this.metadata.close();
        Thread t1 = this.asyncFetch(topic, 500L);
        t1.join();
        Assert.assertTrue((this.backgroundError.get().getClass() == KafkaException.class ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.backgroundError.get().toString().contains("Requested metadata update after close"));
        this.clearBackgroundError();
    }

    @Test(expected=IllegalStateException.class)
    public void testMetadataUpdateAfterClose() {
        this.metadata.close();
        this.metadata.update(Cluster.empty(), Collections.emptySet(), 1000L);
    }

    private static void checkTimeToNextUpdate(long refreshBackoffMs, long metadataExpireMs) {
        long now = 10000L;
        if (metadataExpireMs > now || refreshBackoffMs > now) {
            throw new IllegalArgumentException("metadataExpireMs and refreshBackoffMs must be smaller than 'now'");
        }
        long largerOfBackoffAndExpire = Math.max(refreshBackoffMs, metadataExpireMs);
        Metadata metadata = new Metadata(refreshBackoffMs, metadataExpireMs, true);
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now));
        metadata.update(Cluster.empty(), Collections.emptySet(), now);
        Assert.assertEquals((long)largerOfBackoffAndExpire, (long)metadata.timeToNextUpdate(now));
        metadata.requestUpdate();
        Assert.assertEquals((long)refreshBackoffMs, (long)metadata.timeToNextUpdate(now));
        metadata.update(Cluster.empty(), Collections.emptySet(), now);
        Assert.assertEquals((long)largerOfBackoffAndExpire, (long)metadata.timeToNextUpdate(now));
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now += largerOfBackoffAndExpire));
        Assert.assertEquals((long)0L, (long)metadata.timeToNextUpdate(now + 1L));
    }

    @Test
    public void testTimeToNextUpdate() {
        MetadataTest.checkTimeToNextUpdate(100L, 1000L);
        MetadataTest.checkTimeToNextUpdate(1000L, 100L);
        MetadataTest.checkTimeToNextUpdate(0L, 0L);
        MetadataTest.checkTimeToNextUpdate(0L, 100L);
        MetadataTest.checkTimeToNextUpdate(100L, 0L);
    }

    @Test
    public void testTimeToNextUpdate_RetryBackoff() {
        long now = 10000L;
        this.metadata.failedUpdate(now, null);
        Assert.assertEquals((long)this.refreshBackoffMs, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.requestUpdate();
        Assert.assertEquals((long)this.refreshBackoffMs, (long)this.metadata.timeToNextUpdate(now));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now += this.refreshBackoffMs));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now + 1L));
    }

    @Test
    public void testTimeToNextUpdate_OverwriteBackoff() {
        long now = 10000L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), now);
        this.metadata.add("new-topic");
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(Cluster.empty(), Collections.emptySet(), now);
        this.metadata.setTopics((Collection)this.metadata.topics());
        Assert.assertEquals((long)this.metadataExpireMs, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.setTopics(Collections.singletonList("another-new-topic"));
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(Cluster.empty(), Collections.emptySet(), now);
        this.metadata.needMetadataForAllTopics(true);
        Assert.assertEquals((long)0L, (long)this.metadata.timeToNextUpdate(now));
        this.metadata.update(Cluster.empty(), Collections.emptySet(), now);
        this.metadata.needMetadataForAllTopics(true);
        Assert.assertEquals((long)this.metadataExpireMs, (long)this.metadata.timeToNextUpdate(now));
    }

    @Test
    public void testMetadataUpdateWaitTime() throws Exception {
        long time = 0L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Assert.assertFalse((String)"No update needed.", (this.metadata.timeToNextUpdate(time) == 0L ? 1 : 0) != 0);
        try {
            this.metadata.awaitUpdate(this.metadata.requestUpdate(), 0L);
            Assert.fail((String)"Wait on metadata update was expected to timeout, but it didn't");
        }
        catch (TimeoutException timeoutException) {
            // empty catch block
        }
        long twoSecondWait = 2000L;
        try {
            this.metadata.awaitUpdate(this.metadata.requestUpdate(), 2000L);
            Assert.fail((String)"Wait on metadata update was expected to timeout, but it didn't");
        }
        catch (TimeoutException timeoutException) {
            // empty catch block
        }
    }

    @Test
    public void testFailedUpdate() {
        long time = 100L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1000L));
        this.metadata.failedUpdate(1100L, null);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1100L));
        Assert.assertEquals((long)100L, (long)this.metadata.lastSuccessfulUpdate());
        this.metadata.needMetadataForAllTopics(true);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Assert.assertEquals((long)100L, (long)this.metadata.timeToNextUpdate(1000L));
    }

    @Test
    public void testUpdateWithNeedMetadataForAllTopics() {
        long time = 0L;
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        this.metadata.needMetadataForAllTopics(true);
        List<String> expectedTopics = Collections.singletonList("topic");
        this.metadata.setTopics(expectedTopics);
        this.metadata.update(new Cluster(null, Collections.singletonList(new Node(0, "host1", 1000)), Arrays.asList(new PartitionInfo("topic", 0, null, null, null), new PartitionInfo("topic1", 0, null, null, null)), Collections.emptySet(), Collections.emptySet()), Collections.emptySet(), 100L);
        Assert.assertArrayEquals((String)"Metadata got updated with wrong set of topics.", (Object[])expectedTopics.toArray(), (Object[])this.metadata.topics().toArray());
        this.metadata.needMetadataForAllTopics(false);
    }

    @Test
    public void testClusterListenerGetsNotifiedOfUpdate() {
        long time = 0L;
        MockClusterResourceListener mockClusterListener = new MockClusterResourceListener();
        ClusterResourceListeners listeners = new ClusterResourceListeners();
        listeners.maybeAdd((Object)mockClusterListener);
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, false, listeners);
        String hostName = "www.example.com";
        Cluster cluster = Cluster.bootstrap(Arrays.asList(new InetSocketAddress(hostName, 9002)));
        this.metadata.update(cluster, Collections.emptySet(), time);
        Assert.assertFalse((String)"ClusterResourceListener should not called when metadata is updated with bootstrap Cluster", (boolean)MockClusterResourceListener.IS_ON_UPDATE_CALLED.get());
        this.metadata.update(new Cluster("dummy", Arrays.asList(new Node(0, "host1", 1000)), Arrays.asList(new PartitionInfo("topic", 0, null, null, null), new PartitionInfo("topic1", 0, null, null, null)), Collections.emptySet(), Collections.emptySet()), Collections.emptySet(), 100L);
        Assert.assertEquals((String)"MockClusterResourceListener did not get cluster metadata correctly", (Object)"dummy", (Object)mockClusterListener.clusterResource().clusterId());
        Assert.assertTrue((String)"MockClusterResourceListener should be called when metadata is updated with non-bootstrap Cluster", (boolean)MockClusterResourceListener.IS_ON_UPDATE_CALLED.get());
    }

    @Test
    public void testListenerGetsNotifiedOfUpdate() {
        long time = 0L;
        final HashSet topics = new HashSet();
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        this.metadata.addListener(new Metadata.Listener(){

            public void onMetadataUpdate(Cluster cluster, Set<String> unavailableTopics) {
                topics.clear();
                topics.addAll(cluster.topics());
            }
        });
        this.metadata.update(new Cluster(null, Arrays.asList(new Node(0, "host1", 1000)), Arrays.asList(new PartitionInfo("topic", 0, null, null, null), new PartitionInfo("topic1", 0, null, null, null)), Collections.emptySet(), Collections.emptySet()), Collections.emptySet(), 100L);
        Assert.assertEquals((String)"Listener did not update topics list correctly", new HashSet<String>(Arrays.asList("topic", "topic1")), topics);
    }

    @Test
    public void testListenerCanUnregister() {
        long time = 0L;
        final HashSet topics = new HashSet();
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        Metadata.Listener listener = new Metadata.Listener(){

            public void onMetadataUpdate(Cluster cluster, Set<String> unavailableTopics) {
                topics.clear();
                topics.addAll(cluster.topics());
            }
        };
        this.metadata.addListener(listener);
        this.metadata.update(new Cluster("cluster", Collections.singletonList(new Node(0, "host1", 1000)), Arrays.asList(new PartitionInfo("topic", 0, null, null, null), new PartitionInfo("topic1", 0, null, null, null)), Collections.emptySet(), Collections.emptySet()), Collections.emptySet(), 100L);
        this.metadata.removeListener(listener);
        this.metadata.update(new Cluster("cluster", Arrays.asList(new Node(0, "host1", 1000)), Arrays.asList(new PartitionInfo("topic2", 0, null, null, null), new PartitionInfo("topic3", 0, null, null, null)), Collections.emptySet(), Collections.emptySet()), Collections.emptySet(), 100L);
        Assert.assertEquals((String)"Listener did not update topics list correctly", new HashSet<String>(Arrays.asList("topic", "topic1")), topics);
    }

    @Test
    public void testTopicExpiry() throws Exception {
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, true, new ClusterResourceListeners());
        long time = 0L;
        this.metadata.add("topic1");
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time += 300000L);
        Assert.assertFalse((String)"Unused topic not expired", (boolean)this.metadata.containsTopic("topic1"));
        this.metadata.add("topic2");
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        for (int i = 0; i < 3; ++i) {
            this.metadata.update(Cluster.empty(), Collections.emptySet(), time += 150000L);
            Assert.assertTrue((String)"Topic expired even though in use", (boolean)this.metadata.containsTopic("topic2"));
            this.metadata.add("topic2");
        }
        HashSet<String> topics = new HashSet<String>();
        topics.add("topic4");
        this.metadata.setTopics(topics);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time += 300000L);
        Assert.assertFalse((String)"Unused topic not expired", (boolean)this.metadata.containsTopic("topic4"));
    }

    @Test
    public void testNonExpiringMetadata() throws Exception {
        this.metadata = new Metadata(this.refreshBackoffMs, this.metadataExpireMs, true, false, new ClusterResourceListeners());
        long time = 0L;
        this.metadata.add("topic1");
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time += 300000L);
        Assert.assertTrue((String)"Unused topic expired when expiry disabled", (boolean)this.metadata.containsTopic("topic1"));
        this.metadata.add("topic2");
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time);
        for (int i = 0; i < 3; ++i) {
            this.metadata.update(Cluster.empty(), Collections.emptySet(), time += 150000L);
            Assert.assertTrue((String)"Topic expired even though in use", (boolean)this.metadata.containsTopic("topic2"));
            this.metadata.add("topic2");
        }
        HashSet<String> topics = new HashSet<String>();
        topics.add("topic4");
        this.metadata.setTopics(topics);
        this.metadata.update(Cluster.empty(), Collections.emptySet(), time += this.metadataExpireMs * 2L);
        Assert.assertTrue((String)"Unused topic expired when expiry disabled", (boolean)this.metadata.containsTopic("topic4"));
    }

    private void clearBackgroundError() {
        this.backgroundError.set(null);
    }

    private Thread asyncFetch(final String topic, final long maxWaitMs) {
        Thread thread = new Thread(){

            @Override
            public void run() {
                try {
                    while (MetadataTest.this.metadata.fetch().partitionsForTopic(topic).isEmpty()) {
                        MetadataTest.this.metadata.awaitUpdate(MetadataTest.this.metadata.requestUpdate(), maxWaitMs);
                    }
                }
                catch (Exception e) {
                    MetadataTest.this.backgroundError.set(e);
                }
            }
        };
        thread.start();
        return thread;
    }
}

