package com.ptteng.rent.user.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.gemantic.common.util.MyListUtil;
import com.ptteng.rent.user.constant.WxConfig;
import com.ptteng.rent.user.model.Orders;
import com.ptteng.rent.user.model.User;
import com.ptteng.rent.user.service.OrdersService;
import com.ptteng.rent.user.service.UserService;
import com.ptteng.rent.user.utils.*;
import com.qding.common.util.DataUtils;
import com.qding.common.util.HttpClientUtil;
import com.qding.community.common.weixin.service.WeiXinService;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Orders  crud
 *
 * @author ss127
 * @date 2018/6/7
 */
@Controller
public class OrdersController {
    private static final Log log = LogFactory.getLog(OrdersController.class);

    @Autowired
    private OrdersService ordersService;
    @Autowired
    private HttpClientUtil httpClientUtil;
    @Autowired
    private UserService userService;
    @Autowired
    private WeiXinService weixinService;
    @Autowired
    private JdbcTemplate jdbcTemplate;

    /**
     * 创建订单
     *
     * @param request
     * @param response
     * @param model
     * @param lesseeId         承租人ID
     * @param lessorId         出租人ID
     * @param leaseStartAt     租赁起始时间
     * @param leaseEndAt       租赁结束时间
     * @param price            订单价格
     * @param totalPrice       总支付金额
     * @param lesseeMessage    承租人留言
     * @param positioning      订单提交城市（承租人当前定位城市）
     * @param lesseeServiceFee 承租人服务费
     * @param lessorServiceFee 出租人服务费
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order", method = RequestMethod.POST)
    public String addOrdersJson(HttpServletRequest request,
                                HttpServletResponse response, ModelMap model, Long lesseeId, Long lessorId, Long leaseStartAt, Long leaseEndAt,
                                BigDecimal price, BigDecimal totalPrice, String lesseeMessage, String positioning,
                                BigDecimal lesseeServiceFee, BigDecimal lessorServiceFee)
            throws Exception {

        log.info("url is : /a/u/order?lesseeId=" + lesseeId + "&lessorId=" + lessorId + "&leaseStartAt=" + leaseStartAt + "&leaseEndAt=" + leaseEndAt
                + "&price=" + price + "&totalPrice=" + totalPrice + "&lesseeMessage=" + lesseeMessage +
                "&positioning=" + positioning + "&lesseeServiceFee=" + lesseeServiceFee + "&lessorServiceFee=" +
                lessorServiceFee);

        Integer code = OrdersCheck.checkOrdersParams(lesseeId, lessorId, leaseStartAt, leaseEndAt, price, totalPrice,
                lesseeMessage, positioning, lesseeServiceFee, lessorServiceFee);

        if (!code.equals(0)) {

            model.addAttribute("code", code);

            return "/common/success";

        }

        try {

            Long differenceTime = leaseStartAt - leaseEndAt;

            log.info("difference time is :" + differenceTime);

            // 最少出租时间1小时
            if (Orders.AT_LEAST < differenceTime) {

                log.info("rental time cannot be less than one hour");

                model.addAttribute("code", -3004);

                return "/common/success";

            }

            // 出租开始时间结束时间不能小于当前时间
            if (System.currentTimeMillis() > leaseStartAt || System.currentTimeMillis() > leaseEndAt) {

                log.info("Start time and end time cannot be selected before the current time");

                model.addAttribute("code", -3005);

                return "/common/success";

            }

            User lessor = userService.getObjectById(lessorId);

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            Integer lessorUserType = lessor.getUserType();

            log.info("lessor user type is :" + lessorUserType);

            // 出租人id用户身份为承租人
            if (User.LESSEE.equals(lessorUserType)) {

                log.info("Lessor cannot lessee ");

                model.addAttribute("code", -3015);

                return "/common/success";

            }

            User lessee = userService.getObjectById(lesseeId);

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            Integer lesseeUserType = lessee.getUserType();

            log.info("lessee user type is :" + lesseeUserType);

            if (User.LESSEE.equals(lesseeUserType)) {

                // 承租人最多五单
                Map<String, Object> lesseeOrderParams = DynamicUtil.getLesseeProcessingOrders(lesseeId);

                log.info("lessee order params is :" + lesseeOrderParams);

                List<Long> lesseeOrderIds = ordersService.getIdsByDynamicCondition(Orders.class, lesseeOrderParams, 0, 10);

                log.info("lessee order id size is ；" + lesseeOrderIds.size());

                if (CollectionUtils.isNotEmpty(lesseeOrderIds)) {

                    Integer countLesseeOrders = lesseeOrderIds.size();

                    if (countLesseeOrders > Orders.LESSEE_MAX_ORDERS) {

                        log.info("lessee  accept up to five orders for non-end status");

                        model.addAttribute("code", -3017);

                        return "/common/success";

                    }

                }

            }

            Orders orders = new Orders();

            orders.setLesseeId(lesseeId);

            orders.setLessorId(lessorId);

            orders.setLeaseStartAt(leaseStartAt);

            orders.setLeaseEndAt(leaseEndAt);

            orders.setPrice(price);

            orders.setTotalPrice(totalPrice);

            orders.setLesseeMessage(lesseeMessage);

            String orderNumber = OrderUtils.getOrderNumber();

            log.info("order number is :" + orderNumber);

            orders.setOrderNo(orderNumber);

            orders.setPayStatus(Orders.UNPAID);

            orders.setStatus(Orders.UNPAID);

            orders.setPositioning(positioning);

            orders.setLesseeServiceFee(lesseeServiceFee);

            orders.setLessorServiceFee(lessorServiceFee);

            orders.setPayAt(System.currentTimeMillis());

            User user = userService.getObjectById(lessorId);

            if (DataUtils.isNullOrEmpty(user)) {

                log.info("user is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            if (DataUtils.isNotNullOrEmpty(user.getRecommenderId())) {

                orders.setMid(user.getRecommenderId());

            }

            String skill = user.getSkill();

            log.info("lessor skill is :" + skill);

            if (DataUtils.isNullOrEmpty(skill)) {

                log.info("skill is null");

                model.addAttribute("code", -4024);

                return "/common/success";

            }

            orders.setServiceContent(skill);

            Long id = ordersService.insert(orders);

            log.info("order id is :" + id);

            model.addAttribute("code", 0);
            model.addAttribute("id", id);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add orders error ");
            model.addAttribute("code", -6002);
        }

        return "/common/insert";

    }


    /**
     * 订单详情
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}", method = RequestMethod.GET)
    public String getOrdersJson(HttpServletRequest request,
                                HttpServletResponse response, ModelMap model, @PathVariable Long id,Integer
                                            isGetOrder,Long uid)
            throws Exception {

        log.info("/a/u/orders/" + id);

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("id is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }


            if (Orders.YES.equals(isGetOrder)) {

                if (DataUtils.isNotNullOrEmpty(uid)) {

                    User user = userService.getObjectById(uid);

                    if (DataUtils.isNullOrEmpty(user)) {

                        log.info("user is null");

                        model.addAttribute("code", -2011);

                        return "/common/success";

                    }

                    if (DataUtils.isNotNullOrEmpty(user.getNewOrders()) || User.NEW_ORDER_AND_STATUS < user.getNewOrders()) {

                        user.setNewOrders(User.NEW_ORDER_AND_STATUS);

                    }

                    if (DataUtils.isNotNullOrEmpty(user.getOrdersStatus()) || User.NEW_ORDER_AND_STATUS < user
                            .getOrdersStatus()) {

                        user.setOrdersStatus(User.NEW_ORDER_AND_STATUS);

                    }

                    boolean b = userService.update(user);

                    if (b) {

                        log.info("user update type is success");

                    } else {

                        log.info("user update type is fail");

                        model.addAttribute("code", -6003);

                        return "/common/success";

                    }

                }



            } else {

                log.info("is not get order list");

            }

            // 进行中订单，出租人拒绝承租人退款，是否能进行下一次退款，默认=0，是=1，否=2
            Integer isNextRefund = 0;

            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (Orders.LESSOR_REFUSES_REFUND.equals(status)) {

                // 订单出租人拒绝时间
                Long lessorRefuseAt = orders.getRefuseAt();

                Long interval = System.currentTimeMillis() - lessorRefuseAt;

                log.info("interval is :" + interval);

                if (Orders.TEN_MINUTES_MILL <= interval) {

                    isNextRefund = Orders.IS_NEXT_REFUND;

                } else {

                    isNextRefund = Orders.IS_NOT_NEXT_REFUND;

                }

            }

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("user is null,user id is :" + orders.getLesseeId());

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null,user id is :" + orders.getLessorId());

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            model.addAttribute("code", 0);
            model.addAttribute("orders", orders);
            model.addAttribute("lessee", lessee);
            model.addAttribute("lessor", lessor);
            model.addAttribute("isNextRefund", isNextRefund);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/rent-user-service/orders/json/ordersDetailJson";
    }


    /**
     * 我的订单
     *
     * @param request
     * @param response
     * @param model
     * @param uid      用户id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{uid}/list", method = RequestMethod.GET)
    public String myOrderList(HttpServletRequest request,
                              HttpServletResponse response, ModelMap model, @PathVariable Long uid,Integer
                                          isGetOrderList)
            throws Exception {

        log.info("/a/u/orders/" + uid + "/list");

        if (DataUtils.isNullOrEmpty(uid)) {

            log.info("id is :" + uid);

            model.addAttribute("code", -2033);

            return "/common/success";

        }

        try {

            User user = userService.getObjectById(uid);

            if (DataUtils.isNullOrEmpty(user)) {

                log.info("user is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            if (Orders.YES.equals(isGetOrderList)) {

                if (DataUtils.isNotNullOrEmpty(user.getNewOrders()) || User.NEW_ORDER_AND_STATUS < user.getNewOrders()) {

                    user.setNewOrders(User.NEW_ORDER_AND_STATUS);

                }

                if (DataUtils.isNotNullOrEmpty(user.getOrdersStatus()) || User.NEW_ORDER_AND_STATUS < user
                        .getOrdersStatus()) {

                    user.setOrdersStatus(User.NEW_ORDER_AND_STATUS);

                }

                boolean b = userService.update(user);

                if (b) {

                    log.info("user update type is success");

                } else {

                    log.info("user update type is fail");

                    model.addAttribute("code", -6003);

                    return "/common/success";

                }

            } else {

                log.info("is not get order list");

            }



            List<Orders> orderList = new ArrayList<>();

            List<User> userList = new ArrayList<>();

            Integer total = 0;

            Integer userType = user.getUserType();

            log.info("user type is :" + userType);

//            if (User.LESSEE.equals(userType)) {

                String getLesseeIdOrders = "select id from orders where lessee_id=" + uid + " and (status in (1,2,3," +
                        "4,5,6,7,8,10,13,14) or lessee_evaluation_status=1) order by create_at desc";

                List<Map<String, Object>> getLesseeIdOrdersResultMapList = jdbcTemplate.queryForList(getLesseeIdOrders);

                log.info("getLesseeIdOrdersResultMapList size is :" + getLesseeIdOrdersResultMapList.size());

                List<Long> lesseeIds = new ArrayList<>();

                if (CollectionUtils.isNotEmpty(getLesseeIdOrdersResultMapList)) {

                    for (Map<String, Object> lesseeId : getLesseeIdOrdersResultMapList) {

                        lesseeIds.add((Long) lesseeId.get("id"));

                    }

                }

                log.info("lesseeIds size is:" + lesseeIds.size());

                if (CollectionUtils.isNotEmpty(lesseeIds)) {

                    orderList = ordersService.getObjectsByIds(lesseeIds);

                    total = orderList.size();

                }


                log.info("total is :" + total);

                List<Long> uids = MyListUtil.getDistinctList(Orders.class.getDeclaredField("lessorId"),
                        orderList);


                log.info("lessorIdsByOrders is :" + uids);

//                userList = userService.getObjectsByIds(lessorIdsByOrders);

//            } else if (User.LESSOR.equals(userType)) {

                String getLessorIdOrders = "select id from orders where lessor_id=" + uid + " and (status in (1,2,4,6,13) or lessor_evaluation_status=1)  order by create_at desc";

                List<Map<String, Object>> getLessorIdOrdersResultMapList = jdbcTemplate.queryForList(getLessorIdOrders);

                log.info("getLessorIdOrdersResultMapList size is :" + getLessorIdOrdersResultMapList.size());

                log.info("==============================");

                log.info("getLessorIdOrdersResultMapList" + getLessorIdOrdersResultMapList);

                List<Long> lessorIds = new ArrayList<>();

                if (CollectionUtils.isNotEmpty(getLessorIdOrdersResultMapList)) {

                    for (Map<String, Object> lessorId : getLessorIdOrdersResultMapList) {

                        lessorIds.add((Long) lessorId.get("id"));

                    }

                }

                log.info("lessorIds size is :" + lessorIds.size());
                List<Orders> orderList2 = new ArrayList<>();
                if (CollectionUtils.isNotEmpty(lessorIds)) {

                    orderList2 = ordersService.getObjectsByIds(lessorIds);
                    orderList.addAll(orderList2);
                    total = total+orderList2.size();

                    log.info("total is :" + total);

                }

                List<Long> lesseeIds2 = MyListUtil.getDistinctList(Orders.class.getDeclaredField("lesseeId"), orderList2);
                uids.addAll(lesseeIds2);
                log.info("lessee ids is :" + lesseeIds2);
                HashSet<Long> set = new HashSet<>(uids);
                uids.clear();
                uids.addAll(set);
                userList = userService.getObjectsByIds(uids);

//            }


            model.addAttribute("code", 0);
            model.addAttribute("orderList", orderList);
            model.addAttribute("userList", userList);
            model.addAttribute("total", total);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,uid is  " + uid);
            model.addAttribute("code", -100000);
        }

        return "/rent-user-service/orders/json/myOrderList";
    }


    /**
     * 保存支付结果
     *
     * @param request
     * @param response
     * @param model
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/result/{oid}", method = RequestMethod.PUT)
    public String getOrdersIdsByStatusList1(HttpServletRequest request,
                                            HttpServletResponse response, ModelMap model, @PathVariable Long oid, Integer result) throws Exception {

        log.info("/a/u/order/result/" + oid + "&result=" + result);

        if (DataUtils.isNullOrEmpty(oid)) {

            log.info("oid is :" + oid);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        if (DataUtils.isNullOrEmpty(result)) {

            log.info("result is :" + result);

            model.addAttribute("code", -3002);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(oid);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String openid = lessee.getOpenID();

            log.info("lessee openid is ；" + openid);

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String skill = lessor.getSkill();

            log.info("lessor skill is ；" + skill);

            Long leaseStartAt = orders.getLeaseStartAt();

            log.info("lease start at is :" + leaseStartAt);

            Long leaseEndAt = orders.getLeaseEndAt();

            log.info("lease end at is :" + leaseEndAt);

            Long duration = (leaseEndAt - leaseStartAt) / Orders.AT_LEAST;

            log.info("duration is :" + duration);

            if (Orders.PAY_SUCCESS.equals(result)) {

                orders.setPayStatus(result);

                orders.setStatus(Orders.LESSOR_IS_NOT_CONFIRMED);

                // 承租人下单成功小程序推送
                String sendWxMessage = AppletSendMessage.payOrderSuccessMessage(openid, skill, duration.toString(), orders
                        .getTotalPrice().toString(),lessee.getId(),oid);

                weixinService.updateAccessToken();

                String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

                log.info("send applet res is :" + res);

                SimpleDateFormat sdf = new SimpleDateFormat("yyyy.MM.dd HH:mm");

                String startTime = sdf.format(new Date(Long.parseLong(String.valueOf(leaseStartAt))));

                log.info("start time is :" + startTime);

                String endTime = sdf.format(new Date(Long.parseLong(String.valueOf(leaseEndAt))));

                log.info("end time is :" + endTime);

                String appointment = startTime + "-" + endTime;

                log.info("appointment is :" + appointment);

                String lessorSendMessage = AppletSendMessage.newOrderMessage(lessor.getOpenID(), lessee.getNick(),
                        appointment, orders.getPrice().toString(),lessor.getId(),oid);

                String lessorRes = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, lessorSendMessage,
                        true);

                log.info("send applet lessorRes is :" + lessorRes);

                lessor.setNewOrders(lessor.getNewOrders() + User.PLUS_ONE);

                userService.update(lessor);

            } else if (Orders.PAY_FAIL.equals(result)) {

                orders.setPayStatus(result);

            } else {

                log.info("params is error");

                model.addAttribute("code", -996);

                return "/common/success";

            }

            boolean b = ordersService.update(orders);

            if (b) {

                log.info("order update pay status success");

            } else {

                log.info("order update pay status fail");

                model.addAttribute("code", -3003);

                return "/common/success";

            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + oid);
            model.addAttribute("code", -100000);
        }


        return "/common/success";
    }




    /**
     * 出租人是否接受订单
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @param isAccept 是否接受订单 约定：1=接受，2=拒绝
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/accept", method = RequestMethod.PUT)
    public String lessorIsAcceptOrder(HttpServletRequest request,
                                      HttpServletResponse response, ModelMap model, @PathVariable Long id, Integer isAccept)
            throws Exception {

        log.info("/a/u/order/" + id + "/accept?isAccept=" + isAccept);

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        if (DataUtils.isNullOrEmpty(isAccept)) {

            log.info("is accept is null");

            model.addAttribute("code", -3018);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_NOT_CONFIRMED.equals(status)) {

                log.info("order status is :" + status + ",The current status of the order is incorrect");

                model.addAttribute("code", -3008);

                return "/common/success";

            }

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String openid = lessee.getOpenID();

            log.info("lessee openid is ；" + openid);

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String skill = lessor.getSkill();

            log.info("lessor skill is ；" + skill);

            if (Orders.IS_ACCEPT.equals(isAccept)) {

                log.info("lessor accept lessee order");

                orders.setStatus(Orders.LESSOR_IS_CONFIRMED);

                // 出租人接受承租人订单申请推送
                String sendWxMessage = AppletSendMessage.orderConfirmMessage(openid, skill, orders.getTotalPrice()
                        .toString(),lessee.getId(),id);

                weixinService.updateAccessToken();

                String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

                log.info("send applet res is :" + res);

                lessee.setOrdersStatus(lessee.getOrdersStatus() + User.PLUS_ONE);

                userService.update(lessee);

            } else if (Orders.IS_NOT_ACCEPT.equals(isAccept)) {

                log.info("lessor is not accept lessee order");

                // 待确认出租人拒绝接受承租人订单状态怎么变 TODO
                orders.setStatus(Orders.REFUSE_TO_TAKE_ORDERS);

                lessee.setOrdersStatus(lessee.getOrdersStatus() + User.PLUS_ONE);

                userService.update(lessee);

            } else {

                log.info("Illegal parameters");

                model.addAttribute("code", -996);

                return "/common/success";

            }

            boolean b = ordersService.update(orders);

            if (b) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 订单待确认承租人申请退款
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/refund", method = RequestMethod.PUT)
    public String ordersRefund(HttpServletRequest request,
                               HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("/a/u/order/" + id + "/refund");

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_NOT_CONFIRMED.equals(status)) {

                log.info("order status is :" + status + ",The current status of the order is incorrect");

                model.addAttribute("code", -3008);

                return "/common/success";

            }

            orders.setStatus(Orders.LESSOR_IS_CONFIRMED_LESSEE_REFUND);

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("user is not exist.");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            lessee.setOrdersStatus(lessee.getOrdersStatus() + User.PLUS_ONE);

            String sendWxMessage = AppletSendMessage.refundRequestNotificationMessage(lessor.getOpenID(), lessee
                    .getNick(), orders.getPrice().toString(),lessor.getId(),id);

            weixinService.updateAccessToken();

            String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

            log.info("send applet res is :" + res);


            userService.update(lessee);

            boolean b = ordersService.update(orders);

            if (b) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }


    /**
     * 订单进行中承租人申请退款
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/termination", method = RequestMethod.PUT)
    public String terminationOrder(HttpServletRequest request,
                                   HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("/a/u/order/" + id + "/refund");

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_CONFIRMED.equals(status)) {

                if (!Orders.LESSOR_REFUSES_REFUND.equals(status)) {

                    log.info("order status is :" + status + ",The current status of the order is incorrect");

                    model.addAttribute("code", -3008);

                    return "/common/success";

                }

            }

            orders.setStatus(Orders.PROCESSING_LESSEE_REFUND_LESSOR_IS_NOT_CONFIRMED);

            orders.setRefundAt(System.currentTimeMillis());

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String sendWxMessage = AppletSendMessage.refundRequestNotificationMessage(lessor.getOpenID(), lessee
                    .getNick(), orders.getPrice().toString(),lessor.getId(),id);

            weixinService.updateAccessToken();

            String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

            log.info("send applet res is :" + res);

            boolean b = ordersService.update(orders);

            if (b) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }


            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }


    /**
     * 出租人终止订单（订单进行中状态）
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @param lessorMessage 出租人留言
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/lessor/termination", method = RequestMethod.PUT)
    public String lessorTerminationOrder(HttpServletRequest request,
                                         HttpServletResponse response, ModelMap model, @PathVariable Long id, String lessorMessage)
            throws Exception {

        log.info("/a/u/order/" + id + "/lessor/termination?lessorMessage=" + lessorMessage);

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        if (DataUtils.isNullOrEmpty(lessorMessage)) {

            log.info("lessorMessage is :" + lessorMessage);

            model.addAttribute("code", -3010);

            return "/common/success";

        }


        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_CONFIRMED.equals(status)) {

                if (!Orders.LESSOR_REFUSES_REFUND.equals(status)) {

                    log.info("order status is :" + status + ",The current status of the order is incorrect");

                    model.addAttribute("code", -3008);

                    return "/common/success";

                }

            }

            orders.setStatus(Orders.PROCESSING_LESSOR_TERMINATION_ORDER);

            orders.setLessorMessage(lessorMessage);

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            lessee.setOrdersStatus(lessee.getOrdersStatus() + User.PLUS_ONE);

            userService.update(lessee);

            boolean boo = ordersService.update(orders);

            if (boo) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 进行中承租人申请订单退款出租人是否同意退款
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @param isAgree  是否同意退款 约定：1=同意，2=拒绝；
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/lessor/refund", method = RequestMethod.PUT)
    public String lessorRefund(HttpServletRequest request,
                               HttpServletResponse response, ModelMap model, @PathVariable Long id, Integer
                                       isAgree)
            throws Exception {

        log.info("/a/u/order/" + id + "/lessor/refund=" + isAgree);

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        if (DataUtils.isNullOrEmpty(isAgree)) {

            log.info("isAgree is :" + isAgree);

            model.addAttribute("code", -3011);

            return "/common/success";

        }


        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.PROCESSING_LESSEE_REFUND_LESSOR_IS_NOT_CONFIRMED.equals(status)) {

                if (!Orders.LESSOR_REFUSES_REFUND.equals(status)) {

                    log.info("order status is :" + status + ",The current status of the order is incorrect");

                    model.addAttribute("code", -3019);

                    return "/common/success";

                }

            }


            if (Orders.AGREE.equals(isAgree)) {

                log.info("lessor agree lessee order");

                orders.setStatus(Orders.PROCESSING_LESSEE_REFUND_LESSOR_IS_CONFIRMED);

                User lessee = userService.getObjectById(orders.getLesseeId());

                if (DataUtils.isNullOrEmpty(lessee)) {

                    log.info("user is not exist.");

                    model.addAttribute("code", -2011);

                    return "/common/success";

                }

                lessee.setOrdersStatus(lessee.getOrdersStatus() + User.PLUS_ONE);

                userService.update(lessee);

            } else if (Orders.DISAGREE.equals(isAgree)) {

                log.info("lessor disagree lessee order");

                orders.setStatus(Orders.LESSOR_REFUSES_REFUND);

                orders.setRefuseAt(System.currentTimeMillis());

            } else {

                log.info("Illegal parameters");

                model.addAttribute("code", -996);

                return "/common/success";

            }

            boolean boo = ordersService.update(orders);

            if (boo) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 进行中订单，承租人结束订单，订单状态变为已结束
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/lessee/end", method = RequestMethod.PUT)
    public String lesseeEndOrder(HttpServletRequest request,
                                 HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("/a/u/order/" + id + "/lessee/end");

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        try {

            User user = (User) request.getAttribute("user");

            log.info("=======================" + user.getId());

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            if (!user.getId().equals(orders.getLesseeId())) {

                log.info("user id error");

                model.addAttribute("code", -3013);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_CONFIRMED.equals(status)) {

                if (!Orders.LESSOR_REFUSES_REFUND.equals(status)) {

                    log.info("order status is :" + status + ",The current status of the order is incorrect");

                    model.addAttribute("code", -3008);

                    return "/common/success";

                }

            }

            orders.setStatus(Orders.COMPLETE_THE_ORDER_NORMALLY);

            orders.setEndOfOrderAt(System.currentTimeMillis());

            orders.setLessorEvaluationStatus(Orders.NOT_EVALUATION);

            orders.setLesseeEvaluationStatus(Orders.NOT_EVALUATION);

            // 发送小程序推送
            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String openid = lessee.getOpenID();

            log.info("lessee openid is ；" + openid);

            Long leaseStartAt = orders.getLeaseStartAt();

            log.info("lease start at is :" + leaseStartAt);

            Long leaseEndAt = orders.getLeaseEndAt();

            log.info("lease end at is :" + leaseEndAt);

            Long duration = (leaseEndAt - leaseStartAt) / Orders.AT_LEAST;

            log.info("duration is :" + duration);

            // 承租人下单成功小程序推送
            String sendWxMessage = AppletSendMessage.lesseeOrderEndMessage(openid, duration.toString(), orders
                    .getTotalPrice().toString(),lessee.getId(),id);

            weixinService.updateAccessToken();

            String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

            log.info("send applet res is :" + res);

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null,user id is :" + orders.getLessorId());

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String lessorSendMessage = AppletSendMessage.orderEndMessage(lessor.getOpenID(), orders.getPrice()
                    .toString(),lessor.getId(),id);

            weixinService.updateAccessToken();

            String lessorRes = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, lessorSendMessage, true);

            log.info("send applet lessorRes is :" + lessorRes);

            List<User> userList = new ArrayList<>();

            lessee.setLesseeOrders(lessee.getLesseeOrders() + User.NEW_ORDER);

            lessor.setLessorOrders(lessor.getLessorOrders() + User.NEW_ORDER);

            userList.add(lessor);

            userList.add(lessee);

            boolean b = userService.updateList(userList);

            if (b) {

                log.info("user update add orders success");

            } else {

                log.info("ouser update add orders fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            boolean boo = ordersService.update(orders);

            if (boo) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 订单正常结束
     *
     * @param request
     * @param response
     * @param model
     * @param id
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/a/u/order/{id}/end", method = RequestMethod.PUT)
    public String endOrder(HttpServletRequest request,
                           HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {

        log.info("/a/u/order/" + id + "/lessee/end");

        if (DataUtils.isNullOrEmpty(id)) {

            log.info("oid is :" + id);

            model.addAttribute("code", -4003);

            return "/common/success";

        }

        try {

            Orders orders = ordersService.getObjectById(id);

            if (DataUtils.isNullOrEmpty(orders)) {

                log.info("order is not exist");

                model.addAttribute("code", -3000);

                return "/common/success";

            }

            // 校验支付状态
            Integer payStatus = orders.getPayStatus();

            log.info("order pay status is :" + payStatus);

            if (Orders.PAY_FAIL.equals(payStatus)) {

                log.info("order pay status is fail");

                model.addAttribute("code", -3007);

                return "/common/success";

            }

            // 校验订单状态
            Integer status = orders.getStatus();

            log.info("order status is :" + status);

            if (!Orders.LESSOR_IS_CONFIRMED.equals(status)) {

                log.info("order status is :" + status + ",The current status of the order is incorrect");

                model.addAttribute("code", -3008);

                return "/common/success";

            }

            Long orderEndAt = orders.getLeaseEndAt();

            log.info("order ent at is :" + orderEndAt);

            Long currentAt = System.currentTimeMillis();

            if (orderEndAt > currentAt) {

                log.info("Order not ended");

                model.addAttribute("code", -3014);

                return "/common/success";

            }

            User lessee = userService.getObjectById(orders.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String openid = lessee.getOpenID();

            log.info("lessee openid is ；" + openid);

            Long leaseStartAt = orders.getLeaseStartAt();

            log.info("lease start at is :" + leaseStartAt);

            Long leaseEndAt = orders.getLeaseEndAt();

            log.info("lease end at is :" + leaseEndAt);

            Long duration = (leaseEndAt - leaseStartAt) / Orders.AT_LEAST;

            log.info("duration is :" + duration);

            // 承租人下单成功小程序推送
            String sendWxMessage = AppletSendMessage.lesseeOrderEndMessage(openid, duration.toString(), orders
                    .getTotalPrice().toString(),lessee.getId(),id);

            weixinService.updateAccessToken();

            String res = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, sendWxMessage, true);

            log.info("send applet res is :" + res);

            User lessor = userService.getObjectById(orders.getLessorId());

            if (DataUtils.isNullOrEmpty(lessor)) {

                log.info("lessor is null,user id is :" + orders.getLessorId());

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String lessorSendMessage = AppletSendMessage.orderEndMessage(lessor.getOpenID(), orders.getPrice()
                    .toString(),lessor.getId(),id);

            weixinService.updateAccessToken();

            String lessorRes = weixinService.sendAppletMessage(WxConfig.appID, WxConfig.appSecret, lessorSendMessage, true);

            log.info("send applet lessorRes is :" + lessorRes);

            orders.setStatus(Orders.COMPLETE_THE_ORDER_NORMALLY);

            boolean boo = ordersService.update(orders);

            if (boo) {

                log.info("order update status success");

            } else {

                log.info("order update status fail");

                model.addAttribute("code", -3009);

                return "/common/success";

            }

            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get orders error,id is  " + id);
            model.addAttribute("code", -100000);
        }

        return "/common/success";
    }

    /**
     * 微信预下单
     *
     * @param request
     * @param response
     * @param model
     * @param oid
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/wx/pay/info/{oid}", method = RequestMethod.GET)
    public String getPayInfo(HttpServletRequest request, HttpServletResponse response, ModelMap model, @PathVariable Long oid) throws ServiceException, ServiceDaoException {
        log.info("order : " + oid );

        XmlUtil xmlUtil = new XmlUtil();
        String name = "租友乐";
        log.info("remoteHost is " + TxtUtil.getIpAddress(request));
        try {
            log.info("==========检验是否拥有本订单==========");
            Orders order = ordersService.getObjectById(oid);
            if (null == order) {
                log.info("already pay the order");
                model.addAttribute("code", -3007);
                return "/common/success";
            }

            User lessee = userService.getObjectById(order.getLesseeId());

            if (DataUtils.isNullOrEmpty(lessee)) {

                log.info("lessee is null");

                model.addAttribute("code", -2011);

                return "/common/success";

            }

            String nonce_str = getRandomStringByLength(16);
            String timestamp = create_timestamp();


            String calcSign = DigestUtils.md5Hex(order.getOrderNo() + order.getPrice() + WxConfig.key);

            String sourceIp = TxtUtil.getIpAddress(request);
            String[] split = sourceIp.split(",");

            sourceIp = split[0];
            log.info("sourceIp is " + sourceIp);
            //调用微信预下单接口
            String perSign = "appid=" + WxConfig.APPID
                    + "&attach=" + calcSign
                    + "&body=" + name
                    + "&mch_id=" + WxConfig.MCH_ID
                    + "&nonce_str=" + nonce_str
                    + "&notify_url=" + WxConfig.notify_url
                    + "&openid=" + lessee.getOpenID()
                    + "&out_trade_no=" + order.getOrderNo()
                    + "&spbill_create_ip=" + sourceIp
                    + "&total_fee=" + order.getTotalPrice().multiply(new BigDecimal(100)).intValue()
                    + "&trade_type=JSAPI"
                    + "&key=" + WxConfig.key;

            log.info("perSign is : " + perSign);

            String sign = MD5.MD5Encode(perSign);

            sign = exChange(sign);

            log.info("sign is : " + sign);

            String url = "<xml>" +
                    "<appid>" + WxConfig.APPID + "</appid>" +
                    "<attach>" + calcSign + "</attach>" +
                    "<body>" + name + "</body>" +
                    "<mch_id>" + WxConfig.MCH_ID + "</mch_id>" +
                    "<nonce_str>" + nonce_str + "</nonce_str>" +
                    "<notify_url>" + WxConfig.notify_url + "</notify_url>" +
                    "<openid>" + lessee.getOpenID() + "</openid>" +
                    "<out_trade_no>" + order.getOrderNo() + "</out_trade_no>" +
                    "<spbill_create_ip>" + sourceIp + "</spbill_create_ip>" +
                    "<total_fee>" + order.getTotalPrice().multiply(new BigDecimal(100)).intValue() + "</total_fee>" +
                    "<trade_type>JSAPI</trade_type>" +
                    "<sign>" + sign + "</sign>" +
                    "</xml>";

            log.info("url is : " + url);

            String httpsresponse = httpClientUtil.sendPostRequestByJava(WxConfig.ORDER_URL, url);
            log.info("get order url result is " + httpsresponse);
            List<String> arr = new ArrayList<>();
            arr.add("prepay_id");
            Map<String, String> result = xmlUtil.xmlParams(httpsresponse, arr);
            log.info("result is " + result);
            String prepay_id = result.get("prepay_id");

            log.info("prepay_id is :" + prepay_id);

            String perPaySign = "appId=" + WxConfig.APPID
                    + "&nonceStr=" + nonce_str
                    + "&package=prepay_id=" + prepay_id
                    + "&signType=" + "MD5"
                    + "&timeStamp=" + timestamp
                    + "&key=" + WxConfig.key;

            model.addAttribute("package", "prepay_id=" + prepay_id);
            model.addAttribute("prepay_id", prepay_id);


            String paySign = MD5.MD5Encode(perPaySign);
            model.addAttribute("code", 0);
            model.addAttribute("nonceStr", nonce_str);
            model.addAttribute("signType", "MD5");
            model.addAttribute("timeStamp", timestamp);
            model.addAttribute("paySign", paySign);


        } catch (Throwable t) {
            log.error("get signature info error");
            t.printStackTrace();
        }
        return "/weixin/order/show";

    }

    /**
     * 微信支付结果查询
     *
     * @param request
     * @param response
     * @param model
     * @param oid
     * @return
     */
    @RequestMapping(value = "/a/wx/pay/result/{oid}", method = RequestMethod.GET)
    public String getPayResult(HttpServletRequest request, HttpServletResponse response, ModelMap model, @PathVariable Long oid) {
        log.info("order : " + oid + " get pay info ");

        XmlUtil xmlUtil = new XmlUtil();

        try {

            Orders order = ordersService.getObjectById(oid);
            if (null == order) {
                log.info("already pay the order");
                model.addAttribute("code", -3007);
                return "/common/success";
            }


            String nonce_str = getRandomStringByLength(16);
            String timestamp = create_timestamp();

            //调用微信订单查询接口
            String perSign = "appid=" + WxConfig.APPID
                    + "&mch_id=" + WxConfig.MCH_ID
                    + "&nonce_str=" + nonce_str
                    + "&out_trade_no=" + order.getOrderNo()
                    + "&key=" + WxConfig.key;
            String sign = MD5.MD5Encode(perSign);
            sign = exChange(sign);

            String url = "<xml>" +
                    "<appid>" + WxConfig.APPID + "</appid>" +
                    "<mch_id>" + WxConfig.MCH_ID + "</mch_id>" +
                    "<nonce_str>" + nonce_str + "</nonce_str>" +
                    "<out_trade_no>" + order.getOrderNo() + "</out_trade_no>" +
                    "<sign>" + sign + "</sign>" +
                    "</xml>";

            String httpsresponse = httpClientUtil.sendPostRequestByJava(WxConfig.orderPayResult_url, url);
            //需要返回的参数
            List<String> params = new ArrayList<>();
//            /*用户标识*/
//            params.add("openid");
            /*交易状态*/
            params.add("trade_state");
//            /*订单金额*/
//            params.add("total_fee");
            /*商户订单号*/
            params.add("out_trade_no");
//            /*支付完成时间*/
//            params.add("time_end");
            /*支付状态描述*/
            params.add("trade_state_desc");

            Map<String, String> result = xmlUtil.xmlParams(httpsresponse, params);
            log.info("get result is " + result.get("trade_state"));


            model.addAttribute("code", 0);
            model.addAttribute("result", result);
        } catch (Throwable t) {
            log.error("get signature info error");
            t.printStackTrace();
        }
        return "/weixin/order/showResult";

    }

    private static String getRandomStringByLength(int length) {
        String base = "abcdefghijklmnopqrstuvwxyz0123456789";
        Random random = new Random();
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < length; i++) {
            int number = random.nextInt(base.length());
            sb.append(base.charAt(number));
        }
        return sb.toString();
    }

    private static String exChange(String str) {
        StringBuffer sb = new StringBuffer();
        if (str != null) {
            for (int i = 0; i < str.length(); i++) {
                char ch = str.charAt(i);
                if ('a' <= ch && ch <= 'z') {
                    sb.append(Character.toUpperCase(ch));
                } else {
                    sb.append(ch);
                }
            }
        }

        return sb.toString();
    }

    private static String create_timestamp() {
        return Long.toString(System.currentTimeMillis() / 1000);
    }
}

