package com.ptteng.rent.etl;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.rent.etl.util.DynamicUtil;
import com.ptteng.rent.user.model.Orders;
import com.ptteng.rent.user.model.User;
import com.ptteng.rent.user.service.OrdersService;
import com.ptteng.rent.user.service.UserService;
import com.qding.community.common.weixin.service.WeiXinService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author cjsff
 * 2018/6/29
 * 检测进行中状态订单，订单租赁时间耗尽，转换订单状态为正常结束（结束订单）
 */
public class OrdersRunOutOfTimeEtl {

    private static final Log log = LogFactory.getLog(OrdersRunOutOfTimeEtl.class);

    private OrdersService ordersService;
    private UserService userService;
    @Autowired
    private WeiXinService weixinService;

    /**
     * 一次任务检测多少条数据
     */
    private static final int TASK_LEN = 100;

    /**
     * 空转任务间隔休息毫秒数
     */
    private static final long SLEEP_MILLISECOND = 15000;

    private Long interval = 2000L;

    public OrdersRunOutOfTimeEtl() {
        super();
    }

    public void process() throws InterruptedException {

        while (true) {

            try {

                log.info("--------------ETL 3-----------------");

                log.info("find run out of time etl is start!");

                // 查询正常结束的订单
                List<Orders> normalEndOrderList = findNormalEndOrders();

                log.info("normal end order list size :" + normalEndOrderList.size());

                if (CollectionUtils.isEmpty(normalEndOrderList)) {

                    log.info("orders push etl not get any id ,sleep " + SLEEP_MILLISECOND + " ms ");

//                    Thread.sleep(SLEEP_MILLISECOND);

                }else {
                    processOrder(normalEndOrderList);
                }
                Thread.sleep(SLEEP_MILLISECOND);


            } catch (Throwable t) {

                t.printStackTrace();

                Thread.sleep(SLEEP_MILLISECOND);

                log.error("process goods bytime status error ,sleep " + t.getMessage());

            }

        }

    }


    /**
     * 查询正常结束订单
     * @return
     */
    private List<Orders> findNormalEndOrders() {

        log.info("============------------===============");

        log.info("find normal end order method start!!!");

        List<Orders> normalEndOrderList = new ArrayList();

        List<Orders> runOutOfTimeOrders = new ArrayList<>();

        Map<String, Object> conditions = DynamicUtil.getRunOutOfTimeOrders();

        log.info("get normal end order params ；" + conditions);

        try {

            List<Long> normalEndOrderIds = this.ordersService.getIdsByDynamicCondition(Orders.class, conditions,
                    0, TASK_LEN);

            if (CollectionUtils.isNotEmpty(normalEndOrderIds)) {

                normalEndOrderList = this.ordersService.getObjectsByIds(normalEndOrderIds);

            }

            for (Orders orders : normalEndOrderList) {

                if (orders.getLeaseEndAt() < System.currentTimeMillis()) {

                    runOutOfTimeOrders.add(orders);

                }

            }


        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t);
        }

        return runOutOfTimeOrders;
    }


    /**
     * 给出租人转账
     * @param normaoEndOrderList
     * @throws ServiceException
     * @throws ServiceDaoException
     * @throws InterruptedException
     * @throws IOException
     */
    private void processOrder(List<Orders> normaoEndOrderList) throws ServiceException, ServiceDaoException,
            InterruptedException,
            IOException {

        log.info("***********************************");

        log.info("Transfer to lessor");

        if (CollectionUtils.isNotEmpty(normaoEndOrderList)) {
            List<Long> lessorIds = new ArrayList<>();
            List<Long> lesseeIds = new ArrayList<>();
            for (Orders orders : normaoEndOrderList) {
                lesseeIds.add(orders.getLesseeId());
                lessorIds.add(orders.getLessorId());
                orders.setStatus(Orders.COMPLETE_THE_ORDER_NORMALLY);

                orders.setLessorEvaluationStatus(Orders.NOT_EVALUATION);

                orders.setLesseeEvaluationStatus(Orders.NOT_EVALUATION);

                orders.setEndOfOrderAt(System.currentTimeMillis());

            }

            boolean boo = ordersService.updateList(normaoEndOrderList);

            if (boo) {

                log.info("order update status success");
                //订单结束后，更新用户订单数量
                List<User> lessorList = userService.getObjectsByIds(lessorIds);
                for (User lessor : lessorList) {
                    lessor.setLessorOrders(lessor.getLessorOrders() + User.NEW_ORDER);
                }

                List<User> lesseeList = userService.getObjectsByIds(lesseeIds);
                for (User lessee : lesseeList) {
                    lessee.setLesseeOrders(lessee.getLesseeOrders() + User.NEW_ORDER);
                }

                lesseeList.addAll(lessorList);
                boo = userService.updateList(lesseeList);
                log.info("user update status " + boo);
            } else {

                log.info("order update status fail");

                log.info("-------------------------");

            }

        }

    }


    public OrdersService getOrdersService() {
        return ordersService;
    }

    public void setOrdersService(OrdersService ordersService) {
        this.ordersService = ordersService;
    }

    public Long getInterval() {
        return interval;
    }

    public void setInterval(Long interval) {
        this.interval = interval;
    }

    public UserService getUserService() {
        return userService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

}
