package com.ptteng.keeper.home.controller;

import com.gemantic.common.util.MyTimeUtil;
import com.ptteng.keeper.common.constants.Constants;
import com.ptteng.keeper.common.model.ApplyingRecord;
import com.ptteng.keeper.common.model.Product;
import com.ptteng.keeper.common.service.ApplyingRecordService;
import com.ptteng.keeper.common.service.ProductService;
import com.ptteng.keeper.common.service.UserService;
import com.ptteng.keeper.home.util.DynamicUtil;
import com.ptteng.keeper.home.util.LogUtil;
import com.ptteng.keeper.home.util.StringUtil;
import com.qding.common.util.DataUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Product  crud
 *
 * @author magenm
 * @Date 2014-4-16 13:43
 */
@Controller
public class ProductController {
    private static final Log log = LogFactory.getLog(ProductController.class);

    @Autowired
    private ProductService productService;
    @Autowired
    private UserService userService;
    @Autowired
    private ApplyingRecordService applyingRecordService;

    /**
     * show 产品详情
     *
     * @param id 产品id
     * @return product json
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/product/{id}", method = RequestMethod.GET)
    public String getProductDetail(HttpServletRequest request,
                                   HttpServletResponse response, ModelMap model,
                                   @PathVariable Long id) throws Exception {

        log.info(" getProductDetail: id= " + id);
        if (DataUtils.isNullOrEmpty(id)) {
            model.addAttribute("code", -1100);
            return "/common/success";
        }
        try {
            Product product = productService.getObjectById(id);
            if (DataUtils.isNullOrEmpty(product)) {
                model.addAttribute("code", -10);
                log.info("Data does not exist,id is " + id);
                return "/common/success";
            }
            model.addAttribute("code", 0);

            model.addAttribute("product", product);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get product list error");
            // for test
            model.addAttribute("code", -100000);
        }

        return "/keeper/product/json/productDetailJson";
    }


    /**
     * show 产品列表
     *
     * @param type 产品类型 1贷款 2信用卡
     * @return product array
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/u/product/list", method = RequestMethod.GET)
    public String getProductList(HttpServletRequest request,
                                 HttpServletResponse response, ModelMap model,
                                 Integer type
    ) throws Exception {

        log.info("/a/u/product/list: type= " + type);
        if (DataUtils.isNullOrEmpty(type)) {
            type = Product.TYPE_LENDING;
        }
        try {
            List<Product> productList = new ArrayList<>();
            List<Long> ids = productService.getProductIdsByStatusAndTypeOrderByStatusChangeAt(Product.STATUS_UP, type, 0, Integer.MAX_VALUE);
            if (ids.size()>0){
                productList = productService.getObjectsByIds(ids);
            }
            log.info("get product list size is" + ids.size());

            model.addAttribute("code", 0);
            model.addAttribute("productList", productList);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get product list error,type is  " + type);
            // for test
            model.addAttribute("code", -100000);
        }

        return "/keeper/product/json/productListJson";
    }


    /**
     * show 申请产品
     *
     * @param invitationCode 邀请码
     * @param amount         额度
     * @param deadline       贷款期限
     * @param name           姓名
     * @param idCard         身份证
     * @param mobile         手机号
     * @param productId      手机号
     * @return 申请结果
     * @throws Exception 异常信息
     * @author lishaobo
     * @date 2018/8/4
     */
    @RequestMapping(value = "/a/product", method = RequestMethod.POST)
    public String addProductJson(HttpServletRequest request,
                                 HttpServletResponse response, ModelMap model, String invitationCode, BigDecimal amount, Integer deadline, String name,
                                 String idCard, String mobile, Long productId) throws Exception {

        log.info("apply product :" + LogUtil.convertParam("invitationCode", invitationCode, "amount", amount, "deadline",
                deadline, "name", name, "idCard", idCard, "mobile", mobile, "productId", productId));
        if (DataUtils.isNullOrEmpty(productId)) {
            model.addAttribute("code", -3000);
            return "common/success";
        }
        if (DataUtils.isNullOrEmpty(name)) {
            model.addAttribute("code", -3001);
            return "common/success";
        }
        if (DataUtils.isNullOrEmpty(mobile)) {
            model.addAttribute("code", -3002);
            return "common/success";
        }
        if (DataUtils.isNullOrEmpty(idCard)) {
            model.addAttribute("code", -3003);
            return "common/success";
        }
        if ( !StringUtil.isIdCard(idCard)){
            model.addAttribute("code", -3009);
            return "common/success";

        }
        try {
            ApplyingRecord applyingRecord = new ApplyingRecord();
            applyingRecord.setName(name);
            applyingRecord.setMobile(mobile);
            applyingRecord.setIdCard(idCard);
            applyingRecord.setStatus(ApplyingRecord.STATUS_APPLYING);

            Product product = productService.getObjectById(productId);
            if (DataUtils.isNullOrEmpty(product)) {
                model.addAttribute("code", -10);
                return "common/success";
            }
            //类型
            if (product.getType().equals(Product.TYPE_LENDING)) {
                if (DataUtils.isNullOrEmpty(amount)) {
                    model.addAttribute("code", -3004);
                    return "common/success";
                }
                if (DataUtils.isNullOrEmpty(deadline)) {
                    model.addAttribute("code", -3005);
                    return "common/success";
                } else {
                    //是否在额度范围内
                    log.info("amount should be in " + product.getAmountStart() + " - " + product.getAmountEnd());
                    if (amount.compareTo(product.getAmountStart()) < 0 || amount.compareTo(product.getAmountEnd()) > 0) {
                        model.addAttribute("code", -3006);
                        return "common/success";
                    }
                    log.info("deadline should be in " + product.getDeadlineStart() + " - " + product.getDeadlineEnd());

                    //是否在期限范围
                    if (deadline < product.getDeadlineStart() || deadline > product.getDeadlineEnd()) {
                        model.addAttribute("code", -3007);
                        return "common/success";
                    }

                }
                applyingRecord.setAmount(amount);
                applyingRecord.setDeadline(deadline);
            }
            applyingRecord.setProductId(productId);
            applyingRecord.setType(product.getType());
            if (Product.REBATE_MODE_AMOUNT.equals(product.getRebateMode())){
                applyingRecord.setVerifyAmount(product.getRebateNum());
            }
            Long start = MyTimeUtil.getTodayZeroTimeMillions();
            Long end = start + 86399999;
            //判断身份证/手机号/产品申请记录
            Map<String, Object> map = DynamicUtil.getApplyRecordByMobileAndIdCardAndProductId(mobile, idCard, productId, start, end, true);
            BigInteger total = (BigInteger) applyingRecordService.getObjectByDynamicCondition(ApplyingRecord.class, map, 0, Integer.MAX_VALUE);
            log.info("mobile,idCard,productId record total is " + total);
            if (total.compareTo(BigInteger.TEN) >= 0) {
                model.addAttribute("code", -3008);
                return "/common/success";
            }

            //邀请码
            if (DataUtils.isNotNullOrEmpty(invitationCode)) {
                Long uid = userService.getUserIdByInvitationCode(invitationCode);
                if (DataUtils.isNullOrEmpty(uid)) {
                    model.addAttribute("code", -2010);
                    return "common/success";
                }
                log.info("recommend user id is " + uid);
                //记录邀请人信息
                applyingRecord.setCreateBy(uid);
                applyingRecord.setUpdateBy(uid);
            } else {
                applyingRecord.setCreateBy(Constants.DEFAULT_LONG_NONE);
                applyingRecord.setUpdateBy(Constants.DEFAULT_LONG_NONE);
            }
            applyingRecord.setProxyAmount(BigDecimal.ZERO);
            applyingRecord.setVerifyAmount(BigDecimal.valueOf(-1));
            applyingRecord.setFirstProxyAmount(BigDecimal.ZERO);
            applyingRecord.setSecondProxyAmount(BigDecimal.ZERO);

            Long resultId = applyingRecordService.insert(applyingRecord);
            product.setApplyCount(product.getApplyCount() + 1);
            productService.update(product);
            log.info("create apply record success ,id is " + resultId + " ,product apply count -> " + product.getApplyCount());
            model.addAttribute("code", 0);
        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("add product error ");
            model.addAttribute("code", -1);
        }

        return "/common/success";
    }


}

