/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je;

import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.LockMode;
import com.sleepycat.je.OperationStatus;
import com.sleepycat.je.SequenceConfig;
import com.sleepycat.je.SequenceStats;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.Transaction;
import com.sleepycat.je.TransactionConfig;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.txn.Locker;
import com.sleepycat.je.txn.LockerFactory;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.logging.Level;
import java.util.logging.Logger;

public class Sequence {
    private static final byte FLAG_INCR = 1;
    private static final byte FLAG_WRAP = 2;
    private static final byte FLAG_OVER = 4;
    private static final int MAX_DATA_SIZE = 50;
    private static final byte CURRENT_VERSION = 1;
    private Database db;
    private DatabaseEntry key;
    private boolean wrapAllowed;
    private boolean increment;
    private boolean overflow;
    private long rangeMin;
    private long rangeMax;
    private long storedValue;
    private int cacheSize;
    private long cacheValue;
    private long cacheLast;
    private int nGets;
    private int nCachedGets;
    private TransactionConfig autoCommitConfig;
    private Logger logger;

    Sequence(Database db, Transaction txn, DatabaseEntry key, SequenceConfig config) throws DatabaseException {
        SequenceConfig useConfig;
        if (db.getDatabaseImpl().getSortedDuplicates()) {
            throw new IllegalArgumentException("Sequences not supported in databases configured for duplicates");
        }
        SequenceConfig sequenceConfig = useConfig = config != null ? config : SequenceConfig.DEFAULT;
        if (useConfig.getRangeMin() >= useConfig.getRangeMax()) {
            throw new IllegalArgumentException("Minimum sequence value must be less than the maximum");
        }
        if (useConfig.getInitialValue() > useConfig.getRangeMax() || useConfig.getInitialValue() < useConfig.getRangeMin()) {
            throw new IllegalArgumentException("Initial sequence value is out of range");
        }
        if (useConfig.getRangeMin() > useConfig.getRangeMax() - (long)useConfig.getCacheSize()) {
            throw new IllegalArgumentException("The cache size is larger than the sequence range");
        }
        if (useConfig.getAutoCommitNoSync()) {
            this.autoCommitConfig = new TransactionConfig();
            this.autoCommitConfig.setNoSync(true);
        } else {
            this.autoCommitConfig = null;
        }
        this.db = db;
        this.key = this.copyEntry(key);
        this.logger = db.getEnvironment().getEnvironmentImpl().getLogger();
        Locker locker = null;
        Cursor cursor = null;
        OperationStatus status = OperationStatus.NOTFOUND;
        try {
            locker = LockerFactory.getWritableLocker(db.getEnvironment(), txn, db.isTransactional(), false, db.getDatabaseImpl().isReplicated(), this.autoCommitConfig);
            cursor = new Cursor(db, locker, null);
            if (useConfig.getAllowCreate()) {
                this.rangeMin = useConfig.getRangeMin();
                this.rangeMax = useConfig.getRangeMax();
                this.increment = !useConfig.getDecrement();
                this.wrapAllowed = useConfig.getWrap();
                this.storedValue = useConfig.getInitialValue();
                status = cursor.putNoOverwrite(key, this.makeData());
                if (status == OperationStatus.KEYEXIST) {
                    if (useConfig.getExclusiveCreate()) {
                        throw new DatabaseException("ExclusiveCreate=true and the sequence record already exists.");
                    }
                    if (!this.readData(cursor, null)) {
                        throw new DatabaseException("Sequence record removed during openSequence.");
                    }
                    status = OperationStatus.SUCCESS;
                }
            } else {
                if (!this.readData(cursor, null)) {
                    throw new DatabaseException("AllowCreate=false and the sequence record does not exist.");
                }
                status = OperationStatus.SUCCESS;
            }
        }
        finally {
            if (cursor != null) {
                cursor.close();
            }
            if (locker != null) {
                locker.operationEnd(status);
            }
        }
        this.cacheSize = useConfig.getCacheSize();
        this.cacheValue = this.storedValue;
        this.cacheLast = this.increment ? this.storedValue - 1L : this.storedValue + 1L;
    }

    public void close() throws DatabaseException {
    }

    public synchronized long get(Transaction txn, int delta) throws DatabaseException {
        if (delta <= 0) {
            throw new IllegalArgumentException("Sequence delta must be greater than zero");
        }
        if (this.rangeMin > this.rangeMax - (long)delta) {
            throw new IllegalArgumentException("Sequence delta is larger than the range");
        }
        boolean cached = true;
        boolean wrapped = false;
        if (this.increment && (long)delta > this.cacheLast - this.cacheValue + 1L || !this.increment && (long)delta > this.cacheValue - this.cacheLast + 1L) {
            cached = false;
            int adjust = delta > this.cacheSize ? delta : this.cacheSize;
            Locker locker = null;
            Cursor cursor = null;
            OperationStatus status = OperationStatus.NOTFOUND;
            try {
                locker = LockerFactory.getWritableLocker(this.db.getEnvironment(), txn, this.db.isTransactional(), false, this.db.getDatabaseImpl().isReplicated(), this.autoCommitConfig);
                cursor = new Cursor(this.db, locker, null);
                this.readDataRequired(cursor, LockMode.RMW);
                if (this.overflow) {
                    throw new DatabaseException("Sequence overflow " + this.storedValue);
                }
                BigInteger availBig = this.increment ? BigInteger.valueOf(this.rangeMax).subtract(BigInteger.valueOf(this.storedValue)) : BigInteger.valueOf(this.storedValue).subtract(BigInteger.valueOf(this.rangeMin));
                if (availBig.compareTo(BigInteger.valueOf(adjust)) < 0) {
                    int availInt = (int)availBig.longValue();
                    if (availInt < delta) {
                        if (this.wrapAllowed) {
                            this.storedValue = this.increment ? this.rangeMin : this.rangeMax;
                            wrapped = true;
                        } else {
                            this.overflow = true;
                            adjust = 0;
                        }
                    } else {
                        adjust = availInt;
                    }
                }
                if (!this.increment) {
                    adjust = -adjust;
                }
                this.storedValue += (long)adjust;
                cursor.put(this.key, this.makeData());
                status = OperationStatus.SUCCESS;
            }
            finally {
                if (cursor != null) {
                    cursor.close();
                }
                if (locker != null) {
                    locker.operationEnd(status);
                }
            }
            this.cacheValue = this.storedValue - (long)adjust;
            this.cacheLast = this.storedValue + (long)(this.increment ? -1 : 1);
        }
        long retVal = this.cacheValue;
        this.cacheValue = this.increment ? (this.cacheValue += (long)delta) : (this.cacheValue -= (long)delta);
        ++this.nGets;
        if (cached) {
            ++this.nCachedGets;
        }
        if (this.logger.isLoggable(Level.FINEST)) {
            this.logger.log(Level.FINEST, "Sequence.get value=" + retVal + " cached=" + cached + " wrapped=" + wrapped);
        }
        return retVal;
    }

    public Database getDatabase() throws DatabaseException {
        return this.db;
    }

    public DatabaseEntry getKey() throws DatabaseException {
        return this.copyEntry(this.key);
    }

    public SequenceStats getStats(StatsConfig config) throws DatabaseException {
        if (config == null) {
            config = StatsConfig.DEFAULT;
        }
        if (!config.getFast()) {
            Cursor cursor = this.db.openCursor(null, null);
            try {
                this.readDataRequired(cursor, LockMode.READ_UNCOMMITTED);
            }
            finally {
                cursor.close();
            }
        }
        SequenceStats stats = new SequenceStats(this.nGets, this.nCachedGets, this.storedValue, this.cacheValue, this.cacheLast, this.rangeMin, this.rangeMax, this.cacheSize);
        if (config.getClear()) {
            this.nGets = 0;
            this.nCachedGets = 0;
        }
        return stats;
    }

    private void readDataRequired(Cursor cursor, LockMode lockMode) throws DatabaseException {
        if (!this.readData(cursor, lockMode)) {
            throw new DatabaseException("The sequence record has been deleted while it is open.");
        }
    }

    private boolean readData(Cursor cursor, LockMode lockMode) throws DatabaseException {
        DatabaseEntry data = new DatabaseEntry();
        OperationStatus status = cursor.getSearchKey(this.key, data, lockMode);
        if (status != OperationStatus.SUCCESS) {
            return false;
        }
        ByteBuffer buf = ByteBuffer.wrap(data.getData());
        byte version = buf.get();
        byte flags = buf.get();
        boolean unpacked = version < 1;
        this.rangeMin = LogUtils.readLong(buf, unpacked);
        this.rangeMax = LogUtils.readLong(buf, unpacked);
        this.storedValue = LogUtils.readLong(buf, unpacked);
        this.increment = (flags & 1) != 0;
        this.wrapAllowed = (flags & 2) != 0;
        this.overflow = (flags & 4) != 0;
        return true;
    }

    private DatabaseEntry makeData() {
        byte[] data = new byte[50];
        ByteBuffer buf = ByteBuffer.wrap(data);
        byte flags = 0;
        if (this.increment) {
            flags = (byte)(flags | 1);
        }
        if (this.wrapAllowed) {
            flags = (byte)(flags | 2);
        }
        if (this.overflow) {
            flags = (byte)(flags | 4);
        }
        buf.put((byte)1);
        buf.put(flags);
        LogUtils.writePackedLong(buf, this.rangeMin);
        LogUtils.writePackedLong(buf, this.rangeMax);
        LogUtils.writePackedLong(buf, this.storedValue);
        return new DatabaseEntry(data, 0, buf.position());
    }

    private DatabaseEntry copyEntry(DatabaseEntry entry) {
        byte[] data;
        int len = entry.getSize();
        if (len == 0) {
            data = LogUtils.ZERO_LENGTH_BYTE_ARRAY;
        } else {
            data = new byte[len];
            System.arraycopy(entry.getData(), entry.getOffset(), data, 0, data.length);
        }
        return new DatabaseEntry(data);
    }
}

