/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-redis-js/response */
var utils = require('vertx-js/util/utils');
var Buffer = require('vertx-js/buffer');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JResponse = Java.type('io.vertx.redis.client.Response');

/**
 The response received from the REDIS server. Redis responses can have several representations:

 <ul>
     <li>simple string - C string</li>
     <li>integer - 64bit integer value</li>
     <li>bulk - byte array</li>
     <li>multi - list</li>
 </ul>

 Due to the dynamic nature the response object will try to cast the received response to the desired type. A special
 case should be noted that multi responses are also handled by the response object as it implements the iterable
 interface. So in this case constructs like for loops on the response will give you access to the underlying elements.

 @class
*/
var Response = function(j_val) {

  var j_response = j_val;
  var that = this;

  var __super_type = this.type;
  var __super_toString = this.toString;
  var __super_toLong = this.toLong;
  var __super_toInteger = this.toInteger;
  var __super_toShort = this.toShort;
  var __super_toByte = this.toByte;
  var __super_toBoolean = this.toBoolean;
  var __super_toBuffer = this.toBuffer;
  var __super_get = this.get;
  var __super_get = this.get;
  var __super_getKeys = this.getKeys;
  var __super_size = this.size;
  /**
   The response return type.

   @public

   @return {Object} the type.
   */
  this.type =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnEnum(j_response["type()"]()) ;
    } else if (typeof __super_type != 'undefined') {
      return __super_type.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a String.

   @public

   @return {string} string value
   */
  this.toString =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["toString()"]() ;
    } else if (typeof __super_toString != 'undefined') {
      return __super_toString.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a Long.

   @public

   @return {number} long value.
   */
  this.toLong =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnLong(j_response["toLong()"]()) ;
    } else if (typeof __super_toLong != 'undefined') {
      return __super_toLong.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a Integer.

   @public

   @return {number} int value.
   */
  this.toInteger =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["toInteger()"]() ;
    } else if (typeof __super_toInteger != 'undefined') {
      return __super_toInteger.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a Short.

   @public

   @return {number} short value.
   */
  this.toShort =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["toShort()"]() ;
    } else if (typeof __super_toShort != 'undefined') {
      return __super_toShort.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a Byte.

   @public

   @return {number} byte value.
   */
  this.toByte =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["toByte()"]() ;
    } else if (typeof __super_toByte != 'undefined') {
      return __super_toByte.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as a Boolean.

   @public

   @return {boolean} boolean value.
   */
  this.toBoolean =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["toBoolean()"]() ;
    } else if (typeof __super_toBoolean != 'undefined') {
      return __super_toBoolean.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this response as Buffer.

   @public

   @return {Buffer} buffer value.
   */
  this.toBuffer =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Buffer, j_response["toBuffer()"]()) ;
    } else if (typeof __super_toBuffer != 'undefined') {
      return __super_toBuffer.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this multi response value at a string key. Note that REDIS does not support strings as keys but by convention
   it encodes hashes in lists where index i is the key, and index i+1 is the value.

   @public
   @param key {string} the required key. 
   @return {Response} Response value.
   */
  this.get =  function() {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='number') {
      return utils.convReturnVertxGen(Response, j_response["get(int)"](__args[0])) ;
    } else if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(Response, j_response["get(java.lang.String)"](__args[0])) ;
    } else if (typeof __super_get != 'undefined') {
      return __super_get.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this multi response keys from a hash. Note that REDIS does not support strings as keys but by convention
   it encodes hashes in lists where index i is the key, and index i+1 is the value.

   @public

   @return {Array.<string>} the set of keys.
   */
  this.getKeys =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnSet(j_response["getKeys()"]()) ;
    } else if (typeof __super_getKeys != 'undefined') {
      return __super_getKeys.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get this size of this multi response.

   @public

   @return {number} the size of the multi.
   */
  this.size =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_response["size()"]() ;
    } else if (typeof __super_size != 'undefined') {
      return __super_size.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_response;
};

Response._jclass = utils.getJavaClass("io.vertx.redis.client.Response");
Response._jtype = {accept: function(obj) {
    return Response._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(Response.prototype, {});
    Response.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
Response._create = function(jdel) {var obj = Object.create(Response.prototype, {});
  Response.apply(obj, arguments);
  return obj;
}
module.exports = Response;