/*
 * Decompiled with CFR 0.152.
 */
package org.apache.falcon.entity;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.jsp.el.ELException;
import org.apache.commons.lang3.StringUtils;
import org.apache.falcon.FalconException;
import org.apache.falcon.Pair;
import org.apache.falcon.entity.ClusterHelper;
import org.apache.falcon.entity.EntityUtil;
import org.apache.falcon.entity.FeedHelper;
import org.apache.falcon.entity.FeedInstanceStatus;
import org.apache.falcon.entity.Storage;
import org.apache.falcon.entity.common.FeedDataPath;
import org.apache.falcon.entity.v0.AccessControlList;
import org.apache.falcon.entity.v0.SchemaHelper;
import org.apache.falcon.entity.v0.cluster.Cluster;
import org.apache.falcon.entity.v0.feed.Feed;
import org.apache.falcon.entity.v0.feed.Location;
import org.apache.falcon.entity.v0.feed.LocationType;
import org.apache.falcon.entity.v0.feed.Locations;
import org.apache.falcon.expression.ExpressionHelper;
import org.apache.falcon.hadoop.HadoopClientFactory;
import org.apache.falcon.retention.EvictedInstanceSerDe;
import org.apache.falcon.retention.EvictionHelper;
import org.apache.falcon.security.CurrentUser;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemStorage
extends Configured
implements Storage {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemStorage.class);
    private final StringBuffer instancePaths = new StringBuffer();
    private final StringBuilder instanceDates = new StringBuilder();
    public static final String FEED_PATH_SEP = "#";
    public static final String LOCATION_TYPE_SEP = "=";
    public static final String FILE_SYSTEM_URL = "${nameNode}";
    private final String storageUrl;
    private final List<Location> locations;

    public FileSystemStorage(Feed feed) {
        this(FILE_SYSTEM_URL, feed.getLocations());
    }

    protected FileSystemStorage(String storageUrl, Locations locations) {
        this(storageUrl, locations.getLocations());
    }

    protected FileSystemStorage(String storageUrl, List<Location> locations) {
        if (storageUrl == null || storageUrl.length() == 0) {
            throw new IllegalArgumentException("FileSystem URL cannot be null or empty");
        }
        if (locations == null || locations.size() == 0) {
            throw new IllegalArgumentException("FileSystem Locations cannot be null or empty");
        }
        this.storageUrl = storageUrl;
        this.locations = locations;
    }

    protected FileSystemStorage(String uriTemplate) throws URISyntaxException {
        String[] feedLocs;
        if (uriTemplate == null || uriTemplate.length() == 0) {
            throw new IllegalArgumentException("URI template cannot be null or empty");
        }
        String rawStorageUrl = null;
        ArrayList<Location> rawLocations = new ArrayList<Location>();
        for (String rawPath : feedLocs = uriTemplate.split(FEED_PATH_SEP)) {
            String[] typeAndPath = rawPath.split(LOCATION_TYPE_SEP);
            String processed = typeAndPath[1].replaceAll("\\$\\{", "_D__START_").replaceAll("}", "_CLOSE_");
            URI uri = new URI(processed);
            if (rawStorageUrl == null) {
                rawStorageUrl = uri.getScheme() + "://" + uri.getAuthority();
            }
            String path = uri.getPath();
            String finalPath = path.replaceAll("_D__START_", "\\$\\{").replaceAll("_CLOSE_", "\\}");
            Location location = new Location();
            location.setPath(finalPath);
            location.setType(LocationType.valueOf((String)typeAndPath[0]));
            rawLocations.add(location);
        }
        this.storageUrl = rawStorageUrl;
        this.locations = rawLocations;
    }

    @Override
    public Storage.TYPE getType() {
        return Storage.TYPE.FILESYSTEM;
    }

    public String getStorageUrl() {
        return this.storageUrl;
    }

    public List<Location> getLocations() {
        return this.locations;
    }

    @Override
    public String getUriTemplate() {
        String feedPathMask = this.getUriTemplate(LocationType.DATA);
        String metaPathMask = this.getUriTemplate(LocationType.META);
        String statsPathMask = this.getUriTemplate(LocationType.STATS);
        String tmpPathMask = this.getUriTemplate(LocationType.TMP);
        StringBuilder feedBasePaths = new StringBuilder();
        feedBasePaths.append(LocationType.DATA.name()).append(LOCATION_TYPE_SEP).append(feedPathMask);
        if (metaPathMask != null) {
            feedBasePaths.append(FEED_PATH_SEP).append(LocationType.META.name()).append(LOCATION_TYPE_SEP).append(metaPathMask);
        }
        if (statsPathMask != null) {
            feedBasePaths.append(FEED_PATH_SEP).append(LocationType.STATS.name()).append(LOCATION_TYPE_SEP).append(statsPathMask);
        }
        if (tmpPathMask != null) {
            feedBasePaths.append(FEED_PATH_SEP).append(LocationType.TMP.name()).append(LOCATION_TYPE_SEP).append(tmpPathMask);
        }
        return feedBasePaths.toString();
    }

    @Override
    public String getUriTemplate(LocationType locationType) {
        return this.getUriTemplate(locationType, this.locations);
    }

    public String getUriTemplate(LocationType locationType, List<Location> locationList) {
        Location locationForType = null;
        for (Location location : locationList) {
            if (location.getType() != locationType) continue;
            locationForType = location;
            break;
        }
        if (locationForType == null || StringUtils.isEmpty((CharSequence)locationForType.getPath())) {
            return null;
        }
        Path locationPath = new Path(locationForType.getPath());
        if (this.isRelativePath(locationPath = locationPath.makeQualified(this.getDefaultUri(), this.getWorkingDir()))) {
            locationPath = new Path(this.storageUrl + locationPath);
        }
        return locationPath.toString();
    }

    private boolean isRelativePath(Path locationPath) {
        return locationPath.toUri().getAuthority() == null && this.isStorageUrlATemplate();
    }

    private boolean isStorageUrlATemplate() {
        return this.storageUrl.startsWith(FILE_SYSTEM_URL);
    }

    private URI getDefaultUri() {
        return new Path(this.isStorageUrlATemplate() ? "/" : this.storageUrl).toUri();
    }

    public Path getWorkingDir() {
        return new Path(CurrentUser.isAuthenticated() ? "/user/" + CurrentUser.getUser() : "/");
    }

    @Override
    public boolean isIdentical(Storage toCompareAgainst) throws FalconException {
        if (!(toCompareAgainst instanceof FileSystemStorage)) {
            return false;
        }
        FileSystemStorage fsStorage = (FileSystemStorage)toCompareAgainst;
        List<Location> fsStorageLocations = fsStorage.getLocations();
        return this.getLocations().size() == fsStorageLocations.size() && StringUtils.equals((CharSequence)this.getUriTemplate(LocationType.DATA, this.getLocations()), (CharSequence)this.getUriTemplate(LocationType.DATA, fsStorageLocations)) && StringUtils.equals((CharSequence)this.getUriTemplate(LocationType.STATS, this.getLocations()), (CharSequence)this.getUriTemplate(LocationType.STATS, fsStorageLocations)) && StringUtils.equals((CharSequence)this.getUriTemplate(LocationType.META, this.getLocations()), (CharSequence)this.getUriTemplate(LocationType.META, fsStorageLocations)) && StringUtils.equals((CharSequence)this.getUriTemplate(LocationType.TMP, this.getLocations()), (CharSequence)this.getUriTemplate(LocationType.TMP, fsStorageLocations));
    }

    public static Location getLocation(List<Location> locations, LocationType type) {
        for (Location loc : locations) {
            if (loc.getType() != type) continue;
            return loc;
        }
        return null;
    }

    @Override
    public void validateACL(AccessControlList acl) throws FalconException {
        try {
            for (Location location : this.getLocations()) {
                String pathString = this.getRelativePath(location);
                Path path = new Path(pathString);
                FileSystem fileSystem = HadoopClientFactory.get().createProxiedFileSystem(path.toUri(), this.getConf());
                if (!fileSystem.exists(path)) continue;
                FileStatus fileStatus = fileSystem.getFileStatus(path);
                Set<String> groups = CurrentUser.getGroupNames();
                if (fileStatus.getOwner().equals(acl.getOwner()) || groups.contains(acl.getGroup())) {
                    return;
                }
                LOG.error("Permission denied: Either Feed ACL owner {} or group {} doesn't match the actual file owner {} or group {} for file {}", new Object[]{acl, acl.getGroup(), fileStatus.getOwner(), fileStatus.getGroup(), path});
                throw new FalconException("Permission denied: Either Feed ACL owner " + acl + " or group " + acl.getGroup() + " doesn't match the actual " + "file owner " + fileStatus.getOwner() + " or group " + fileStatus.getGroup() + "  for file " + path);
            }
        }
        catch (IOException e) {
            LOG.error("Can't validate ACL on storage {}", (Object)this.getStorageUrl(), (Object)e);
            throw new RuntimeException("Can't validate storage ACL (URI " + this.getStorageUrl() + ")", e);
        }
    }

    @Override
    public StringBuilder evict(String retentionLimit, String timeZone, Path logFilePath) throws FalconException {
        TimeZone tz = TimeZone.getTimeZone(timeZone);
        try {
            for (Location location : this.getLocations()) {
                this.fileSystemEvictor(this.getUriTemplate(location.getType()), retentionLimit, tz, logFilePath);
            }
            EvictedInstanceSerDe.serializeEvictedInstancePaths(HadoopClientFactory.get().createProxiedFileSystem(logFilePath.toUri(), this.getConf()), logFilePath, this.instancePaths);
        }
        catch (IOException e) {
            throw new FalconException("Couldn't evict feed from fileSystem", e);
        }
        catch (ELException e) {
            throw new FalconException("Couldn't evict feed from fileSystem", e);
        }
        return this.instanceDates;
    }

    private void fileSystemEvictor(String feedPath, String retentionLimit, TimeZone timeZone, Path logFilePath) throws IOException, ELException, FalconException {
        Path normalizedPath = new Path(feedPath);
        FileSystem fs = HadoopClientFactory.get().createProxiedFileSystem(normalizedPath.toUri());
        feedPath = normalizedPath.toUri().getPath();
        LOG.info("Normalized path: {}", (Object)feedPath);
        Pair<Date, Date> range = EvictionHelper.getDateRange(retentionLimit);
        List<Path> toBeDeleted = this.discoverInstanceToDelete(feedPath, timeZone, (Date)range.first, fs);
        if (toBeDeleted.isEmpty()) {
            LOG.info("No instances to delete.");
            return;
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmm");
        dateFormat.setTimeZone(timeZone);
        Path feedBasePath = fs.makeQualified(FeedHelper.getFeedBasePath(feedPath));
        for (Path path : toBeDeleted) {
            this.deleteInstance(fs, path, feedBasePath);
            Date date = FeedHelper.getDate(feedPath, new Path(path.toUri().getPath()), timeZone);
            this.instanceDates.append(dateFormat.format(date)).append(',');
            this.instancePaths.append(path).append(",");
        }
    }

    private List<Path> discoverInstanceToDelete(String inPath, TimeZone timeZone, Date start, FileSystem fs) throws IOException {
        FileStatus[] files = this.findFilesForFeed(fs, inPath);
        if (files == null || files.length == 0) {
            return Collections.emptyList();
        }
        ArrayList<Path> toBeDeleted = new ArrayList<Path>();
        for (FileStatus file : files) {
            Date date = FeedHelper.getDate(inPath, new Path(file.getPath().toUri().getPath()), timeZone);
            LOG.debug("Considering {}", (Object)file.getPath().toUri().getPath());
            LOG.debug("Date: {}", (Object)date);
            if (date == null || this.isDateInRange(date, start)) continue;
            toBeDeleted.add(file.getPath());
        }
        return toBeDeleted;
    }

    private FileStatus[] findFilesForFeed(FileSystem fs, String feedBasePath) throws IOException {
        Matcher matcher = FeedDataPath.PATTERN.matcher(feedBasePath);
        while (matcher.find()) {
            String var = feedBasePath.substring(matcher.start(), matcher.end());
            feedBasePath = feedBasePath.replaceAll(Pattern.quote(var), "*");
            matcher = FeedDataPath.PATTERN.matcher(feedBasePath);
        }
        LOG.info("Searching for {}", (Object)feedBasePath);
        return fs.globStatus(new Path(feedBasePath));
    }

    private boolean isDateInRange(Date date, Date start) {
        return date.compareTo(start) >= 0;
    }

    private void deleteInstance(FileSystem fs, Path path, Path feedBasePath) throws IOException {
        if (!fs.delete(path, true)) {
            throw new IOException("Unable to delete instance: " + path);
        }
        LOG.info("Deleted instance: {}", (Object)path);
        this.deleteParentIfEmpty(fs, path.getParent(), feedBasePath);
    }

    private void deleteParentIfEmpty(FileSystem fs, Path parent, Path feedBasePath) throws IOException {
        if (feedBasePath.equals((Object)parent)) {
            LOG.info("Not deleting feed base path: {}", (Object)parent);
        } else {
            FileStatus[] files = fs.listStatus(parent);
            if (files != null && files.length == 0) {
                LOG.info("Parent path: {} is empty, deleting path", (Object)parent);
                if (!fs.delete(parent, true)) {
                    throw new IOException("Unable to delete parent path:" + parent);
                }
                LOG.info("Deleted empty dir: {}", (Object)parent);
                this.deleteParentIfEmpty(fs, parent.getParent(), feedBasePath);
            }
        }
    }

    @Override
    public List<FeedInstanceStatus> getListing(Feed feed, String clusterName, LocationType locationType, Date start, Date end) throws FalconException {
        Calendar calendar = Calendar.getInstance();
        List<Location> clusterSpecificLocation = FeedHelper.getLocations(FeedHelper.getCluster(feed, clusterName), feed);
        Location location = FileSystemStorage.getLocation(clusterSpecificLocation, locationType);
        try {
            FileSystem fileSystem = HadoopClientFactory.get().createProxiedFileSystem(this.getConf());
            Cluster cluster = ClusterHelper.getCluster(clusterName);
            Properties baseProperties = FeedHelper.getClusterProperties(cluster);
            baseProperties.putAll((Map<?, ?>)FeedHelper.getFeedProperties(feed));
            ArrayList<FeedInstanceStatus> instances = new ArrayList<FeedInstanceStatus>();
            Date feedStart = FeedHelper.getCluster(feed, clusterName).getValidity().getStart();
            TimeZone tz = feed.getTimezone();
            Date alignedStart = EntityUtil.getNextStartTime(feedStart, feed.getFrequency(), tz, start);
            String basePath = location.getPath();
            while (!end.before(alignedStart)) {
                Properties allProperties = ExpressionHelper.getTimeVariables(alignedStart, tz);
                allProperties.putAll((Map<?, ?>)baseProperties);
                String feedInstancePath = ExpressionHelper.substitute(basePath, allProperties);
                FileStatus fileStatus = this.getFileStatus(fileSystem, new Path(feedInstancePath));
                FeedInstanceStatus instance = new FeedInstanceStatus(feedInstancePath);
                Date date = FeedHelper.getDate(basePath, new Path(feedInstancePath), tz);
                instance.setInstance(SchemaHelper.formatDateUTC((Date)date));
                if (fileStatus != null) {
                    instance.setCreationTime(fileStatus.getModificationTime());
                    ContentSummary contentSummary = fileSystem.getContentSummary(fileStatus.getPath());
                    if (contentSummary != null) {
                        long size = contentSummary.getSpaceConsumed();
                        instance.setSize(size);
                        if (!StringUtils.isEmpty((CharSequence)feed.getAvailabilityFlag())) {
                            FileStatus doneFile = this.getFileStatus(fileSystem, new Path(fileStatus.getPath(), feed.getAvailabilityFlag()));
                            if (doneFile != null) {
                                instance.setStatus(FeedInstanceStatus.AvailabilityStatus.AVAILABLE);
                            } else {
                                instance.setStatus(FeedInstanceStatus.AvailabilityStatus.PARTIAL);
                            }
                        } else {
                            instance.setStatus(size > 0L ? FeedInstanceStatus.AvailabilityStatus.AVAILABLE : FeedInstanceStatus.AvailabilityStatus.EMPTY);
                        }
                    }
                }
                instances.add(instance);
                calendar.setTime(alignedStart);
                calendar.add(feed.getFrequency().getTimeUnit().getCalendarUnit(), feed.getFrequency().getFrequencyAsInt());
                alignedStart = calendar.getTime();
            }
            return instances;
        }
        catch (IOException e) {
            LOG.error("Unable to retrieve listing for {}:{}", new Object[]{locationType, this.getStorageUrl(), e});
            throw new FalconException("Unable to retrieve listing for (URI " + this.getStorageUrl() + ")", e);
        }
    }

    @Override
    public FeedInstanceStatus.AvailabilityStatus getInstanceAvailabilityStatus(Feed feed, String clusterName, LocationType locationType, Date instanceTime) throws FalconException {
        List<FeedInstanceStatus> result = this.getListing(feed, clusterName, locationType, instanceTime, instanceTime);
        return result.get(0).getStatus();
    }

    public FileStatus getFileStatus(FileSystem fileSystem, Path feedInstancePath) {
        FileStatus fileStatus = null;
        try {
            fileStatus = fileSystem.getFileStatus(feedInstancePath);
        }
        catch (IOException ignore) {
            // empty catch block
        }
        return fileStatus;
    }

    public Configuration getConf() {
        Configuration conf = new Configuration();
        conf.set("fs.defaultFS", this.storageUrl);
        return conf;
    }

    private String getRelativePath(Location location) {
        Matcher matcher = FeedDataPath.PATTERN.matcher(location.getPath());
        boolean timedPath = matcher.find();
        if (timedPath) {
            return location.getPath().substring(0, matcher.start());
        }
        return location.getPath();
    }

    public String toString() {
        return "FileSystemStorage{storageUrl='" + this.storageUrl + '\'' + ", locations=" + this.locations + '}';
    }
}

