package com.ptteng.controller;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.ptteng.common.skill.model.Message;
import com.ptteng.common.skill.model.School;
import com.ptteng.common.skill.model.SchoolPersonRelation;
import com.ptteng.common.skill.model.User;
import com.ptteng.common.skill.service.MessageService;
import com.ptteng.common.skill.service.SchoolPersonRelationService;
import com.ptteng.common.skill.service.SchoolService;
import com.ptteng.common.skill.service.UserService;
import com.ptteng.model.SchoolRelationInfo;
import com.ptteng.util.DataValidatorUtil;
import com.ptteng.util.DynamicUtil;
import com.qding.common.util.DataUtils;
import com.qding.common.util.http.cookie.CookieUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.util.TextUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * School  crud
 *
 * @author magenm
 * @Date 2014-4-16 13:43
 */
@Controller
public class SchoolController {
    private static final Log log = LogFactory.getLog(SchoolController.class);

    private static final int PERSON_COUNT = 3;
    private static final int CHARGER_COUNT = 1;
    private static final int SENIOR_COUNT = 2;

    @Autowired
    private SchoolService schoolService;

    @Autowired
    private UserService userService;

    @Autowired
    private SchoolPersonRelationService schoolPersonRelationService;

    @Autowired
    private CookieUtil cookieUtil;

    @Autowired
    private MessageService messageService;


    //获取校园详情
    @RequestMapping(value = "/a/u/school/{id}", method = RequestMethod.GET)
    public String getSchoolJson(HttpServletRequest request,
                                HttpServletResponse response, ModelMap model, @PathVariable Long id)
            throws Exception {
        log.info("get data : id= " + id);
        try {
            List<SchoolPersonRelation> schoolPersonRelationList = new ArrayList<>();
            List<User> userList = new ArrayList<>();
            List<Long> relationIds = new ArrayList<>();
            List<Long> uids = new ArrayList<>();

            School school = schoolService.getObjectById(id);
            log.info("get school address " + school.getId());

            //获取校园相关人员信息
            Map<String, Object> relationParams = DynamicUtil.getIdsBySchoolId(id);
            log.info("get relationParams:" + relationParams);
            relationIds = schoolPersonRelationService.getIdsByDynamicCondition(SchoolPersonRelation.class,
                    relationParams, 0, Integer.MAX_VALUE);
            log.info("relationIds====" + relationIds.size());
            if (CollectionUtils.isNotEmpty(relationIds)) {
                schoolPersonRelationList = schoolPersonRelationService.getObjectsByIds(relationIds);
                //根据人员id，获取人员信息
                for (SchoolPersonRelation schoolPersonRelation : schoolPersonRelationList) {
                    uids.add(schoolPersonRelation.getPersonId());
                }
                userList = userService.getObjectsByIds(uids);
            }

            model.addAttribute("schoolPersonRelationList", schoolPersonRelationList);
            model.addAttribute("userList", userList);
            model.addAttribute("code", 0);
            model.addAttribute("school", school);

        } catch (Throwable t) {
            t.printStackTrace();
            log.error(t.getMessage());
            log.error("get school error,id is  " + id);
            model.addAttribute("code", -100000);
        }
        return "/common-skill-service/school/json/schoolDetailJson";
    }

    //新增校园信息
    @RequestMapping(value = "/a/u/school/{schoolId}", method = RequestMethod.POST)
    public String insertSchoolJson(@RequestBody SchoolRelationInfo schoolRelationInfo, HttpServletRequest request,
                                   HttpServletResponse response, ModelMap model,
                                   @PathVariable("schoolId") Long schoolId) throws Exception {

        log.info("schoolId===" + schoolId);
        log.info("insertSchoolJson===" + schoolRelationInfo);

        School school = schoolRelationInfo.getSchoolInfo();
        List<SchoolPersonRelation> schoolPersonRelationList = schoolRelationInfo.getSchoolPersonRelationList();

        //新增的時候，嚴格限制輸入參數，总数为3人，1个负责人，2个辅导师兄
        int chargerCount = 0;
        int seniorCount = 0;
        int totalCount = schoolPersonRelationList.size();
        for (SchoolPersonRelation schoolPersonRelation : schoolPersonRelationList) {
            if (schoolPersonRelation.getType() == SchoolPersonRelation.Person_Type_Charger) {
                chargerCount++;
            }
            if (schoolPersonRelation.getType() == SchoolPersonRelation.Person_Type_Senior) {
                seniorCount++;
            }
        }
        if (schoolPersonRelationList != null
                && chargerCount == CHARGER_COUNT
                && seniorCount == SENIOR_COUNT
                && totalCount == PERSON_COUNT) {
            String uid = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
            boolean result = updateSchool(uid, model, "INSERT", school);
            if (result) {
                updatePerson(uid, model, "INSERT", schoolId, schoolPersonRelationList);
            }
        } else {
            model.addAttribute("code", -60017);
        }
        return "/data/json";
    }


    //修改校园信息
    @RequestMapping(value = "/a/u/school/{schoolId}", method = RequestMethod.PUT)
    public String updateSchoolJson(HttpServletRequest request, ModelMap model, @RequestBody SchoolRelationInfo schoolRelationInfo,
                                   @PathVariable("schoolId") Long schoolId) throws Exception {

        log.info("schoolId===" + schoolId);
        log.info("updateSchoolJson===" + schoolRelationInfo.toString());


        School school = schoolRelationInfo.getSchoolInfo();
        List<SchoolPersonRelation> schoolPersonRelationList = schoolRelationInfo.getSchoolPersonRelationList();

        String uid = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
        boolean result = updateSchool(uid, model, "UPDATE", school);
        if (result) {
            updatePerson(uid, model, "UPDATE", schoolId, schoolPersonRelationList);
        }

        return "/data/json";
    }

    private ModelMap updatePerson(String uid, ModelMap model, String type, Long schoolId, List<SchoolPersonRelation> currentRelationList) {
        log.info("updatePerson===" + currentRelationList);


        if (currentRelationList == null) {
            model.addAttribute("code", -60013);
            return model;
        }

        //验证输入参数是否正确
        for (SchoolPersonRelation schoolPersonRelation : currentRelationList) {
            schoolPersonRelation.setSchoolId(schoolId);
            if (!DataValidatorUtil.checkSchoolPersonRelation(schoolPersonRelation, model)) {
                return model;
            }
        }

        //校园-人员关系表修改数据
        if (currentRelationList.size() > 0) {
            try {
                if (type.equals("UPDATE")) {
                    log.info("UPDATE person==");
                    Map<String, Object> getIdsBySchoolIdParams = DynamicUtil.getIdsBySchoolId(schoolId);
                    List<Long> relationIds = schoolPersonRelationService.getIdsByDynamicCondition(SchoolPersonRelation.class, getIdsBySchoolIdParams, 0, Integer.MAX_VALUE);
                    List<SchoolPersonRelation> dbRelations = schoolPersonRelationService.getObjectsByIds(relationIds);
                    List<SchoolPersonRelation> resultRelations = new ArrayList<>();
                    int legalIdCount = 0; //判断校园相关人员的记录id是否正确
                    for (SchoolPersonRelation dbRelation : dbRelations) {
                        for (SchoolPersonRelation currentRelation : currentRelationList) {
                            if ((dbRelation.getId() + "").equals(currentRelation.getId() + "")) {
                                legalIdCount = legalIdCount + 1;
                                currentRelation.setSchoolId(schoolId);
                                currentRelation.setCreateAt(dbRelation.getCreateAt());
                                currentRelation.setCreateBy(dbRelation.getCreateBy());
                                currentRelation.setUpdateAt(System.currentTimeMillis());
                                currentRelation.setUpdateBy(Long.parseLong(uid));
                                resultRelations.add(currentRelation);
                                break;
                            }
                        }
                    }
                    log.info("legalIdCount===" + legalIdCount);
                    if (legalIdCount == PERSON_COUNT) {
                        schoolPersonRelationService.updateList(resultRelations);
                    } else {
                        model.addAttribute("code", -60025);
                        return model;
                    }
                }
                if (type.equals("INSERT")) {
                    log.info("INSERT person==");
                    //防止重复插入（如果该校园已有关联的人员，则不能再插入）
                    Map<String, Object> params = DynamicUtil.getIdsBySchoolId(schoolId);
                    List<Long> ids = schoolPersonRelationService.getIdsByDynamicCondition(SchoolPersonRelation.class,
                            params, 0, Integer.MAX_VALUE);
                    if (ids != null && ids.size() > 0) {
                        model.addAttribute("code", -60016);
                        return model;
                    }

                    List<SchoolPersonRelation> resultRelations = new ArrayList<>();
                    //傳遞過來的用戶id合法時，再保存數據
                    List<Long> uids = new ArrayList<>();
                    for (SchoolPersonRelation currentRelation : currentRelationList) {
                        uids.add(currentRelation.getPersonId());
                    }

                    //验证人员id输入是否正确
                    if (userService.getObjectsByIds(uids).size() == PERSON_COUNT) {
                        for (SchoolPersonRelation currentRelation : currentRelationList) {
                            currentRelation.setSchoolId(schoolId);
                            currentRelation.setCreateAt(System.currentTimeMillis());
                            currentRelation.setUpdateAt(System.currentTimeMillis());
                            currentRelation.setCreateBy(Long.parseLong(uid));
                            currentRelation.setUpdateBy(Long.parseLong(uid));
                            resultRelations.add(currentRelation);
                        }
                        schoolPersonRelationService.insertList(resultRelations);
                    } else {
                        model.addAttribute("code", -60022);
                        return model;
                    }
                }
                model.addAttribute("code", 0);
            } catch (Exception e) {
                e.printStackTrace();
                model.addAttribute("code", -100000);
            }
        }
        return model;
    }

    private boolean updateSchool(String uid, ModelMap model, String type, School school) {
        log.info("updateSchool==");
        boolean result = false;
        if (school == null) {
            model.addAttribute("code", -60012);
            return false;
        }
        try {
            School dbSchool = schoolService.getObjectById(school.getId());
            if (dbSchool != null) {
                if (type.equals("INSERT")) {
                    log.info("INSERT school=getStatus=" + school.getStatus());
                    if (school.getStatus() != null && (school.getStatus() == School.SCHOOL_IS_NOT_TOP
                            || school.getStatus() == School.SCHOOL_IS_TOP)) {
                        dbSchool.setStatus(school.getStatus());
                    } else {
                        model.addAttribute("code", -60023);
                        return false;
                    }
                    dbSchool.setCreateBy(Long.parseLong(uid));
                    dbSchool.setCreateAt(System.currentTimeMillis());
                    dbSchool.setIsCorporate(School.SCHOOL_IS_CORPORATE);
                    dbSchool.setStatus(School.SCHOOL_OFF_LINE);
                    dbSchool.setIsTop(School.SCHOOL_IS_NOT_TOP);
                }
                if (DataUtils.isNotNullOrEmpty(school.getContactQq())) {
                    dbSchool.setContactQq(school.getContactQq());
                }
                if (DataUtils.isNotNullOrEmpty(school.getContactPhone())) {
                    dbSchool.setContactPhone(school.getContactPhone());
                }
                if (DataUtils.isNotNullOrEmpty(school.getContactName())) {
                    dbSchool.setContactName(school.getContactName());
                }
                if (DataUtils.isNotNullOrEmpty(school.getAddress())) {
                    dbSchool.setAddress(school.getAddress());
                }
                if (DataUtils.isNotNullOrEmpty(school.getSchoolAbstract())) {
                    dbSchool.setSchoolAbstract(school.getSchoolAbstract());
                }
                if (DataUtils.isNotNullOrEmpty(school.getBadge())) {
                    dbSchool.setBadge(school.getBadge());
                }
                if (DataUtils.isNotNullOrEmpty(school.getStatus())) {
                    dbSchool.setStatus(school.getStatus());
                }
                dbSchool.setUpdateAt(System.currentTimeMillis());
                dbSchool.setUpdateBy(Long.parseLong(uid));
                //对校园的各个字段做严格限制
                if (!DataValidatorUtil.checkSchool(school, model)) {
                    return false;
                }
                result = schoolService.update(dbSchool);
                model.addAttribute("code", 0);
            } else {
                result = false;
                model.addAttribute("code", -60024);
            }
        } catch (Throwable t) {
            t.printStackTrace();
            log.error("update school error,id is  " + school.getId() + "==message==" + t.getMessage());
            result = false;
            model.addAttribute("code", -100000);
        }
        return result;
    }

    //学员加入/退出校园
    @RequestMapping(value = "/a/u/school/join/{schoolId}", method = RequestMethod.POST)
    public String joinSchoolJson(ModelMap model, Long uid, String join, @PathVariable("schoolId") Long schoolId) {

        log.info("join school : join= " + join + "==schoolId==" + schoolId + "==userId==" + uid);

        try {
            User user = userService.getObjectById(uid);
            if (DataUtils.isNotNullOrEmpty(user)) {
                switch (join) {
                    case User.JOIN_SCHOOL:
                        //防加入多个校园
                        if (DataUtils.isNotNullOrEmpty(user.getSchoolId())) {
                            model.addAttribute("code", -60000);
                        } else {
                            user.setSchoolId(schoolId);
                            updateUserAndSendMsg(model, user, User.JOIN_SCHOOL);
                        }
                        break;
                    case User.QUIT_SCHOOL:
                        if (DataUtils.isNotNullOrEmpty(user.getSchoolId())) {
                            user.setSchoolId(null);
                            updateUserAndSendMsg(model, user, join);
                        } else {
                            model.addAttribute("code", -60001);
                        }
                        break;
                    default:
                        model.addAttribute("code", -2003);
                        break;
                }
            } else {
                model.addAttribute("code", -5003);
            }
        } catch (Throwable t) {
            t.printStackTrace();
            log.error("update user error,id is  " + uid + "==message==" + t.getMessage());
            model.addAttribute("code", -100000);
        }
        return "/data/json";
    }

    private void updateUserAndSendMsg(ModelMap model, User user, String type)
            throws ServiceException, ServiceDaoException {

        log.info("user===" + user.getId());
        userService.update(user);

        //前台要展示的消息
        Message sendMsg = new Message();
        sendMsg.setUnread(Message.Status_Unread);
        //组装消息基本信息
        sendMsg = organizeMsg(sendMsg, user);

        //后台存储的消息，校园日报筛选时的过滤条件
        Message storeMsg = null;

        log.info("type===" + type);
        switch (type) {
            case User.JOIN_SCHOOL:
                storeMsg = new Message();
                storeMsg = organizeMsg(storeMsg, user);
                storeMsg.setContent(Message.JOIN_MSG);
                storeMsg.setType(Message.JOIN_SCHOOL);
                storeMsg.setUnread(Message.Status_read);

                String joinContent = "<span class=\"font-16px system-alert-outline ng-scope\">欢迎加入修真院<a data-mid=\"97395\" ng-click=\"sendAlertMsg(li.id)\" target=\"_blank\" href=\"/community?mark=1\n\" class=\"font-orange\">校园社区</a>！</span>";
                sendMsg.setContent(joinContent);

                List<Message> messageList = new ArrayList<>();
                messageList.add(storeMsg);
                messageList.add(sendMsg);
                List<Message> result = messageService.insertList(messageList);
                log.info("result===" + result.size());
                break;
            case User.QUIT_SCHOOL:
                //判断用户之前是否有过加入校园记录
                Map<String, Object> joinSchoolParam = DynamicUtil.getJoinSchoolParam(user.getId());
                Long dbMsgId = ((BigInteger) messageService.getObjectByDynamicCondition(Message.class, joinSchoolParam, 0, Integer.MAX_VALUE)).longValue();
                log.info("quit dbMsgId===" + dbMsgId);
                if (DataUtils.isNotNullOrEmpty(dbMsgId)) {
                    //如果有加入过校园社区，则把之前的加入记录删除，确保只有1条加入记录，
                    // 筛选校园日报的时候，直接取这个加入时间作为过滤条件
                    messageService.delete(dbMsgId);
                    String quitContent = "<span class=\"font-16px system-alert-outline ng-scope\"><span class=\"font-green\">您</span>已退出修真院<a data-mid=\"97395\" ng-click=\"sendAlertMsg(li.id)\" target=\"_blank\" href=\"/community?mark=1\n\" class=\"font-orange\">校园社区</a>。</span>";
                    sendMsg.setContent(quitContent);

                    Long mid = messageService.insert(sendMsg);
                    log.info("mid===" + mid);
                }
                break;
            default:
                break;
        }

        model.addAttribute("code", 0);
    }

    private Message organizeMsg(Message msg, User user) {
        msg.setReceiveName(user.getNick());
        msg.setSendName("admin");
        msg.setSendId(0L);
        msg.setReceiveId(user.getId());
        msg.setRelationId(user.getId());
        return msg;
    }


    //校园上下架、置顶、刪除
    @RequestMapping(value = "/a/u/school", method = RequestMethod.PUT)
    public String updateSchoolInfo(HttpServletRequest request, ModelMap model, Long schoolId, Integer isCorporate,
                                   Integer status, Integer isTop) throws Exception {

        String uid = cookieUtil.getKeyIdentity(request, CookieUtil.USER_ID);
        log.info("updateSchoolInfo=schoolId==" + schoolId + "=isCorporate=" + isCorporate + "=status=" + status + "=isTop=" + isTop);

        School dbSchool = schoolService.getObjectById(schoolId);

        //首先要判断传过来的schoolId是否合法，即改学校是否存在，学校是不是合作校园
        if (dbSchool == null || dbSchool.getIsCorporate() == null) {
            model.addAttribute("code", -60024);
            return "/common/success";
        }
        //退出合作校园，需要清空相关的数据
        if (DataUtils.isNotNullOrEmpty(isCorporate) && isCorporate == School.SCHOOL_IS_NOT_CORPORATE) {
            clearSchoolData(uid, dbSchool, model);
            return "/common/success";
        }
        if (DataUtils.isNotNullOrEmpty(status)) {
            dbSchool.setStatus(status);
        }
        if (DataUtils.isNotNullOrEmpty(isTop)) {
            dbSchool.setIsTop(isTop);
        }
        if (DataUtils.isNotNullOrEmpty(status) || DataUtils.isNotNullOrEmpty(isTop)) {
            dbSchool.setUpdateBy(Long.parseLong(uid));
            dbSchool.setUpdateAt(System.currentTimeMillis());
            schoolService.update(dbSchool);
            model.addAttribute("code", 0);
        } else {
            model.addAttribute("code", -1001);
        }
        return "/common/success";
    }

    private void clearSchoolData(String uid, School dbSchool, ModelMap model) throws Exception {
        log.info("clearSchoolData===");
        dbSchool.setUpdateBy(Long.parseLong(uid));
        dbSchool.setUpdateAt(System.currentTimeMillis());
        dbSchool.setIsTop(null);
        dbSchool.setContactQq(null);
        dbSchool.setContactPhone(null);
        dbSchool.setContactName(null);
        dbSchool.setAddress(null);
        dbSchool.setSchoolAbstract(null);
        dbSchool.setBadge(null);
        dbSchool.setStatus(null);
        dbSchool.setIsCorporate(null);
        schoolService.update(dbSchool);

        //1、根据校园id查找校园人员关联表中对应的人员id集合和关联数据id集合
        //2、删除关联表中相关的数据
        //3、根据1的id集合，将用户的school_id字段置为空
        List<Long> uids = new ArrayList<>();
        List<Long> relationIds = new ArrayList<>();
        List<User> resultUsers = new ArrayList<>();
        List<User> joinedUsers = new ArrayList<>();

        Map<String, Object> relationIdParams = DynamicUtil.getIdsBySchoolId(dbSchool.getId());
        relationIds = schoolPersonRelationService.getIdsByDynamicCondition(SchoolPersonRelation.class, relationIdParams, 0, Integer.MAX_VALUE);
        log.info("relationIds==" + relationIds.size());

        Map<String, Object> uidsParams = DynamicUtil.getUidsBySchoolId(dbSchool.getId());
        //先获取相应的数据，再删除记录
        uids = userService.getIdsByDynamicCondition(SchoolPersonRelation.class, uidsParams, 0, Integer.MAX_VALUE);
        if (uids.size() > 0) {
            joinedUsers = userService.getObjectsByIds(uids);
            for (User user : joinedUsers) {
                user.setSchoolId(null);
                resultUsers.add(user);
            }
            userService.updateList(resultUsers);
        }

        schoolPersonRelationService.deleteList(SchoolPersonRelation.class, relationIds);

        model.addAttribute("code", 0);
    }


}

