package com.ptteng.onway.platform.service.waimai.dish.impls;

import com.gemantic.common.exception.ServiceDaoException;
import com.gemantic.common.exception.ServiceException;
import com.google.common.collect.Lists;
import com.jiabangou.bdwmsdk.api.BdWmClient;
import com.jiabangou.bdwmsdk.api.DishService;
import com.jiabangou.bdwmsdk.exception.BdWmErrorException;
import com.jiabangou.bdwmsdk.model.Category;
import com.jiabangou.bdwmsdk.model.CategoryUpdate;
import com.jiabangou.bdwmsdk.model.Dish;
import com.jiabangou.bdwmsdk.model.DishDetail;
import com.ptteng.onway.platform.model.DishGroupWmDTO;
import com.ptteng.onway.platform.model.DishNormWmDTO;
import com.ptteng.onway.platform.service.waimai.BaiduAbstractService;
import com.ptteng.onway.platform.service.waimai.dish.WaimaiDishService;
import com.ptteng.onway.platform.service.waimai.other.ConvertUtils;
import com.ptteng.onway.platform.service.waimai.other.WaimaiConstants;
import com.ptteng.onway.platform.service.waimai.other.model.WaimaiDishMapping;
import com.ptteng.onway.platform.service.waimai.other.model.WaimaiStoreMapping;
import org.apache.log4j.Logger;

import java.util.List;

/**
 * Author wanglei
 * Created on 18-1-9
 */
public class BaiduDishServiceImpl extends BaiduAbstractService implements WaimaiDishService {

    private static Logger logger = Logger.getLogger(BaiduDishServiceImpl.class);

    /**
     * 如果百度保存失败就再也得不到百度的分类, 就只能用一个无意义的数据做填充
     */
    private static final String DEFAULT_BAIDU_CATEGORY_ID = "0";

    private final static int STOCK_COUNT = 300;


    @Override
    public void saveDishGroup(Long mchId, Long storeId, Long dishGroupId, DishGroupWmDTO dishGroup) throws ServiceException, ServiceDaoException {
        //检查菜品分组是否可以跟百度平台同步
        //门店配置有个跟第三平台同步的开关  如果门店配置 打开了需要跟百度同步的开关 菜品分组才能同步到百度
        if (!this.checkStorePlatformById(mchId, storeId, WaimaiConstants.PlatformType.BAIDU)) {
            return;
        }

        BdWmClient bdWmClient = getBdWmClient(mchId);
        // 如果为空表示不支持同步到百度外卖
        if (bdWmClient == null) {
            return;
        }

        // 如果没有找到对应的餐厅对应就不处理了
        WaimaiStoreMapping storeMapping = findStoreMapping(mchId, storeId, WaimaiConstants.PlatformType.BAIDU);
        if (storeMapping == null) {
            return;
        }
        DishService dishService = bdWmClient.getDishService();
        String storeMappingId = storeMapping.getMappingId();

        try {

            String dishGroupOldName = dishGroup.getOldName() != null ? dishGroup.getOldName() : dishGroup.getName();
            Category oldCategory = dishService.getCategory(storeMappingId, dishGroupOldName);
            if (oldCategory != null) {
                // 百度外卖么的分类没有删除功能, 所以只能再去从百度外卖接口里面再去查找是否有同名的分类
                CategoryUpdate categoryUpdate = new CategoryUpdate();
                categoryUpdate.setName(dishGroup.getName());
                categoryUpdate.setOld_name(oldCategory.getName());
                categoryUpdate.setRank(dishGroup.getGroupRank().intValue());
                categoryUpdate.setShop_id(storeMappingId);
                dishService.updateCategory(categoryUpdate);
                saveDishCategoryMapping(mchId, storeId, dishGroupId, categoryUpdate.getName(),
                        DEFAULT_BAIDU_CATEGORY_ID, WaimaiConstants.PlatformType.BAIDU);
            } else {

                Category category = new Category();
                category.setName(dishGroup.getName());
                category.setRank(dishGroup.getGroupRank().intValue());
                category.setShop_id(storeMappingId);
                Long categoryId = dishService.createCategory(category);
                saveDishCategoryMapping(mchId, storeId, dishGroupId, category.getName(),
                        String.valueOf(categoryId), WaimaiConstants.PlatformType.BAIDU);
            }
        } catch (BdWmErrorException bdWmErrorException) {
            logger.error(bdWmErrorException.getMessage(), bdWmErrorException);
        }
    }

    @Override
    public void updateDishGroup(Long mchId, Long storeId, Long dishGroupId, DishGroupWmDTO dishGroup) throws ServiceException, ServiceDaoException {
        this.saveDishGroup(mchId, storeId, dishGroupId, dishGroup);
    }

    @Override
    public void onlineDish(Long mchId, Long storeId, Long dishNormId, DishNormWmDTO dishNormWmDTO) throws ServiceException, ServiceDaoException {
        //检查菜品是否需要在百度平台上线
        if (!checkDishPlatformById(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU)) {
            return;
        }

        BdWmClient bdWmClient = getBdWmClient(mchId);
        // 如果为空表示不支持同步到百度外卖
        if (bdWmClient == null) {
            return;
        }

        // 如果没有找到对应的餐厅对应就不处理了
        WaimaiStoreMapping storeMapping = findStoreMapping(mchId, storeId, WaimaiConstants.PlatformType.BAIDU);
        if (storeMapping == null) {
            return;
        }

        DishService dishService = bdWmClient.getDishService();

        WaimaiDishMapping dishMapping = findDishMapping(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU);

        String dishMappingId;

        if (dishMapping == null) {
            dishMappingId = String.valueOf(dishNormId);
        } else {
            dishMappingId = dishMapping.getMappingId();
        }


        try {

            checkDishGroupAndCreate(mchId, storeId, dishNormWmDTO.getDishGroupWmDTO(), dishService, storeMapping.getMappingId());
            DishDetail dishDetail = dishService.get(storeMapping.getMappingId(), dishMappingId);
            if (dishDetail != null) {
                dishService.online(storeMapping.getMappingId(), dishMappingId);
            }
        } catch (BdWmErrorException e) {
            logger.error(e.getMessage(), e);
        }
    }

    private void checkDishGroupAndCreate(Long mchId, Long storeId, DishGroupWmDTO dishGroupWmDTO,
                                         DishService dishService, String storeMappingId) throws ServiceException, ServiceDaoException {
        try {
            Category oldCategory = dishService.getCategory(storeMappingId, dishGroupWmDTO.getName());
            if (oldCategory == null) {
                // 没有找到同名的分类表明可以进行创建
                Category category = new Category();
                category.setName(dishGroupWmDTO.getName());
                category.setRank(dishGroupWmDTO.getGroupRank().intValue());
                category.setShop_id(storeMappingId);
                long categoryId = dishService.createCategory(category);
                saveDishCategoryMapping(mchId, storeId, dishGroupWmDTO.getId(), category.getName(),
                        String.valueOf(categoryId), WaimaiConstants.PlatformType.BAIDU);
            }
        } catch (BdWmErrorException e) {
            logger.error(e.getMessage(), e);
        }
    }

    @Override
    public void offlineDish(Long mchId, Long storeId, Long dishNormId) throws ServiceException, ServiceDaoException {
        //检查菜品是否需要在百度平台上线
        if (!checkDishPlatformById(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU)) {
            return;
        }

        BdWmClient bdWmClient = getBdWmClient(mchId);
        // 如果为空表示不支持同步到百度外卖
        if (bdWmClient == null) {
            return;
        }

        WaimaiStoreMapping storeMapping = findStoreMapping(mchId, storeId, WaimaiConstants.PlatformType.BAIDU);
        // 如果没有找到对应的餐厅对应就不处理了
        if (storeMapping == null) {
            return;
        }

        WaimaiDishMapping dishMapping = findDishMapping(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU);
        if (dishMapping == null) {
            return;
        }

        try {
            DishService dishService = bdWmClient.getDishService();
            dishService.offline(String.valueOf(storeMapping.getMappingId()), String.valueOf(dishNormId));
        } catch (BdWmErrorException bdWmErrorException) {
            logger.error(bdWmErrorException.getMessage(), bdWmErrorException);
        }
    }

    @Override
    public void updateDish(Long mchId, Long storeId, Long dishNormId, DishNormWmDTO dishNormWmDTO) throws ServiceException, ServiceDaoException {
        //检查菜品是否需要在百度平台上线
        if (!checkDishPlatformById(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU)) {
            return;
        }

        BdWmClient bdWmClient = getBdWmClient(mchId);
        // 如果为空表示不支持同步到百度外卖
        if (bdWmClient == null) {
            return;
        }

        // 如果没有找到对应的餐厅对应就不处理了
        WaimaiStoreMapping storeMapping = findStoreMapping(mchId, storeId, WaimaiConstants.PlatformType.BAIDU);
        if (storeMapping == null) {
            return;
        }

        DishService dishService = bdWmClient.getDishService();

        WaimaiDishMapping dishMapping = findDishMapping(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU);

        String dishMappingId;

        if (dishMapping == null) {
            dishMappingId = String.valueOf(dishNormId);
        } else {
            dishMappingId = dishMapping.getMappingId();
        }

        Dish dish = buildBdWmDish(storeMapping.getMappingId(), dishNormWmDTO, dishMappingId);
        try {
            checkDishGroupAndCreate(mchId, storeId, dishNormWmDTO.getDishGroupWmDTO(), dishService, storeMapping.getMappingId());
            DishDetail dishDetail = dishService.get(storeMapping.getMappingId(), dishMappingId);
            if (dishDetail != null) {
                dishService.update(dish);
            }
        } catch (BdWmErrorException bdWmErrorException) {
            logger.error(bdWmErrorException.getMessage(), bdWmErrorException);
        }
    }

    private Dish buildBdWmDish(String storeMappingId, DishNormWmDTO dishNormWmDTO, String mappingId) throws ServiceException {
        Dish bdWmDish = new Dish();

        bdWmDish.setShop_id(storeMappingId);
        bdWmDish.setDish_id(mappingId);
        bdWmDish.setPrice(dishNormWmDTO.getPrice());
        bdWmDish.setPic(dishNormWmDTO.getPic() + WaimaiConstants.ImgStyle.BAIDU_IMG_STYLE);
        bdWmDish.setMin_order_num(dishNormWmDTO.getMinOrderNum());
        bdWmDish.setPackage_box_num(dishNormWmDTO.getPackageBoxNum());
        bdWmDish.setDescription(dishNormWmDTO.getDescription());
        bdWmDish.setName(dishNormWmDTO.getName());
        bdWmDish.setCategory(Lists.newArrayList(ConvertUtils.convert(dishNormWmDTO.getDishGroupWmDTO(), Category.class)));
        bdWmDish.setStock(STOCK_COUNT);

        return bdWmDish;
    }

    @Override
    public void batchUpdateDish(Long mchId, Long storeId, List<DishNormWmDTO> dishNormWmDTOs) throws ServiceException, ServiceDaoException {
        for (DishNormWmDTO dishNormWmDTO: dishNormWmDTOs){
            updateDish(mchId, storeId, dishNormWmDTO.getDishNormId(), dishNormWmDTO);
        }
    }

    @Override
    public void setThreshold(Long mchId, Long storeId, Long dishNormId, int stock) throws ServiceException, ServiceDaoException {
        //检查菜品是否需要在百度平台上线
        //菜品不需要同步的话 不需要设置库存
        if (!checkDishPlatformById(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU)) {
            return;
        }

        BdWmClient bdWmClient = getBdWmClient(mchId);
        // 如果为空表示不支持同步到百度外卖
        if (bdWmClient == null) {
            return;
        }

        WaimaiStoreMapping storeMapping = findStoreMapping(mchId, storeId, WaimaiConstants.PlatformType.BAIDU);
        // 如果没有找到对应的餐厅对应就不处理了
        if (storeMapping == null) {
            return;
        }

        WaimaiDishMapping dishMapping = findDishMapping(mchId, storeId, dishNormId, WaimaiConstants.PlatformType.BAIDU);
        if (null == dishMapping) {
            return;
        }

        DishService dishService = bdWmClient.getDishService();
        //设置百度外卖库存值
        try {
            dishService.setThreshold(storeMapping.getMappingId(), dishMapping.getMappingId(), stock);
        } catch (BdWmErrorException bdWmErrorException) {
            logger.error(bdWmErrorException.getMessage(), bdWmErrorException);
        }
    }
}
