package com.ptteng.controller;

import com.ptteng.util.ListUtil;
import java.util.List;
import com.ptteng.util.ParamsUtil;
import com.ptteng.yi.nucleus.model.Students;
import com.ptteng.yi.nucleus.model.SubjectExam;
import com.ptteng.yi.nucleus.service.SubjectExamService;
import com.ptteng.yi.nucleus.service.SubjectPracticeService;
import com.qding.common.util.DataUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import com.ptteng.yi.nucleus.model.Note;
import com.ptteng.yi.nucleus.service.NoteService;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 记录学员做的题目相关接口
 *
 * @author zhangxin
 * @Date 2018/01/09
 */
@Controller
public class NoteController {
    private static final Log log = LogFactory.getLog(NoteController.class);
    private static final int CLEAR_WRONG_PRACTICE_ON = 1;
    private static final int CLEAR_WRONG_PRACTICE_OFF = 0;

    @Resource
    private NoteService noteService;

    @Resource
    private SubjectPracticeService subjectPracticeService;

    @Resource
    private SubjectExamService subjectExamService;

    /**
     * 移除单个错题
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/note/error/{id}", method = RequestMethod.PUT)
    public String removeErrorPractice(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            @PathVariable Long id) throws Exception {
        log.info("Interface PUT /a/u/note/error/{id}, Method removeErrorPractice() parameter, id : " + id);

        try {
            Students students = (Students) request.getAttribute("user");

            Long noteId=noteService.getNoteIdByStudentIdAndPracticeId(students.getId(),id);
            Note note=noteService.getObjectById(noteId);
            log.info("get note data by id, note is : " + note);
            if (DataUtils.isNullOrEmpty(note)) {
                log.info("Can not find note data by id, id is : " + id);
                model.addAttribute("code", -14000);
                return "/common/failure";
            }

            note.setResult(Note.RESULT_RIGHT);
            noteService.update(note);
            log.info("update note success");

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method removeErrorPractice() failed, parameter, id : " + id);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 清空所有错题接口
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/note/error", method = RequestMethod.DELETE)
    public String cleanErrorPractices(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Long studentId, Integer subjectType) throws Exception {
        log.info("Interface DELETE /a/u/note/error, Method cleanErrorPractices() parameters, studentId: " + studentId
                + ", subjectType: " + subjectType);

        int validateResult = ParamsUtil.checkParamsOfStudentIdAndSubjectType(studentId, subjectType);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            List<Long> errorPracticeIdList = noteService.getNoteIdsByStudentIdAndSubjectTypeAndResult(studentId,
                    subjectType, Note.RESULT_WRONG, 0, Integer.MAX_VALUE);
            log.info("get error practice id list, list size is : " + errorPracticeIdList.size());

            if (CollectionUtils.isNotEmpty(errorPracticeIdList)) {
                noteService.deleteList(Note.class, errorPracticeIdList);
                log.info("clean up error practices success!");
            }

            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method cleanErrorPractices() failed, parameters, studentId: " + studentId + ", subjectType: "
                    + subjectType);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 记录做题结果
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/note", method = RequestMethod.POST)
    public String recordPracticeResult(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Long studentId, Long practiceId, Integer subjectType, Integer result, Integer isRemoved) throws Exception {
        log.info("Interface POST /a/u/note, Method recordPracticeResult() parameters: studentId : " + studentId
                + ", practiceId" + practiceId + ", subjectType: " + subjectType + ", result: " + result
                + ", isRemoved: " + isRemoved);

        int validateResult = ParamsUtil.checkParamsOfRecordPracticeResult(studentId, practiceId, subjectType, result,
                isRemoved);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
            Long noteId = noteService.getNoteIdByStudentIdAndPracticeId(studentId, practiceId);
            log.info("get note id by studentId and practiceId, note id is : " + noteId);
            if (DataUtils.isNullOrEmpty(noteId)) {
                Note note = new Note();
                note.setStudentId(studentId);
                note.setPracticeId(practiceId);
                note.setSubjectType(subjectType);
                note.setResult(result);
                note.setCreateBy(studentId);
                note.setUpdateBy(studentId);
                noteId = noteService.insert(note);
                log.info("insert note success, note id is : " + noteId);
            } else {
                if (isRemoved == CLEAR_WRONG_PRACTICE_ON || result.equals(Note.RESULT_WRONG)) {
                    Note note = noteService.getObjectById(noteId);
                    log.info("get note data by id, note is : " + note);

                    note.setResult(result);
                    noteService.update(note);
                    log.info("update note success, note id is : " + note.getId());
                }
            }
            model.addAttribute("code", 0);
            return "/common/success";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error("Method recordPracticeResult() failed, parameters: studentId : " + studentId + ", practiceId"
                    + practiceId + ", subjectType: " + subjectType + ", result: " + result);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }

    /**
     * 获取学员练习统计结果
     *
     * @param
     * @return
     * @throws ServiceException
     * @throws ServiceDaoException
     */
    @RequestMapping(value = "/a/u/exercised/record", method = RequestMethod.GET)
    public String getPracticeRecord(HttpServletRequest request, HttpServletResponse response, ModelMap model,
            Long studentId, Integer subjectType) throws Exception {

        log.info("Interface GET /a/u/exercised/record, Method getPracticeRecord() parameter: studentId : " + studentId
                + ", subjectType: " + subjectType);

        int validateResult = ParamsUtil.checkParamsOfStudentIdAndSubjectType(studentId, subjectType);
        if (validateResult != ParamsUtil.VALIDATE_RESULT_SUCCESS) {
            model.addAttribute("code", validateResult);
            return "/common/failure";
        }

        try {
//            Integer total = subjectPracticeService.countSubjectPracticeIdsBySubjectType(subjectType);
            List<Long> total = subjectPracticeService.getSubjectPracticeIdsBySubjectType(subjectType, 0, Integer.MAX_VALUE);
//            Integer correct = noteService.countPracticeIdsByStudentIdAndSubjectTypeAndResult(studentId, subjectType,
//                    Note.RESULT_RIGHT);
            List<Long> correct = noteService.getPracticeIdsByStudentIdAndSubjectTypeAndResult(studentId, subjectType,
                    Note.RESULT_RIGHT, 0, Integer.MAX_VALUE);
//            Integer error = noteService.countPracticeIdsByStudentIdAndSubjectTypeAndResult(studentId, subjectType,
//                    Note.RESULT_WRONG);
            List<Long> error = noteService.getPracticeIdsByStudentIdAndSubjectTypeAndResult(studentId, subjectType,
                    Note.RESULT_WRONG, 0, Integer.MAX_VALUE);
            log.info("get total, correct, error practice size is : " + total + ", " + correct + ", " + error);

            List<SubjectExam> examList = subjectExamService.getSubjectExamListByStudentIdAndType(studentId, subjectType,
                    0, Integer.MAX_VALUE);
            log.info("get exam data list, list size is : " + examList.size());
            Integer bestMockScore = null;
            if (1 == subjectType) {
                bestMockScore = ListUtil.getMaxValueFromObjectListByField(examList, SubjectExam.class, "right");
                log.info("get bestMockScore is : " + bestMockScore);
            } else {
                bestMockScore = 2 * ListUtil.getMaxValueFromObjectListByField(examList, SubjectExam.class, "right");
                log.info("get bestMockScore is : " + bestMockScore);
            }
            model.addAttribute("total", total.size());
            model.addAttribute("correct", correct.size());
            model.addAttribute("error", error.size());
            model.addAttribute("code", 0);
            model.addAttribute("bestMockScore", bestMockScore);
            return "/yi-nucleus-service/note/json/practiceRecord";
        } catch (Throwable t) {
            log.error(t.getMessage());
            log.error(
                    "Method getPracticeRecord() parameter: studentId : " + studentId + ", subjectType: " + subjectType);
            model.addAttribute("code", -1);
            return "/common/failure";
        }
    }
}
